/*
 * GNU.FREE 2002
 *
 * Copyright (c) 1999, 2000, 2001, 2002
 * The Free Software Foundation (www.fsf.org)
 *
 * GNU.FREE Co-ordinator: Jason Kitcat <jeep@free-project.org>
 *
 * GNU site: http://www.gnu.org/software/free/
 * 
 * FREE e-democracy site: http://www.free-project.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program (COPYING); if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package FreeInstall;

import org.apache.xerces.parsers.DOMParser;
import org.xml.sax.ErrorHandler;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.NamedNodeMap;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import java.io.IOException;

/**
 * <code>ElectionDefinition</code> provides the methods for reading and using Election Definition
 * language XML files which define what kind of vote/count system to use, who the candidates are
 * etc..
 *
 * Based on code from tutorial at http://www.ecerami.com and from O'Reilly "Java & XML"
 *
 * @version 0.2 1 December 2001
 * @author Jason Kitcat
 * @since 1.9
 */
public class ElectionDefinition implements ErrorHandler {
    
    // Constructor
    public ElectionDefinition (String xmlFile) {
        //  Create a Xerces DOM Parser
        DOMParser parser = new DOMParser();
    
        //  Turn Validation on
        try {
            parser.setFeature("http://xml.org/sax/features/validation", true);
        } catch (SAXNotRecognizedException e) {
	        Install.NORM.error("Election Definition error (SAXNotRecognized): " + e);
        } catch (SAXNotSupportedException e) {
	        Install.NORM.error("Election Definition error (SAXNotSupported): " + e);
        }
    
        //  Register Error Handler
        parser.setErrorHandler(this);

        //  Parse the Document      
        //  and traverse the DOM
        try {
            parser.parse(xmlFile);
            Document document = parser.getDocument();
            getInstallVars(document);
        } catch (SAXException e) {
	        Install.NORM.error("Election Definition error: " + e);
        } catch (IOException e) {
	        Install.NORM.error("Election Definition error: " + e);
        }
    } // eof constructor
    
    /**
     * Find each property that we need for installation, collect all attributes & data for storing
     * and later use.
     *
     * @param  doc  A document for us to find tags in.
     */
    private void getInstallVars(Document doc) {
    
    	NodeList result = doc.getElementsByTagName("gnufree:name");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_name = childnodes.item(0).getNodeValue(); // store value
        } else {
        	Install.NORM.error("Election Definition error: No name specified!");
        }
        
    	result = doc.getElementsByTagName("gnufree:jurisdiction");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_jurisdiction = childnodes.item(0).getNodeValue(); // store value
        } else {
        	Install.NORM.error("Election Definition error: No jurisdiction specified!");
        }

    	result = doc.getElementsByTagName("gnufree:contact_email");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_contact_email = childnodes.item(0).getNodeValue(); // store value
        } else {
        	Install.NORM.error("Election Definition error: No contact email specified!");
        }

    	result = doc.getElementsByTagName("gnufree:website");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_website = childnodes.item(0).getNodeValue(); // store value
        } else {
        	Install.NORM.error("Election Definition error: No website specified!");
        }

    	result = doc.getElementsByTagName("gnufree:jnlp");
		NamedNodeMap attributes = result.item(0).getAttributes();
		for (int x=0; x< attributes.getLength(); x++) {
			Node current = attributes.item(x);
			Install.edl_jnlp_attr[x] = new String(current.getNodeValue()).toLowerCase(); // store attribute
		}

    	result = doc.getElementsByTagName("gnufree:auth_system");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_auth_system = childnodes.item(0).getNodeValue(); // store value
     		
			attributes = result.item(0).getAttributes();
			for (int x=0; x< attributes.getLength(); x++) {
				Node current = attributes.item(x);
				Install.edl_auth_system_attr[x] = new String(current.getNodeValue()).toLowerCase(); // store attribute
			}
        } else {
        	Install.NORM.error("Election Definition error: No auth system specified!");
        }

    	result = doc.getElementsByTagName("gnufree:vote_system");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_vote_system = childnodes.item(0).getNodeValue(); // store value
     		
			attributes = result.item(0).getAttributes();
			for (int x=0; x< attributes.getLength(); x++) {
				Node current = attributes.item(x);
				Install.edl_vote_system_attr[x] = new String(current.getNodeValue()).toLowerCase(); // store attribute
			}
        } else {
        	Install.NORM.error("Election Definition error: No vote system specified!");
        }

    	result = doc.getElementsByTagName("gnufree:count_system");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_count_system = childnodes.item(0).getNodeValue(); // store value
     		
			attributes = result.item(0).getAttributes();
			for (int x=0; x< attributes.getLength(); x++) {
				Node current = attributes.item(x);
				Install.edl_count_system_attr[x] = new String(current.getNodeValue()).toLowerCase(); // store attribute
			}
        } else {
        	Install.NORM.error("Election Definition error: No count system specified!");
        }
        
    	result = doc.getElementsByTagName("gnufree:write_in");
        if (result != null) {
           	NodeList childnodes = result.item(0).getChildNodes();
     		Install.edl_write_in = new String(childnodes.item(0).getNodeValue()).toLowerCase(); // store value
        } else {
        	Install.NORM.error("Election Definition error: No write in option specified!");
        }

    	result = doc.getElementsByTagName("gnufree:choice");
        if (result != null) {
        	Install.edl_number_of_choices = result.getLength();
            for (int i=0; i< result.getLength(); i++) {  // iterate through tag matches
            	NodeList childnodes = result.item(i).getChildNodes();
            	for (int z=0; z< childnodes.getLength(); z++) {  // iterate through tag match children
       			    Install.edl_choice[i] = childnodes.item(z).getNodeValue(); // store value
			    }
		        attributes = result.item(i).getAttributes();
		        for (int x=0; x< attributes.getLength(); x++) {
		        	Node current = attributes.item(x);
		        	Install.edl_choice_attr[i][x] = new String(current.getNodeValue()).toLowerCase(); // store attribute
		        }
		    }
        } else {
        	Install.NORM.error("Election Definition error: No choice specified!");
        }
	} // EOF getInstallVars
    
    //  Warning Event Handler
    public void warning (SAXParseException e)
        throws SAXException {
        Install.NORM.warn("Election Definition warning: " + e);
    }

    //  Error Event Handler
    public void error (SAXParseException e)
        throws SAXException {
        Install.NORM.error("Election Definition error: " + e);
    }
    
    //  Fatal Error Event Handler
    public void fatalError (SAXParseException e)
        throws SAXException {
        Install.NORM.fatal("Election Definition fatal error: " + e);
    }
    
} //EOF class

