/*
 * GNU.FREE 2001
 *
 * Copyright (c) 1999, 2000, 2001 The Free Software Foundation (www.fsf.org)
 *
 * GNU.FREE Co-ordinator: Jason Kitcat <jeep@thecouch.org>
 *
 * GNU site: http://www.gnu.org/software/gnu.free/gnufree.html
 * 
 * FREE e-democracy site: http://www.thecouch.org/free/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program (gpl.txt); if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
 
package ERServer;
 
import java.util.*;
import Free.util.*;

/**
 * Processes the Strings passed to it by <code>TCPServer</code> but throws
 * all errors back up to <code>TCPServer</code> which can better handle
 * problems by elegantly closing connections.
 *
 * The source code is very readable and best explains the logic.
 *
 * @version 0.82  07 February 2001
 * @author Jason Kitcat
 */
public class ERServerProtocol {

	  
	/**
	 * <code>process</code> analyses Strings sent from <code>TCPServer</code>
	 * and decides whether the data is in fact valid packets.
	 *
	 * If so then the appropriate replies are formulated and returned to 
	 * <code>TCPServer</code> for sending to the client.
	 *
	 * @param inputData   Contains the contents of a received 
	 * @returns   String with a code to return to client
	 */
	protected static String process(String inputData) throws Exception {

		String outputData="";
		
		if (inputData.charAt(0) == 'D') {
		
			/* diagnostic */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
			
				ERServer.NORM.info("Diagnostic packet received");
			
				if (p.getMessage().equals("TEST")) {
					outputData= "D|OK|" + AuthSys.makeDigest("OK") + "\r\n";
				} else {
					ERServer.NORM.warn("Unrecognized diagnostic command.");
					outputData = "ERROR";
				}
			
			} else {
				ERServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}
			
		} else if (inputData.charAt(0) == 'E') {
		
			/* ER check */
			Packet p = new Packet(inputData);
			
			/* NOTE: If the Auth System is changed from the default code
			 * provide with FREE then the number below will also need to be
			 * changed to the MINUMUM number of characters long an Auth packet
			 * can be in your system.
			 * Remember the MAC is always 31 characters long!
			 */
			if (inputData.length() >= 47) {  // simple validity check to save work
			
				if (AuthSys.checkDigest(p)) {  // tamper check
				
					try {
						String data, name, code, pword;
						int i=0;
						int t=0;
					
						data = p.getMessage();
					
						// parse to get seperate data fields
						while (data.charAt(i)!='-') {
							i++;
						}
						name = data.substring(0,i);

						t = i+2;
					
						while (data.charAt(t)!='-') {
							t++;
						}
					
						code = data.substring((i+1),t);									
						pword = data.substring(t+1);
						
						// another simple validity check to save time
						if (code.equals("")|code.equals(" ")) {
							ERServer.NORM.warn("Bad auth code data in packet.");
							outputData = "ERROR";
						} else if (pword.equals("")|pword.equals(" ")) {
							ERServer.NORM.warn("Bad auth pword data in packet.");
							outputData = "ERROR";
						} else if (name.equals("")|name.equals(" ")) {
							ERServer.NORM.warn("Bad auth name data in packet.");
							outputData = "ERROR";
						} else {
							/* data seems ok so do commit to doing database work */
							try {
								if (DBase.checkER(name, code, pword)) {
									// before sending confirmation make auth key
									String ak = AuthKey.build(name,code,pword,2);

									outputData = "A|TRUE-" + ak + "|" + AuthSys.makeDigest("TRUE-" + ak) + "\r\n";
									ERServer.NORM.info("Voter authorised!");
								} else {
									// datbase check didn't match or user has already voted
									outputData = "A|FALSE-NOKEY|" + AuthSys.makeDigest("FALSE-NOKEY") + "\r\n";
									ERServer.NORM.info("Voter authorisation failed!");
								}
							} catch (Exception e) {
								outputData = "ERROR";
								ERServer.NORM.error("Auth dbase/auth key exception: " + e.getMessage());
							}
								
						}
						
					} catch (IndexOutOfBoundsException e) {
						ERServer.NORM.error("Unrecognized or bad packet received.");
						outputData = "ERROR";
					}

				} else {
					ERServer.NORM.error("Packet corrupted or altered!");
					outputData = "ERROR";
				}

			} else {
				ERServer.NORM.warn("Bad Packet!");
				outputData = "ERROR";
			}

		} else if (inputData.charAt(0) == 'K') {
		
			/* auth key packet */
			ERServer.NORM.info("Auth Key packet received");
			
			Packet p = new Packet(inputData);
			
			if (inputData.length() >= 88) {  // simple validity check to save work
			
				if (AuthSys.checkDigest(p)) {  // tamper check
					DBase.confirmVoted(p.getMessage());
					outputData = "K|" + p.getMessage() + "|" + AuthSys.makeDigest(p.getMessage()) + "\r\n";
				} else {
					ERServer.NORM.warn("Packet corrupted or altered!");
					outputData = "ERROR";
				}
				
			} else {
				ERServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}

		} else if (inputData.charAt(0) == 'Q') {
		
			/* verification query */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
				if (p.getMessage().equals("TOTALVOTED")) {
					int total = DBase.usersVoted();
					outputData= "Q|" + Integer.toString(total) + "|" + AuthSys.makeDigest(Integer.toString(total)) + "\r\n";
					ERServer.NORM.info("Total users voted calculated and sent.");
				} else {
					ERServer.NORM.warn("Unrecognized verification query.");
					outputData = "ERROR";
				}
			} else {
				ERServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}
			
		} else if (inputData.charAt(0) == 'X') {
		
			/* end communication */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
				outputData = "DONE";
			} else {
				ERServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}
			
		} else {
				ERServer.NORM.warn("Unrecognized packet received.");
				outputData = "ERROR";
		}
		
		return outputData;

	}  //EOF process


} //EOF Class
