/*
 * FREE 2000
 *
 * Copyright (c) Jason Kitcat, University of Warwick
 *
 * jeep@thecouch.org   http://www.thecouch.org/free/
 *
 * Knoll Cottage, Sutton Place, Nr.Abinger, Dorking, Surrey RH5 6RN United Kingdom
 *
 * Tel: 07956 886 508  Fax: 0870 052 7114
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program (gpl.txt); if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
  
package Free;

import java.util.*;
import Free.util.*;

/**
 * Processes the Strings passed to it by <code>TCPServer</code> but throws
 * all errors back up to <code>TCPServer</code> which can better handle
 * problems by elegantly closing connections.
 *
 * Also included are data structures and methods for storing and retrieving 
 * check date/time stamps for the FREE security system to provide persistence
 * between calls of <code>process()</code>.
 * 
 * @version 0.11  10 November 2000
 * @author Jason Kitcat
 */
public class ServerProtocol {

	private static Vector checkDates = new Vector(1);
	private static Vector checkVotes = new Vector(1);
	private static Vector checkAuthKey = new Vector(1);
	
	/*
	 * Everything above is to keep the protocols neatly in this class
	 * and thus maintain persistence between calls of process()
	 * I feel this undoubtedly makes for clearer reading overall.
	 */
	 

	
	/**
	 * adds an entry to the checkVotes Vector for later use.
	 *
	 * @param  n  The index at which to insert
	 * @param  data  Packet message string to store
	 */
	 private static void setVotes(int n, String data) {
	 
	 	checkVotes.insertElementAt(data, n);
	 
	 } //EOF setVotes


	/**
	 * adds an entry to the checkAuthKey Vector for later use.
	 *
	 * @param  n  The index at which to insert
	 * @param  data  Packet message string to store
	 */
	 private static void setCheckAK(int n, String data) {
	 
	 	checkAuthKey.insertElementAt(data, n);
	 
	 } //EOF setCheckAK


	/**
	 * adds an entry to the checkDates Vector for later verifications.
	 *
	 * @param cdate   Date to insert
	 * @returns    The index of where the date was inserted
	 */
	 private static int setCheck(String cdate) {
	 		
	 	int r = checkDates.size();

	 	checkDates.insertElementAt(cdate, r);
	 		 	
	 	return r;
	 
	 } //EOF setCheck
	 
	 
	 /**
	  * tries to find a matching date for security check.
	  *
	  * @param cdate   Date to find & match
	  *
	  * @returns   A string array with the result of the matching vote and the associated AuthKey
	  */
	  private static String[] getCheck(String cdate) {
	  
	  	int i=0;
	  	boolean test=false;
	  	String[] r = new String[] {"",""};
	  	
		i = checkDates.indexOf(cdate);

		if (i==-1) {
			r[0] = "FALSE";
		} else {
			r[0] = checkVotes.elementAt(i).toString(); // collect vote
			r[1] = checkAuthKey.elementAt(i).toString(); // collect AuthKey
			checkDates.removeElementAt(i);  // clear data
 			checkVotes.removeElementAt(i);
 		}

	  	return r;
	  		
	} //EOF getCheck
	  
	  
	/**
	 * <code>process</code> analyses Strings sent from <code>TCPServer</code>
	 * and decides whether the data is in fact valid packets.
	 *
	 * If so then the appropriate replies are formulated and returned to 
	 * <code>TCPServer</code> for sending to the client.
	 *
	 * @param inputData   Contains the contents of a received packet
	 * @returns   String with a packet to return to client
	 */
	protected static String process(String inputData) throws Exception {

		String outputData="";
		
		if (inputData.charAt(0) == 'C') {
		
			/* vote or total confirm */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
			
				String t[];
			
				t = getCheck(p.getMessage()); //compare second time stamp
			
				/* respond according to regional or totaller mode */
				if (RTServer.serverType == 'R') {
					if (!t[0].equals("FALSE")) {  //date security ok
							/* register vote */														
							try {
							
								// AuthKey check, if ok store vote
								if (DBase.authKeyCheck(t[1])) {
								
									// now store the vote							
									DBase.storeVote(t[0]);
									RTServer.NORM.info("Vote stored in database");
									// confirm vote to user
									outputData = "C|OK|" + AuthSys.makeDigest("OK") + "\r\n";
									
								} else {
									// otherwise we have an error
									RTServer.NORM.error("AuthKey Check failed!");
									outputData = "ERROR";
								}
								
							} catch (Exception e) {
								RTServer.NORM.error("Vote registration error: " + e.getMessage());
								outputData = "ERROR";
							}
					} else {
							RTServer.NORM.warn("Vote security failure");
							outputData = "ERROR";
					}
				} else {
					if (!t.equals("FALSE")) {  //date security ok
							/* register total */
							try {
								DBase.storeTotal(t[0]);
								RTServer.NORM.info("Total stored in database");
								outputData = "C|OK|" + AuthSys.makeDigest("OK") + "\r\n"; //confirm total
							} catch (Exception e) {
								RTServer.NORM.error("Totals registration error: " + e.getMessage());
								outputData = "ERROR";
							}
					} else {
							RTServer.NORM.warn("Totals security failure.");
							outputData = "ERROR";
					}
				}
				
			} else {
				RTServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}
			
		} else if (inputData.charAt(0) == 'D') {
		
			/* diagnostic */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
			
				RTServer.NORM.info("Diagnostic packet received");
			
				if (p.getMessage().equalsIgnoreCase("TEST")) {
					outputData= "D|OK|" + AuthSys.makeDigest("OK") + "\r\n";
				} else {
					RTServer.NORM.warn("Unrecognized diagnostic command.");
					outputData = "ERROR";
				}
			
			} else {
				RTServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}
			
		} else if (inputData.charAt(0) == 'T') {
		
			/* total for party */
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
			
				if (p.getMessage().equals("DONE")) {
					RTServer.rServersDone = RTServer.rServersDone + 1;
					RTServer.NORM.info("Received " + RTServer.rServersDone + "/" + RTServer.rServers + " totals.");
					if (RTServer.rServersDone == RTServer.rServers) { // if all results in
						DBase.calcResult();
					}
					outputData = "C|DONE|" + AuthSys.makeDigest("DONE") + "\r\n"; // confirm
				} else {
					// commence security check			
					setVotes(setCheck(p.getMessage().substring(0,28)), p.getMessage().substring(29)); //store time stamp, and total packet
					outputData = "C|STAMP|" + AuthSys.makeDigest("STAMP") + "\r\n"; // ask for time stamp check
				}
			} else {
				RTServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}

		} else if (inputData.charAt(0) == 'V') {
		
			/* normal vote */
			if (inputData.length() > 120) {  // simple validity check to save work
			
				Packet p = new Packet(inputData);
			
				if (AuthSys.checkDigest(p)) {  // tamper check

					// split packet message into vote, AuthKey and time stamp					
					String check, vote, FCAuthKey;
					int i = 0;
					int t = 0;
					
					String split = p.getMessage();

					while (split.charAt(i)!='-') {
						i++;
					}
					check = split.substring(0,i);

					t = i+2;
					
					while (split.charAt(t)!='-') {
						t++;
					}
					
					vote = split.substring((i+1),t);									
					FCAuthKey = split.substring(t+1);
						
					if (vote.equals("")|vote.equals(" ")) {  // check to see if there is a party in the data
						RTServer.NORM.warn("No vote data in packet");
						outputData = "ERROR";
					} else {
						// commence security check						
						//store time stamp, and vote packet
						int checkIndex = setCheck(check);
						setVotes(checkIndex, vote);
						setCheckAK(checkIndex, FCAuthKey);
						outputData = "C|STAMP|" + AuthSys.makeDigest("STAMP") + "\r\n"; //ask for time stamp check
					}
				} else {
					RTServer.NORM.warn("Packet corrupted or altered!");
					outputData = "ERROR";
				}
			} else {
				RTServer.NORM.warn("Bad packet!");
				outputData = "ERROR";
			}
			
		} else if (inputData.charAt(0) == 'X') {
		
			/* end communication*/
			Packet p = new Packet(inputData);
			
			if (AuthSys.checkDigest(p)) {  // tamper check
			
				outputData = "DONE";
			} else {
				RTServer.NORM.warn("Packet corrupted or altered!");
				outputData = "ERROR";
			}

		} else {
				RTServer.NORM.warn("Unrecognized packet received.");
				outputData = "ERROR";
		}
		
		return outputData;

	}  //EOF process


} //EOF Class
