(require 'light "PATHNAME")

;;; TeX mode with lightning completion.
;;; Version of Thu Jun 18 15:27:12 1992
;;; Modified Fri May 28 14:13:06 PDT 1993
;;;
;;; Written by Mark Haiman and Nick Reingold.
;;; Comments, suggestions, etc. should be sent to 
;;; mhaiman@macaulay.udsd.edu

(provide 'ultex)

;;;
;;;  USER VARIABLES
;;;

(defvar ultex-tree-file "textree.el"
  "*Name of file containing the completion \"tree\"") 
(defvar ultex-latex-skel-file "skel.tex"
  "*File containing blank latex document")
(defvar ultex-greek-keys-file "greek.el"
  "*File defining the greek keymap")
(defvar ultra-tex-mode-hook nil
  "*Hook for Ultra TeX mode.")
(defvar ultra-tex-append-space-to-defs t
  "*If non-nil, add a space to TeX control sequence names grabbed from
file or buffer.")
(defvar ultex-skel-hook nil
  "*If non-nil this function is called when the ultex-latex-skel-file
is inserted.")

(defvar ultra-tex-mode-abbrev-table nil
  "Abbrev table used while in ultra-tex mode.")
(defvar ultra-tex-mode-syntax-table nil
  "Syntax table used while in ultra-tex mode.")

;; \C-c\C-b \C-c\C-v \C-c\C-p should be reserved for use by TeX-shell
(defvar ultra-tex-mode-map nil)
(if ultra-tex-mode-map 
    nil
  (setq ultra-tex-mode-map (make-keymap))
  (define-key ultra-tex-mode-map "\"" 'ultex-insert-quote)
  (define-key ultra-tex-mode-map "\\" 'ultex-fast-cs)
  (define-key ultra-tex-mode-map "\C-\\" 'ultex-fast-noslash)
  (define-key ultra-tex-mode-map "\C-m" 'ultex-over-and-out)
  (define-key ultra-tex-mode-map "\C-[\C-m" 'newline)
  (define-key ultra-tex-mode-map "\C-j" 'ultex-par)
  (define-key ultra-tex-mode-map "{" 'ultex-insert-braces)
  (define-key ultra-tex-mode-map "$" 'ultex-insert-dollars)
  (define-key ultra-tex-mode-map "^" 'ultex-superscript)
  (define-key ultra-tex-mode-map "." 'ultex-maybe-dots)
  (define-key ultra-tex-mode-map "_" 'ultex-subscript)
  (define-key ultra-tex-mode-map "\C-i" 'ultex-tab-resume)
  (define-key ultra-tex-mode-map "\C-cg" 'ultex-grab-defs)
  (define-key ultra-tex-mode-map "\C-c\C-r" 'ultex-reset-current-alist)
  (define-key ultra-tex-mode-map "\C-c\C-g" 'ultex-redefine-greek-key)
  (define-key ultra-tex-mode-map "\C-c\C-s" 'get-latex-skel)
  (define-key ultra-tex-mode-map "\C-c\C-n" 'ultex-new-cs)
  (define-key ultra-tex-mode-map "\C-c\C-d" 'ultex-delete-csname-light)
  (define-key ultra-tex-mode-map "\C-[A"
    '(lambda nil (interactive) (ultex-font "cal")))
  (define-key ultra-tex-mode-map "\C-[B"
    '(lambda nil (interactive) (ultex-font "bf")))
  (define-key ultra-tex-mode-map "\C-[C"
    '(lambda nil (interactive) (ultex-font "sc")))
  (define-key ultra-tex-mode-map "\C-[E"
    '(lambda nil (interactive) (ultex-font "em")))
  (define-key ultra-tex-mode-map "\C-[F"
    '(lambda nil (interactive) (ultex-font "sf")))
  (define-key ultra-tex-mode-map "\C-[I"
    '(lambda nil (interactive) (ultex-font "it")))
  (define-key ultra-tex-mode-map "\C-[M"
    '(lambda nil (interactive) (ultex-font "mit")))
  (define-key ultra-tex-mode-map "\C-[R"
    '(lambda nil (interactive) (ultex-font "rm")))
  (define-key ultra-tex-mode-map "\C-[S"
    '(lambda nil (interactive) (ultex-font "sl")))
  (define-key ultra-tex-mode-map "\C-[T"
    '(lambda nil (interactive) (ultex-font "tt")))
  (define-key ultra-tex-mode-map "\C-[Z"
    '(lambda nil (interactive) (ultex-font "zz")))
  (define-key ultra-tex-mode-map "\C-[U"
    '(lambda nil (interactive) (ultex-font "up")))
  (define-key ultra-tex-mode-map "`" 'ultex-greek-key))

(defvar ultex-greek-map (make-keymap))
(define-key ultex-greek-map "\e" (make-keymap))
(define-key ultex-greek-map "\\" 'plain)
(define-key ultex-greek-map "|" 'shift)
(define-key ultex-greek-map "\C-\\" 'control)
(define-key ultex-greek-map "\e\\" 'meta)
(define-key ultex-greek-map "\e|" 'meta-shift)
(define-key ultex-greek-map "\e\C-\\" 'meta-control)
(defvar ultex-greek-loaded nil)

;; load time initializations
(defvar ultex-permanent-alist '(nil))	; a cons so it can be changed
					; by side effect
(defvar ultex-current-alist ultex-permanent-alist)

(defun ultex-reset-current-alist ()
  "Reset the current alist to be the permanent alist."
  (interactive)
  (setq ultex-current-alist ultex-permanent-alist)
  (message "TeX \\csname alist reset."))

(defun ultex-reset-perm-alist (&optional find)
  "Reload (by side-effect) permanent alist from ultex-tree-file."
  (let ((alist (read (save-window-excursion
		       (progn
			 (and find (find-file ultex-tree-file))
			 (bury-buffer (current-buffer))
			 (goto-char (point-min))
			 (current-buffer))))))
    (rplaca ultex-permanent-alist (car alist))
    (rplacd ultex-permanent-alist (cdr alist))
    nil))

(defun ultex-set-reset-on-save nil
  "Arrange that permanent alist is reset whenever ultex-tree-file is saved."
  (save-window-excursion
    (find-file ultex-tree-file)
    (bury-buffer (current-buffer))
    (make-local-variable 'write-file-hooks)
    (setq write-file-hooks '(ultex-reset-perm-alist))))

;; entry point
(defun ultra-tex-mode nil
  "Major mode for editing TeX documents.  

\\ starts lightning completion of control sequences.  Use C-\\ for
those that don't start with \\.  Resume key is TAB.  C-c C-n makes new
permanent control sequences for completion.  The file ultex-tree-file
keeps permanent control sequences.  You can edit it. It will be saved
and reloaded automatically whenever you complete.  It must be readable
as an alist by the lisp reader.

` is Greek shift key.  Do `\\, `|, `C-\\, `M-\\, etc. for display of
greek keyboard with various shifts.  Your ultex-greek-keys-file
defines keys in ultex-greek-map.  C-c C-g makes greek key
redefinitions.

ESC A,B,C,E,F,I,M,R,S,T,Z make cAligraphic, Boldface, smallCaps,
Emphasis, sans-seriF, Italic, Math-italic, Roman, Slant, Typewriter
and uZer's-math (defined as \\zz) fonts.

RET moves forward from groupings when that makes sense.  Plain newline
is ESC RET.  LFD ends paragraph when that makes sense.

C-c C-s inserts a basic LaTeX outline from file ultex-latex-skel-file,
then runs ultex-skel-hook.

{ makes a brace pair.  $ makes a $ pair.  $$ makes displayed equation.
Repeated $ give $$ $$ or \\=\\[ \\=\\] format.  \" makes `` or ''.
... makes various \\dots.  ^ and _ include a brace pair.  TAB in array
environments inserts &.

\\\\ can make new \\item's by hooking ultex-next-item in your
completion list.   

Entry into this mode runs text-mode-hook and then ultra-tex-mode-hook.
Special functions: 
\\{ultra-tex-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (use-local-map ultra-tex-mode-map)
  (setq mode-name "UltraTeX")
  (setq major-mode 'ultra-tex-mode)
  (setq local-abbrev-table ultra-tex-mode-abbrev-table)
  (if (null ultra-tex-mode-syntax-table)
      (progn
	(setq ultra-tex-mode-syntax-table (make-syntax-table))
	(set-syntax-table ultra-tex-mode-syntax-table)
	(modify-syntax-entry ?\\ "\\")
	(modify-syntax-entry ?\$ "$$")
	(modify-syntax-entry ?\" ".")
	(modify-syntax-entry ?& ".")
	(modify-syntax-entry ?_ ".")
	(modify-syntax-entry ?@ "_")
	(modify-syntax-entry ?~ " ")
	(modify-syntax-entry ?% "<")
	(modify-syntax-entry ?\f ">")
	(modify-syntax-entry ?\n ">")
	(modify-syntax-entry ?' "w"))
    (set-syntax-table ultra-tex-mode-syntax-table))
  (make-local-variable 'completing-insert-function)
  (setq completing-insert-function 'ultex-fast-noslash)
  (make-local-variable 'require-final-newline)
  (setq require-final-newline t)
  (make-local-variable 'paragraph-start)
;; this first one is my version (jhp).  the second is the original version.
  (setq paragraph-start
	"^[ \t]*$\\|^[\f%]\\|^\\$\\$\\|\\$\\$$\\|^\\\\begin{\\|^\\\\end{\\|.*\\\\\\\\[ ]*$\\|^\\item\\|^\\\\\\[\\|^\\\\\\]")
  (setq paragraph-start
	"^[ \t]*$\\|^[\f%]\\|^\\$\\$\\|\\$\\$$\\|^\\\\begin{\\|^\\\\end{\\|\\\\\\\\$")
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "%+ *")
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'ultex-comment-indent)
  (ultex-reset-perm-alist t)
  (ultex-set-reset-on-save)
  (ultex-grab-defs)
  (run-hooks 'text-mode-hook 'ultra-tex-mode-hook))

;; completion of control sequences
(defun ultex-fast-cs nil
  "Lightning complete control sequence and argument framework."
  (interactive)
  (save-window-excursion (find-file ultex-tree-file)
			 (bury-buffer (current-buffer))
			 (if (buffer-modified-p) (save-buffer)))
  (insert "\\")
  (completing-insert ultex-current-alist nil 1 'point-adjust-hook
		     "TeX \\csnames"))

(defun ultex-fast-noslash nil
  "Lightning complete control sequence, including ones without backslash."
  (interactive)
  (save-window-excursion (find-file ultex-tree-file)
			 (bury-buffer (current-buffer))
			 (if (buffer-modified-p) (save-buffer)))
  (completing-insert ultex-current-alist nil 0 'point-adjust-hook
		     "TeX names"))

(defun ultex-delete-csname (str)
  " Delete the string STR from the current alist of
TeX control sequence names."
  (interactive  "sDelete TeX name: ")
  (let ((x (assoc str ultex-current-alist)))
    (if x
	(progn (setq ultex-current-alist
		     (delq x ultex-current-alist))
	       (message "%s deleted." str))
      (message "%s not found." str))))

(lightnify 'ultex-delete-csname-light 'ultex-delete-csname
	   'ultex-current-alist nil "TeX names")

(setq ultex-array-environs
      '(array tabular eqnarray eqnarray* align gather alignat aligned gathered
	      alignedat split CD matrix smallmatrix pmatrix bmatrix vmatrix
	      Vmatrix xalignat xxalignat cases))

(defun ultex-tab-resume nil
  "Resume lightning completion, or just TAB."
  (interactive)
  (or (completing-insert ultex-current-alist nil -1 'point-adjust-hook)
      (progn
	(and (memq (latex-environment) ultex-array-environs)
	     (insert "&"))
	(indent-for-tab-command))))

(defun ultex-new-cs (arg)
  "Accept a new lightning control sequence.  It pops up a buffer.  You
edit the control sequence there, leaving point at the right position.
When you exit, control-sequence is added to the permanent file and the
current alist.  With arg, the control-sequence will have high priority.
Can add hook to warn when used outside of math mode."
  (interactive "P")
  (let (s n x c)
    (save-window-excursion
      (pop-to-buffer " *TeX Completions Edit*")
      (erase-buffer)
      (insert "\\")
      (message "Make new completion, place point, then exit (ESC C-c).")
      (recursive-edit)
      (setq x (y-or-n-p "Use in math mode exclusively? "))
      (setq s (buffer-string))
      (setq n (- (point) (point-max))))
    (setq c (cons s (cons n (if x 'ultex-math-mode))))
    (setq x (assoc s ultex-permanent-alist))
    (save-window-excursion
      (find-file ultex-tree-file)
      (bury-buffer (current-buffer))
      (if arg (progn (goto-char (point-min))
		     (re-search-forward "(\n"))
	(goto-char (point-max))
	(re-search-backward "\n)"))
      (print c (current-buffer))
      (ultex-set-reset-on-save)
      (save-buffer))			; automatically resets permanent alist
    (if x (message "You have made a duplicate control sequence."))))

;; what about \let ??
(defun ultex-grab-defs (&optional fn)
  "Insert in the TeX completion alist templates for all new
\\newcommand's, \\DeclareMathOperator's, \\def's, \\label's,
\\newenvironment's, and \\newtheorem's in current buffer, or with arg,
from FILE.  If the buffer is read only then ask user if we should
really grab the defs." 
  (interactive "P")
  (and fn (not (stringp fn))
       (setq fn (expand-file-name (read-file-name "Grab defs from file: "))))
  (save-excursion
    (save-window-excursion
      (switch-to-buffer
       (if fn (find-file-noselect fn)
	 (current-buffer)))
      (if (and buffer-read-only
	       (not (y-or-n-p
		     (format  "Grab defs from buffer %s? "
			      (buffer-name (current-buffer))))))
	  (progn (message "Not grabbing defs from this buffer.")
		 nil)
	(goto-char (point-min))
	(if (search-forward "--ultex--dont--grab--"  3000 t) nil
	  ;; new module for (re)newtheorem
	  (goto-char (point-min))
	  (while (re-search-forward
		  "\\\\\\(re\\)?newtheorem\\s-*{\\([a-zA-Z]+\\)\\s-*}"
		  nil t)
	    (let* ((csname (buffer-substring (match-beginning 2)
					     (match-end 2))))
	      (ultex-add-def (list (concat "\\begin{"
					   csname
					   "}"
					   "\\label{}\n\n\\end{"
					   csname
					   "}\n")
				   (- (+ 10 (length csname)))))))
	  ;; new module for (re)newenvironment
	  (goto-char (point-min))
	  (while (re-search-forward
		  "\\\\\\(re\\)?newenvironment\\s-*{\\([a-zA-Z]+\\)\\s-*}"
		  nil t)
	    (let* ((csname (buffer-substring (match-beginning 2)
					     (match-end 2)))
		   (nargs (or (and (looking-at "\\[\\([1-9]\\)\\]")
				   (string-to-int (buffer-substring
						   (match-beginning 1)
						   (match-end 1))))
			      0)))
	      (ultex-add-def (list  (concat "\\begin{"
					    csname
					    "}"
					    (apply 'concat (make-list nargs "{}"))
					    "\\label{}\n\n\\end{"
					    csname
					    "}\n")
				    (+ (- (length csname))
				       -10
				       (if (= 0 nargs) 0
					 (+ 6 (* -2 nargs))))))))
	  (goto-char (point-min))
	  (while
	      (re-search-forward
	       "\\\\DeclareMathOperator[*]?\\s-*{\\(\\\\[a-zA-Z]+\\s-*\\)}"
	       nil t)
	    (let* ((csname (buffer-substring (match-beginning 1)
					     (match-end 1))))
	      (ultex-add-def (list (concat csname " ") 0))))
	  (goto-char (point-min))
	  (while
	      (re-search-forward
	       "\\\\\\(re\\)?newcommand\\s-*{?\\(\\\\[a-zA-Z]+\\|\\\\.\\)\\s-*}?"
	       nil t)
	    (let* ((csname (buffer-substring (match-beginning 2)
					     (match-end 2)))
		   (nargs (or (and (looking-at "\\[\\([1-9]\\)\\]")
				   (string-to-int (buffer-substring
						   (match-beginning 1)
						   (match-end 1))))
			      0)))
	      (if (and (= nargs 0) ultra-tex-append-space-to-defs)
		  (setq csname (concat csname " ")))
	      (ultex-add-def (list (concat csname
					   (apply 'concat (make-list nargs "{}")))
				   (if (= nargs 0) 0 (1+ (* -2 nargs)))))))
	  (goto-char (point-min))
	  (while (re-search-forward
		  "\\\\label{\\([^}]+\\)}" nil t)
	    (let ((csname (buffer-substring (match-beginning 1)
					    (match-end 1))))
	      (ultex-add-def (list (concat "\\ref{" csname "}") 0))
	      (ultex-add-def (list (concat "\\eqref{" csname "}") 0))
	      (ultex-add-def (list (concat "\\pageref{" csname "}") 0))))
	  (goto-char (point-min))
	  (while (re-search-forward
		  "\\\\def\\s-*\\(\\\\[a-zA-Z]+\\|\\\\.\\)\\([^{]*\\){"
		  nil t)
	    (let* ((csname (buffer-substring (match-beginning 1)
					     (match-end 1)))
		   (arg-temp (buffer-substring (match-beginning 2)
					       (match-end 2)))
		   (i 0)
		   (l (1- (length arg-temp))))
	      (string-match "\\s-*" arg-temp)
	      (if (/= (match-end 0) (1+ l)) nil (setq arg-temp "") (setq l -1))
	      (while (< i l)
		(if (/= (aref arg-temp i) ?#) nil
		  (aset arg-temp i ?{) (aset arg-temp (1+ i) ?}))
		(setq i (1+ i)))
	      (if (= l -1) (setq csname (concat csname " ")))
	      (setq i (if (setq i (string-match "{}" arg-temp)) (- i l) 0))
	      (ultex-add-def (list (concat csname arg-temp) i)))))))))

(defun ultex-add-def (def)
  "Push (<csname> <point-adjust> . <hook>) cons DEF on the current
alist if it isn't there already.  If already there, returns def from
alist, else returns nil."
  (or (assoc (car def) ultex-current-alist)
      (prog1 nil
	(setq ultex-current-alist (cons def ultex-current-alist)))))

;; intelligent RET key
(defconst ultex-endings " ?\\\\]
?\\| ?\\$\\$
\\|
?\\\\end{[^}]*}
?\\|\\$\\|\\]\\|[a-z]*}\\(
?\\\\label\\)?\\| ?{\\| ?\\\\}\\| ?\\\\)")

(defun ultex-over-and-out nil
  "Go up one level of \\], \\end{...}, $, ], or }; additionally skip
any immediately following {.  Otherwise, move forward cleverly.
Deletes empty [] and \\label{}."
  (interactive)
  (ultex-make-ref-from-label)
  (or (and (not (bobp))
	   (save-excursion (forward-char -1) (looking-at "\\[\\]"))
	   (progn (delete-char -1) (delete-char 1)
		  (skip-chars-forward "[") (skip-chars-forward "{")
		  t))
      (and (not (bobp))
	   (save-excursion (forward-char -7) (looking-at "\\\\label{}"))
	   (progn (delete-char -7) (delete-char (if (bolp) 2 1))
		  (skip-chars-forward "[") (skip-chars-forward "{")
		  t))
      (if (looking-at ultex-endings)
	  (progn (goto-char (match-end 0))
		 (skip-chars-forward "[") (skip-chars-forward "{"))
	(if (eolp) (if (looking-at "\n\n") (forward-char 1) (newline))
	  (if (looking-at "\\s-*\\\\?\\w+")
	      (re-search-forward "\\s-*\\\\?\\w+")
	    (forward-char 1)))))
  (if (and (eolp)
	   (ultex-macro-line-p))
      (progn (next-line 1) (beginning-of-line))))
  
(defun ultex-make-ref-from-label ()
  (let ((start (point))
	(labelname "")
	end)
    (save-excursion
      (and (looking-at "}")
	   (condition-case nil
	       (progn
		 (forward-char 1)
		 (forward-sexp -1)
		 (setq end (1+ (point)))
		 (forward-char -6)
		 (looking-at "\\\\label{\\([^}]+\\)}"))
	     (error nil))
	   (progn
	     (setq labelname (buffer-substring start end))
	     (message "Adding ref, eqref, and pageref for %s" labelname)
	     (ultex-add-def (list (concat "\\pageref{" labelname "} ")
				  0))
	     (ultex-add-def (list (concat "\\eqref{" labelname "} ")
				  0))
	     (ultex-add-def (list (concat "\\ref{" labelname "} ")
				  0)))))))

;; special predicates 
(defun ultex-macro-line-p nil
  (save-excursion
    (end-of-line)
    (and (save-excursion (beginning-of-line) (looking-at "\\\\"))
	 (save-restriction
	   (save-excursion
	     (narrow-to-region (- (point) (current-column)) (point))
	     (goto-char (point-min))
	     (re-search-forward
	      "^\\(\\\\def\\s-*\\)?\\\\.[a-zA-Z]*" nil t) ; skip leading cs
	     (let ((c t))
	       (while (not (eobp))
		 (setq c (and c (looking-at "{\\|\\[")))
		 (condition-case nil (forward-sexp 1)
		   (error (end-of-line) (setq c nil))))
	       c))))))

(setq ultex-math-start
  "\\\\\\[\\|\\\\(\\|\\\\begin{\\(eq[a-z*]*\\|x*align[a-z*]*\\|gather[a-z*]*\\|multiline\\|[a-z]*math\\)}")
(setq ultex-math-end
  "\\\\\\]\\|\\\\)\\|\\\\end{\\(eq[a-z*]*\\|x*align[a-z*]*\\|gather[a-z*]*\\|multiline\\|[a-z]*math\\)}")

(defvar ultex-math-cs-regexp nil
  "*Regexp to match any TeX control sequence whose argument is set in
math mode.  For example, if you have \\def\\foo{$#1$}, then the regexp
should match \"foo\".")

(setq ultex-math-cs-regexp "\\\\boxed")

(defvar ultex-non-math-cs-regexp nil
  "*Regexp to match any TeX control sequence whose argument is NOT set
in math mode, such as \\vbox.") 

(setq ultex-non-math-cs-regexp "\\\\[a-z]*box\\|\\\\[a-z]*text")

(defun ultex-math-mode (&optional arg)
  "t if point appears to be within math mode, else nil.  If nil, warns
user unless given optional ARG."
  (interactive)
  (let ((origin (point))
	(back t)
	(anybox (concat (or ultex-non-math-cs-regexp "")
			(if (and ultex-non-math-cs-regexp
				 ultex-math-cs-regexp)
			    "\\|" "")
			(or ultex-math-cs-regexp "")))
	(skip (list (point)))
	bstart c)
    (save-excursion
      (while back
	(re-search-backward (concat "\\`\\|\n\n\\|" ultex-math-start
				    "\\|" anybox) nil 'move)
	(setq bstart (point))
	(if (looking-at anybox) 
	    (condition-case nil
		(progn (forward-sexp 2)
		       (if (<= (point) origin)
			   (progn
			     (setq skip (cons bstart (cons (point) skip)))
			     (goto-char bstart))
			 (goto-char bstart)
			 (setq c (and ultex-math-cs-regexp
				      (looking-at ultex-math-cs-regexp))
			       back nil)))
	      (error 
	       (goto-char bstart)
	       (setq c (and ultex-math-cs-regexp
			    (looking-at ultex-math-cs-regexp))
		     back nil)))
	  (setq c (and (looking-at ultex-math-start)
		       (not (progn
			      (re-search-forward ultex-math-start origin t)
			      (re-search-forward ultex-math-end 
						 origin 'move))))
		back nil)
	  (goto-char bstart)))
      (while skip
	(if (re-search-forward "\\([^\\]\\|\\`\\)\\$\\|\\$\\$"
			       (car skip) 'move)
	    (setq c (not c))
	  (if (cdr skip)
	      (progn (goto-char (car (cdr skip)))
		     (setq skip (cdr (cdr skip))))
	    (setq skip nil))))
      (if (or c arg)  nil
	(message "You don't appear to be in math mode.") (ding))
      c)))

;; greek keyboard
(defun read-using-map (map &optional prompt)
  "Like read-key-sequence, but use KEYMAP.  PROMPT is optional."
  (let ((ogmap (current-global-map)) (olmap (current-local-map)))
    (use-global-map map) (use-local-map map)
    (prog1 (read-key-sequence prompt)
      (use-global-map ogmap) (use-local-map olmap))))

(defun ultex-greek-key nil
  "Read keys relative to greek map and insert corresponding control sequence."
  (interactive)
  (or ultex-greek-loaded (progn (load ultex-greek-keys-file)
				(setq ultex-greek-loaded t)))
  (let ((s (lookup-key ultex-greek-map (read-using-map ultex-greek-map))))
    (cond ((not s) (message "Undefined greek key" (ding)))
	  ((symbolp s) (ultex-show-keys s))
	  (t (ultex-math-mode)
	     (insert s)
	     (if (> (current-column) fill-column)
		 (run-hooks 'auto-fill-hook))))))

(defun ultex-redefine-greek-key nil
  "Ask for a key to get a new definition on the greek keyboard."
  (interactive)
  (or ultex-greek-loaded (progn (load ultex-greek-keys-file)
				(setq ultex-greek-loaded t)))
  (let (keys def)
    (setq keys (read-using-map ultex-greek-map "Redefine greek key: "))
    (if (assoc keys
	       '(("\\") ("|") ("\C-\\") ("\e\\") ("\e|") ("\eC-\\")))
	(error (format "Illegal to define greek key \"%s\"" keys)))
    (setq def (read-string (concat "Redefine greek key "
				   (key-description keys)
				   " to be: ")))
    (or (and (lookup-key ultex-greek-map keys)
	     (not (y-or-n-p (concat "Key " (key-description keys)
				    " was " (lookup-key ultex-greek-map keys)
				    ".  Replace old definition? ")))
	     (message "Not changed."))
	(progn (define-key ultex-greek-map keys def)
	       (and (file-writable-p ultex-greek-keys-file)
		    (y-or-n-p "Save change for future sessions? ")
		    (save-window-excursion
		      (switch-to-buffer " *Define Greek Key*")
		      (erase-buffer)
		      (print (list 'define-key 'ultex-greek-map keys def)
			     (current-buffer))
		      (append-to-file (1+ (point-min)) (point-max)
				      ultex-greek-keys-file)))))))
		      
(defun ultex-show-keys (arg)
  "Display a Greek shift (`) keyboard in a window.  ARG is 'plain, 'shift,
'control, 'meta, 'meta-shift, or 'meta-control."
  (interactive)
  (or ultex-greek-loaded (progn (load ultex-greek-keys-file)
				(setq ultex-greek-loaded t)))
  (let ((bufname (cdr (assoc arg
			     '((plain . "*greek keyboard*")
			       (shift . "*Greek Keyboard*")
			       (control . "*^Greek ^Keyboard*")
			       (meta . "*M-greek M-keyboard*")
			       (meta-shift . "*M-Greek M-Keyboard*")
			       (meta-control . "*M-C-Greek M-C-Keyboard"))))))
    (with-output-to-temp-buffer bufname
      (ultex-show-row ultex-123-row arg)
      (ultex-show-row ultex-qwe-row arg)
      (ultex-show-row ultex-asd-row arg)
      (ultex-show-row ultex-zxc-row arg))
    (let ((ow (selected-window))
	  (w (get-buffer-window (get-buffer bufname))))
      (select-window w)
      (enlarge-window (- 9 (window-height w)))
      (select-window ow))))

(defvar ultex-123-row '("" (?1 ?! . nil) (?2 ?@ . ?\C-@) (?3 ?# . nil)
			   (?4 ?$ . nil) (?5 ?% . nil) (?6 ?^ . ?\C-^)
			   (?7 ?& . nil) (?8 ?* . nil) (?9 ?( . nil)
			   (?0 ?) . nil) (?- ?_ . ?\C-_) (?= ?+ . nil)
			   (?` ?~ . nil)))
(defvar ultex-qwe-row '("   " (?q ?Q . ?\C-q) (?w ?W . ?\C-w) (?e ?E . ?\C-e)
			      (?r ?R . ?\C-r) (?t ?T . ?\C-t) (?y ?Y . ?\C-y)
			      (?u ?U . ?\C-u) (?i ?I . ?\C-i) (?o ?O . ?\C-o)
			      (?p ?P . ?\C-p) (?[ ?{ . ?\C-[) (?] ?} . ?\C-])))
(defvar ultex-asd-row '("    " (?a ?A . ?\C-a) (?s ?S . ?\C-s) (?d ?D . ?\C-d)
			       (?f ?F . ?\C-f) (?g ?G . ?\C-g) (?h ?H . ?\C-h)
			       (?j ?J . ?\C-j) (?k ?K . ?\C-k) (?l ?L . ?\C-l)
			       (?; ?: . nil) (?' ?" . nil)))
(defvar ultex-zxc-row '("      "
			(?z ?Z . ?\C-z) (?x ?X . ?\C-x) (?c ?C . ?\C-c)
			(?v ?V . ?\C-v) (?b ?B . ?\C-b) (?n ?N . ?\C-n)
			(?m ?M . ?\C-m) (?, ?< . nil) (?. ?> . nil)
			(?/ ?? . ?\C-?) (?  ?  . nil)))

(defun ultex-show-row (row arg)
  (princ (car row))
  (princ
   (mapconcat
    (function
     (lambda (x)
       (let (str char)
	 (setq char (cond ((memq arg '(plain meta)) (car x))
			      ((memq arg '(shift meta-shift)) (car (cdr x)))
			      (t (cdr (cdr x))))
	       str (and char
			(lookup-key
			 ultex-greek-map
			 (concat (if (memq arg '(plain shift control)) "" "\e")
				 (char-to-string char)))))
	 (substring (concat (if (stringp str) str "") "      ") 1 6))))
    (cdr row) " "))
  (terpri)
  (princ (car row))
  (princ (mapconcat
	  (function (lambda (x)
		      (concat "  " (char-to-string
				    (if (memq arg '(plain meta))
					(car x)
				      (car (cdr x))))
			      "  ")))
	  (cdr row) " "))
  (terpri))

;; miscellaneous utilities
(defun get-latex-skel nil
  "Read in a LaTeX outline."
  (interactive)
  (insert-file ultex-latex-skel-file)
  (run-hooks 'ultex-skel-hook))

(defun ultex-font (font)
  "Make a brace pair with font."
  (cond ((equal font "cal")
	 (insert (concat "\\mathcal{"
			 (char-to-string (upcase (progn (message "{\\cal ?}:")
							(read-char))))
			 "}")))
	((equal font "up")
	 (progn 
	   (insert "\\textup{")
	   (save-excursion (insert "}"))))
	((or (equal font "rm") (equal font "bf") (equal font "sc")
	     (equal font "sf") (equal font "it") (equal font "sl")
	     (equal font "tt"))
	 (progn
	   (insert (concat (if (ultex-math-mode t) "\\math" "\\text")
			   font "{"))
	   (save-excursion (insert "}"))))
	((equal font "em")
	 (progn
	   (insert (concat "\\emph{"))
	   (save-excursion (insert "}"))))
	(t
	 (insert (concat "\\" font "{"))
	 (save-excursion (insert "}"))))
  (if (equal font "cal") (ultex-math-mode)))

(setq ultex-dollar-array '[
      ("$$" . -1)
      ("\n$$\n\n$$" . -3) 
      ("\n\\[\n\n\\]".  -3)
      ("\n\\begin{equation}\\label{}\n\n\\end{equation}\n".-18)
      ("\n\\begin{equation*}\n\n\\end{equation*}\n".-17)
      ("\n\\begin{eqnarray}\n\n\\end{eqnarray}\n".-16)
      ("\n\\begin{eqnarray*}\n\n\\end{eqnarray*}\n".-17)
      ])

(setq ultex-dollar-max (length ultex-dollar-array))
(setq ultex-dollar-back 1)

(setq ultex-dollar-last -1)

(defun ultex-dollar-increment (i)
  (setq i (1+ i))
  (if (>= i ultex-dollar-max)
      ultex-dollar-back
    i))

(defun ultex-insert-dollars ()
  "Make a $ pair."
  (interactive)
  (let* ((next (ultex-dollar-increment ultex-dollar-last))
	 p str back backlen)
    (if (< ultex-dollar-last 0) nil
      (setq p (aref ultex-dollar-array ultex-dollar-last)
	    str (car p)
	    back (cdr p)
	    backlen (+ back (length str)))
      (save-excursion
	(and (> (point) backlen)
	     (backward-char backlen))
	(if (looking-at (regexp-quote str))
	    (delete-char (length str))
	  (setq next 0))))
    (setq p (aref ultex-dollar-array next)
	  str (car p)
	  back (cdr p))
    (insert (car p))
    (forward-char (cdr p))
    (setq ultex-dollar-last next)))

(defun ultex-maybe-dots nil
  "`.' unless it follows `..', then appropriate \\?dots.  Chooses \\ldots
over \\cdots after `,' or excess white space."
  (interactive)
  (if (or (< (current-column) 2)
	  (not (save-excursion (forward-char -2) (looking-at "\\.\\."))))
      (insert ".")
    (delete-char -2)
    (if (not (ultex-math-mode t))
	(insert "\\dots ")
      (if (and (save-excursion
		 (re-search-backward
		  ",\\s-*\\|\\W\\s-+\\|[^\\]\\b\\w+\\s-+"
		  (- (point) 30) t))
	       (= (match-end 0) (point)))
	  (if (and (save-excursion (re-search-backward ",\\s-*" nil t))
		   (= (match-end 0) (point)))
	      (insert "\\ldots,")
	    (insert "\\ldots "))
	(insert "\\cdots ")))
    (if (> (current-column) fill-column) (run-hooks 'auto-fill-hook))))

(defun ultex-superscript nil
  "Set up for superscript."
  (interactive)
  (ultex-math-mode)
  (insert "^{")
  (save-excursion (insert "}")))

(defun ultex-subscript nil
  "Set up for subscript."
  (interactive)
  (ultex-math-mode)
  (insert "_{")
  (save-excursion (insert "}")))

(defun ultex-par nil
  "Two newlines and validate if this looks like the end of paragraph,
otherwise plain newline."
  (interactive)
  (if (and (eolp) (not (bolp)) (not (ultex-math-mode t))
	   (not (ultex-macro-line-p)))
      (ultex-terminate-paragraph nil)
    (newline)))

;; From Ed Reingold, and no, we're not related.
(defun tex-last-unended-begin ()
  "Leave point at the beginning of the last \\begin{...} that is unended."
  (while (and (re-search-backward "\\(\\\\begin{\\)\\|\\(\\\\end{\\)")
              (looking-at "\\\\end{"))
    (tex-last-unended-begin)))

(defun latex-environment ()
  (let (not-found)
    (save-excursion
      (condition-case ERR
	  (tex-last-unended-begin)
	(error (setq not-found t)))
      (if not-found nil
	(re-search-forward "\\\\begin{\\([^}\n]*\\)}")
	(intern (buffer-substring (match-beginning 1) (match-end 1)))))))

(defvar itemizing-environments '(list trivlist itemize
				 description enumerate)
  "A list of names (symbols) of LaTeX environments which use \\\\item.")

(defun ultex-next-item ()
  "After end of an item in an itemize-type environment, drop any \\\\
on the previous line, and start this line with \\item[]"
  (and (memq (latex-environment) itemizing-environments)
       (progn (and (save-excursion 
		     (forward-char -3)
		     (looking-at "\\\\\\\\\n"))
		   (progn (delete-backward-char 3) 
			  (insert "\n")))
	      (insert "\\item []") (forward-char -1))))

;; Functions stolen (and corrected) from TeX mode.

;; The following fixes a bug that appears to be a problem when
;; 'forward-sexp has an argument > 1
(defun ultex-validate-paragraph (start end)
  (condition-case nil
      (save-excursion
	(save-restriction
	  (narrow-to-region start end)
	  (goto-char start)
;	  (forward-sexp (- end start))	; original version--it fails
	  (while (not (eobp)) (forward-sexp)) ; works, more efficiently too
	  t))
    (error nil)))

;; This fixes the bug on newline in auto-fill-mode
(defun ultex-terminate-paragraph (inhibit-validation)
  "Insert two newlines, breaking a paragraph for TeX.
Check for mismatched braces/$'s in paragraph being terminated.
A prefix arg inhibits the checking."
  (interactive "P")
  (or inhibit-validation
      (ultex-validate-paragraph
       (save-excursion
	 (search-backward "\n\n" nil 'move)
	 (point))
       (point))
      (progn 
	(message "Paragraph being closed appears to contain a mismatch")
	(ding)))
  (newline) (newline))			; (insert "\n\n") doesn't auto-fill

(defun ultex-insert-quote (count)
  "Insert ``, '' or \" according to preceding character.
With numeric arg N, always insert N \" characters."
  (interactive "P")
  (if count
      (self-insert-command count)
    (insert
     (cond
      ((or (bobp)
	   (save-excursion
	     (forward-char -1)
	     (looking-at "[ \t\n]\\|\\s(")))
       "``")
      ((= (preceding-char) ?\\)
       ?\")
      (t "''")))))

(defun ultex-comment-indent nil
  (if (looking-at "%%")
      (current-column)
    (skip-chars-backward " \t")
    (max (1+ (current-column)) comment-column)))

(defun ultex-insert-braces nil
  "Make a pair of braces and be poised to type inside of them."
  (interactive)
  (insert ?\{)
  (save-excursion
    (insert ?})))
