;;  Lightning completion
;;  Version of Mon Apr  6 16:08:35 1992

;;  Written by Mark Haiman and Nick Reingold.
;;  Comments, suggestions, etc. should be sent to 
;;  mhaiman@macaulay.ucsd.edu

;;  In-buffer completed insertion with same flexibility as completing-read
;;  and lightning action.  Lightning completion is a minor mode,
;;  signified "Light" on the mode line.
;;  
;;  Main entry point is
;;
;;  (completing-insert TABLE PRED INIT HOOK MESSAGE DISPLAY),
;;  
;;  which see.

;;  Keys controlling the action are determined by variables (default shown):

(defvar lc-keep-key 32
  "*Key to keep current completion in Light mode.") ; SPC
(defvar lc-del-key ?\177
  "*Key to delete last completion unit in Light mode.") ; DEL
(defvar lc-cycle-key ?\C-i
  "*Key to cycle to next completion in Light mode.")
(defvar lc-back-cycle-key ?\C-u
  "*Key to cycle to next completion back in Light mode.")
(defvar lc-display-key ?\C-m
  "*Key to display all completions in Light mode.")
(defvar lc-stop-key ?\C-c
  "*Key to exit Light mode.")
(defvar lc-quote-key ?\C-q
  "*Key to make next char be treated as printing")
(defvar lc-help-key ?\C-h
  "*Help key in Light mode.")

;;  Control keys quit, then do themselves.
;;  Printing keys complete if they can, else beep and warn.
;;  Keep key keeps what's displayed if complete and says so,
;;  else acts like a printing key.
;;  Del key goes back one completion unit or warns and quits if none.
;;  Cycle key traverses possible completions.  Back-cycle key traverses
;;  backwards.  Del here stops traversing.
;;  Display key shows all completions just like minibuffer-completion-help.
;;  Used repeatedly, scrolls around the display window.
;;  Stop key warns, quits without the beep.
;;  Quote key reads next char and treats it as printing.
;;  Help key says which keys these all are.

;;  Subsidiary entry points include:
;;  (word-grabber) gets the INIT argument for completing-insert.
;;  'point-adjust-hook does alist cdrs in ultex format: (point-adjust . hook).
;;  M-x completing-insert-kill.  (From kill ring).
;;  M-x completing-insert-buffer-name.
;;  M-x completing-insert-file-name.
;;  M-x completing-insert-lisp-object.
;;  M-x completing-insert-lisp-function.
;;  M-x completing-insert-lisp-variable.
;;  M-x completing-insert-according-to-mode: modes should set the variable
;;    completing-insert-function to nil or an appropriate command.
;;  'lightnify creates lightning-completing versions of functions that read
;;    the minibuffer interactively

(defvar light-mode nil)
(make-variable-buffer-local 'light-mode)
(or (featurep 'light)
    (setq minor-mode-alist (append minor-mode-alist
				   '((light-mode " Light")))))
(provide 'light)

(defun completing-insert (table pred init &optional hook message display)
  "Lightning-complete string before point in the buffer, relative to
completion TABLE; allowing only completions that satisfy PRED.  These
are used exactly as they are by completing-read, which means this:
  TABLE may be an alist, an obarray, or a function-symbol.  For an
alist, PRED applies to the entries (conses).  For an obarray, PRED
applies to the symbols.  A function symbol will be called with a
STRING as first arg, PRED as second arg and third arg nil, t, or
lambda; according to third arg, the function is supposed to return the
common completion of STRING, all its completions, or the truth-value
of its completeness.  In particular the function can be like
'read-file-name-internal, with PRED the name of a directory.
  Third arg INIT is the number of characters before point to complete
as the initial string.  Barf immediately if this is no match.  If
negative, we are resuming, so return nil unless situation at last quit
agrees with buffer before point; then restore that situation.
  Optional arg HOOK is run on successful completion; gets same kind of
argument as PRED, or the complete string if TABLE is a function symbol.
  On entering, message \"Completing <optional arg MESSAGE>...\" is
displayed.
  Optional arg DISPLAY is a function to call on each possible
completion before displaying.  If the DISPLAY function returns nil,
that string is NOT displayed."
  (condition-case x
      (if (not
	   (or
	    (and (>= init 0)		; starting fresh
		 (prog1			; if so, reset things and be t
		     t
		   (setq lc-stack nil)
		   (let ((grab (buffer-substring (- (point) init) (point)))
			 (n 0))
		     (while (<= n init)
		       (setq lc-stack (cons (substring grab 0 n) lc-stack))
		       (setq n (1+ n)))) ; completions=part grabs
		   (setq lc-table table lc-predicate pred
			 lc-hook hook lc-cycle nil
			 lc-display-filter display)))
	    ;; see if resuming state is consistent:
	    (and
	     lc-stack
	     (let ((state (and lc-cycle
			       (if (string-match
				    (concat "^" (regexp-quote
						 (car lc-stack)))
				    (aref (car lc-cycle) (cdr lc-cycle)))
				   t 'state))))
	       (and (or (eq state 'state)
			(and
			 (>= (point)
			     (+ (point-min) (length (car lc-stack))))
			 (string= (car lc-stack)
				  (buffer-substring
				   (- (point) (length (car lc-stack)))
				   (point)))))
		    (or (null state)
			(looking-at
			 (regexp-quote
			  (substring (aref (car lc-cycle) (cdr lc-cycle))
				     (if (eq state 'state)
					 0 (match-end 0))))))))
	     (eq table lc-table)
	     (equal pred lc-predicate)
	     (equal hook lc-hook)
	     (equal display lc-display-filter))))
	  nil				; trying to resume inconsistently
	(setq lc-old-keymap (current-local-map))
	(let ((map (make-keymap)) (i -1))
	  (while (<= (setq i (1+ i)) 31)
	    (define-key map (char-to-string i) 'lc-exit-and-then))
	  (while (<= (setq i (1+ i)) 126)
	    (define-key map (char-to-string i) 'lc-self-insert-char))
	  (define-key map (char-to-string 127) 'lc-exit-and-then)
	  (define-key map "\C-[" (make-vector 128 'lc-exit-and-then))
	  (if lc-keep-key
	      (define-key map (char-to-string lc-keep-key)
		'lc-keep-if-complete))
	  (if lc-del-key
	      (define-key map (char-to-string lc-del-key) 'lc-delete))
	  (if lc-cycle-key
	      (define-key map (char-to-string lc-cycle-key) 'lc-cycle-forward))
	  (if lc-back-cycle-key
	      (define-key map (char-to-string lc-back-cycle-key)
		'lc-cycle-backward))
	  (if lc-display-key
	      (define-key map (char-to-string lc-display-key)
		'lc-display-completions))
	  (if lc-stop-key
	      (define-key map (char-to-string lc-stop-key) 'lc-quit))
	  (if lc-quote-key
	      (define-key map (char-to-string lc-quote-key) 'lc-quote-char))
	  (if lc-help-key
	      (define-key map (char-to-string lc-help-key) 'lc-help))
	  (use-local-map map))
	(setq light-mode t)
	(set-buffer-modified-p (buffer-modified-p)) ; update mode line
	(setq lc-prev-windows (current-window-configuration))
	(if (or (> 0 init)
		(string= (car lc-stack) "") ; don't attempt to complete ""
		(let ((stat (lc-complete-stack-top "")))
		  (or (stringp stat) (prog1 nil (lc-quit stat)))))
	    (progn
	      (lc-message (concat "Completing" (if message " ") message "..."))
	      (while nil)))		; no-op
	t)				; return t except for bad resume
    (quit (setq unread-command-char ?\C-g))))


(defvar lc-old-keymap)			; keymap save
(make-variable-buffer-local 'lc-old-keymap)

;;state variables
(defvar lc-stack nil)			; top to agree with buffer
(make-variable-buffer-local 'lc-stack)	; except when cycling
(defvar lc-table nil)
(make-variable-buffer-local 'lc-table)
(defvar lc-predicate nil)
(make-variable-buffer-local 'lc-predicate)
(defvar lc-hook nil)
(make-variable-buffer-local 'lc-hook)
(defvar lc-cycle nil)			; nil if not cycling.  If cycling,
(make-variable-buffer-local 'lc-cycle)	; (completion-vector . number)

;(defvar lc-comp-window-prev-buffer nil)	; completions window former occupant
(defvar lc-prev-windows nil)		; state before completions window

(defvar lc-display-filter nil)
(make-variable-buffer-local 'lc-display-filter)

(defun lc-quit (arg &optional quick)
  "Exit lightning completion mode.  ARG nil means because of error.  ARG t
means because successful.  ARG other means intentional quit without being
complete.  Interactively, you get the last."
  (interactive '(lambda))
  (use-local-map lc-old-keymap)
  (setq light-mode nil)
  (set-buffer-modified-p (buffer-modified-p)) ; update mode line
  (or arg (ding))			; yell if an error
  (and lc-prev-windows (set-window-configuration lc-prev-windows))
  (if (eq arg t)
      (let ((name (car lc-stack)))
	(setq lc-stack nil)		; no resume after success
	(lc-message "Completed.")	; do here in case hook sends a message
	(if lc-hook			; on success, call possible hook
	    (funcall lc-hook
		     (cond ((vectorp lc-table) ; table is an obarray
			    (intern-soft name lc-table))
			   ((listp lc-table) ; table is an alist
			    (assoc name lc-table))
			   (t name))))	; table is a function
	(if (> (current-column) fill-column)  (run-hooks 'auto-fill-hook)))
    ;; unsuccessful quit:
    (setq lc-last-display-time nil)
    (lc-message (if arg "Stopped completing." "Can't complete.")
		(<= 0 unread-command-char))))

(defun lc-message (str &optional quick)
  "Same as message except in the minibuffer: then put message at point,
sit, and erase message.  Optional arg QUICK shortens the sit-for"
  (if (not (eq (selected-window) (minibuffer-window)))
      (message str)
    ;; in minibuffer!
    (setq str (concat " [" str "]"))
    (let ((inhibit-quit t))
      (save-excursion (insert str))
      (sit-for (if quick 0 2))
      (delete-char (length str)))))

(defun lc-switch-stack-top (str)
  "Replace top of stack with STR, fixing buffer."
  (let ((inhibit-quit t))
    (delete-backward-char (length (car lc-stack)))
    (insert str)
    (rplaca lc-stack str)))

(defun lc-pop-stack nil
  "Pop the stack, fixing buffer."
  (let ((inhibit-quit t))
    (delete-backward-char (length (car lc-stack)))
    (setq lc-stack (cdr lc-stack))
    (insert (car lc-stack))))

(defun lc-complete-stack-top (more)
  "If possible, replace what's on top of stack, and before point, with
the common completion of that extended by MORE, returning that.  Return
nil if no match.  If result is complete and unique, return t."
  (let* ((str (concat (car lc-stack) more))
	 ;; t:use real table. nil:truly no completions. alist:the completions
	 (all (or (symbolp lc-table)
		  (and (> (length str) 0) (= (aref str 0) ? ))
		  (mapcar 'list (all-completions str lc-table lc-predicate))))
	 (try (and all (try-completion str
				       (if (eq all t) lc-table all)
				       (if (eq all t) lc-predicate))))
	 (str (if (eq try t) str try)))
    (and try
	 (progn (lc-switch-stack-top str)
		(or (eq try t)
		    (try-completion str
				    (if (eq all t) lc-table all)
				    (if (eq all t) lc-predicate)))))))

;; key handlers
(defun lc-exit-and-then nil
  "Intentional unsuccessful quit, then put back char to be read again."
  (interactive)
  (setq unread-command-char
	(+ (aref (this-command-keys) (1- (length (this-command-keys))))
	   (if (> (length (this-command-keys)) 1) 128 0)))
  (lc-quit 'lambda))

(defun lc-self-insert-char nil
  "Complete with this char if possible or warn with no match"
  (interactive)
  (lc-stop-cycling)
  (setq lc-stack (cons (car lc-stack) lc-stack))
  (let ((stat (lc-complete-stack-top (char-to-string last-command-char))))
    (cond ((null stat) (let ((inhibit-quit t))
			 (setq lc-stack (cdr lc-stack))
			 (insert last-command-char)
			 (ding)
			 (lc-message "No match.")
			 (delete-char -1)))
	  ((eq t stat) (lc-quit stat)))))

(defun lc-keep-if-complete nil
  "Quit with success if current stack top is complete.  Otherwise
self-insert."
  (interactive)
  (if lc-cycle
      (let ((try t))
	(lc-absorb-cycling)
	(if (or (not (symbolp lc-table))
		(setq try (funcall lc-table (car lc-stack) lc-predicate nil)))
	    (if (eq try t) (lc-quit t)
	      (lc-switch-stack-top try))
	  (lc-pop-stack)
	  (lc-self-insert-char)))
    (if (cond ((listp lc-table)
	       (assoc (car lc-stack) lc-table))
	      ((vectorp lc-table)
	       (or (and (eq 'obarray lc-table)
			(string= "nil" (car lc-stack)))
		   (intern-soft (car lc-stack) lc-table)))
	      (t (funcall lc-table (car lc-stack) lc-predicate 'lambda)))
	(lc-quit t)
      (lc-self-insert-char))))

(defun lc-stop-cycling nil
  "Stop cycling, delete cycle shown."
  (and lc-cycle 
       (progn (lc-cycle-remove)
	      (setq lc-cycle nil))))

(defun lc-cycle-remove nil
  "Remove last cycle shown."
  (if (string-match (concat "^" (regexp-quote (car lc-stack)))
		    (aref (car lc-cycle) (cdr lc-cycle)))
      (delete-char (- (length (aref (car lc-cycle) (cdr lc-cycle)))
		      (match-end 0)))
    (delete-char (length (aref (car lc-cycle) (cdr lc-cycle))))
    (insert (car lc-stack))))

(defun lc-absorb-cycling nil
  "Stop cycling and push cycle shown on stack."
  (and lc-cycle 
       (progn
	 (lc-cycle-remove)
	 (setq lc-stack (cons (car lc-stack) lc-stack))
	 (lc-switch-stack-top (aref (car lc-cycle) (cdr lc-cycle)))
	 (setq lc-cycle nil))))

(defun lc-cycle-forward (arg)
  "Start cycling through completions, or cycle forward if already cycling."
  (interactive (list 1))
  (if lc-cycle
      (progn (lc-cycle-remove)
	     (rplacd lc-cycle (% (+ (length (car lc-cycle)) (cdr lc-cycle) arg)
				 (length (car lc-cycle)))))
    (setq lc-cycle
	  (cons (apply 'vector
		       (all-completions (car lc-stack) lc-table lc-predicate))
		0))
    (setq arg 0))
  (if (= 0 (length (car lc-cycle)))
      (progn (setq lc-cycle nil) (lc-message "No visible completions."))
    (delete-backward-char (length (car lc-stack)))
    (save-excursion
      (insert (aref (car lc-cycle) (cdr lc-cycle))))
    (if (string-match (concat "^" (regexp-quote (car lc-stack)))
		      (aref (car lc-cycle) (cdr lc-cycle)))
	(forward-char (match-end 0)))))

(defun lc-cycle-backward nil
  "Start cycling through completions, or cycle backward if already cycling."
  (interactive)
  (lc-cycle-forward -1))

(defun lc-delete nil
  "Go back one completion unit.  If cycling this means stop it.  If there
is no previous unit, quit quietly."
  (interactive)
  (if lc-cycle (lc-stop-cycling)
    (if (null (cdr lc-stack)) (lc-quit 'lambda)
      (lc-pop-stack))))

(defvar lc-last-display-time nil)	; "time" measured by stack top eq-ness

(defun lc-display-completions nil
  "Show possible completions, just like minibuffer-completion-help"
  (interactive)
  (lc-stop-cycling)
  (if (eq lc-last-display-time (car lc-stack))
      (let ((ow (selected-window))	; successive displays scroll
	    (w (get-buffer-window " *Completions*")))
	(select-window w)
	(condition-case nil (scroll-up) (error (goto-char (point-min))))
	(select-window ow))
    (setq lc-last-display-time (car lc-stack))
    (let ((all (all-completions (car lc-stack) lc-table lc-predicate))
	  results ans)

      (if (not (fboundp lc-display-filter)) nil
	(while all
	  (setq ans (funcall lc-display-filter (car all)))
	  (and ans
	       (setq results (cons ans results)))
	  (setq all (cdr all)))
	(setq all (nreverse results)))
      (if all
	  (with-output-to-temp-buffer " *Completions*"
	    (display-completion-list
	     (sort all 'string<)))
	(lc-message "No visible completions.")))))

(defun lc-quote-char nil
  "Quote the next key as printing character for lightning completion."
  (interactive)
  (let ((inhibit-quit t))
    (lc-message "^Q- ")
    (setq last-command-char (read-quoted-char))
    (lc-self-insert-char)))

(defun lc-help nil
  "Tell the completion control keys as a message."
  (interactive)
  (message
   (concat
    (and lc-keep-key (concat "Keep="
			     (single-key-description lc-keep-key) " "))
    (and lc-del-key (concat "Del="
			    (single-key-description lc-del-key) " "))
    (and lc-cycle-key (concat "Cycle="
			      (single-key-description lc-cycle-key)
			      (and lc-back-cycle-key
				   (concat ", " (single-key-description
						 lc-back-cycle-key)))
			      " "))
    (and lc-display-key (concat "Show-All="
				(single-key-description lc-display-key) " "))
    (and lc-stop-key (concat "Stop="
			     (single-key-description lc-stop-key) " "))
    (and lc-quote-key (concat "Quote="
			      (single-key-description lc-quote-key) " "))
    (and lc-help-key (concat "Help="
			     (single-key-description lc-help-key) " ")))))

;; utilities

(defun word-grabber nil
  "Moves point to just after the word point is in or after, and
returns length of word."
  (skip-chars-forward "^ \n\t\f\"`'();{}")
  (- (point) (save-excursion (skip-chars-backward "^ \n\t\f\"`'();{}")
			     (point))))

;; (defun word-grabber (&optional regexp) 
;;   "Moves point to just after the word point is in or after, and returns
;; length of word. Optional arg REGEXP is a regular expression which
;; matches any char which is NOT a word constituent.  If nil, the very
;; liberal \"[ \n\t\f\"`'();\{}]\" is used."
;;   (interactive)
;;   (or regexp (setq regexp "[ \n\t\f\"`'();{}]"))
;;   (re-search-forward regexp nil t)
;; ;; screws up unless at the end of the buffer
;;   (forward-char -1)
;;   (- (point) (save-excursion
;;	       (re-search-backward regexp nil t)
;;	       (forward-char 1)
;;	       (point))))

(defun point-adjust-hook (arg)
  "Intended to be used when lc-table is an alist whose elements look
like (<string> <number> . <hook>). Moves point forward <number> chars,
and then runs <hook> (if non-nil)."
  (forward-char (car (cdr arg)))
  (if (cdr (cdr arg)) (funcall (cdr (cdr arg)))))

;; file name completions		    
(defconst lc-literal-file-regexp
  "\\(\\(^\\|/\\)\\(~[^/]*\\|\\.\\.?\\)\\|\\${?[a-zA-Z0-9]*\\)$"
  "Regexp for file names which don't get completed, yet.")
(defconst lc-expand-this-file-regexp
  "\\(\\${[a-zA-Z0-9]*}\\|\\(^\\|/\\)\\.\\.?/\\)$"
  "Regexp for file names which get expanded before completion.")

(defun lc-read-file-name-internal (str dir action)
  "\"Internal\" subroutine for completing-insert-file-name. Do not call this."
  (let (str-dir real-str)
    (cond ((and (null action) (string-match lc-literal-file-regexp str))
	   str)
	  ((progn (setq real-str (expand-file-name
				  (substitute-in-file-name str) dir)
			str-dir (file-name-directory real-str))
		  (not (file-directory-p str-dir)))
	   nil)
	  ((eq action t)
	   (mapcar (function (lambda (x)
			       (expand-file-name x str-dir)))
		   (read-file-name-internal str dir action)))
	  (t
	   (let* ((exp (string-match lc-expand-this-file-regexp str))
		  (str (if exp real-str str))
		  (ans (read-file-name-internal str dir action)))
	     (if (null action)
		 (if (and exp (eq ans t)) str ans)
	       (and (not exp) ans)))))))

(setq lc-ignored-file-extensions
 (concat "\\(" 
	 (mapconcat 'regexp-quote completion-ignored-extensions "\\|")
	 "\\)$"))

(defun lc-file-display-filter (fn)
  (cond ((string-match lc-ignored-file-extensions fn)
	 nil)
	((file-directory-p fn)
	 (if (string= fn (expand-file-name "./"))
	     "./"
	   (if (string= fn (expand-file-name "../"))
	       "../"
	     (concat (file-name-nondirectory (directory-file-name fn))
		     "/"))))
	(t (file-name-nondirectory fn))))

(defun completing-insert-file-name (&optional dir init)
  "Complete file name in buffer at point.  Non-interactively, use directory
DIR (nil for current default-directory); start with INIT chars before point."
  (interactive (list nil (word-grabber)))
  (completing-insert 'lc-read-file-name-internal
		     (or dir default-directory) (or init 0)
		     nil "file names" 'lc-file-display-filter))

;; entry points for object, function, and variable completions
(defun completing-insert-lisp-object nil
  "Complete lisp object in buffer at point."
  (interactive)
  (completing-insert obarray nil (word-grabber) nil "lisp objects"))

(defun completing-insert-lisp-function nil
  "Complete lisp object in buffer at point."
  (interactive)
  (completing-insert obarray 'fboundp (word-grabber) nil "functions"))

(defun completing-insert-lisp-variable nil
  "Complete lisp object in buffer at point."
  (interactive)
  (completing-insert obarray 'boundp (word-grabber) nil "variables"))

;;entry point for buffer name completion
(defun completing-insert-buffer-name nil
  "Complete buffer name in buffer at point."
  (interactive)
  (completing-insert (mapcar (function (lambda (x) (list (buffer-name x))))
			     (buffer-list))
		     nil (word-grabber) nil "buffer names"))

;;entry point for completion from the kill ring
(defun completing-insert-kill nil
  "Complete something from the kill ring in buffer at point."
  (interactive)
  (completing-insert
   (mapcar 'list
	   (apply 'append
		  (mapcar
		   (function
		    (lambda (x)
		      (cons x (and (string-match "\\s-+" x)
				   (list (substring x (match-end 0)))))))
		   kill-ring)))
   nil 0 nil "recent kills"))

;;entry point for universal completion
(defvar completing-insert-function nil
  "*Function to be called by M-x completing-insert-according-to-mode, 
if non-nil")
(make-variable-buffer-local 'completing-insert-function)

(defun completing-insert-according-to-mode nil
  "Start lightning completion.  If possible, resumes stopped
completion.  Otherwise, in the minibuffer, uses its table and
predicate (slightly modified for file name reading).  Failing that,
calls completing-insert-function if the mode has it set.  Final
default is lisp-object completion."
  (interactive)
  (cond ((completing-insert lc-table lc-predicate -1 lc-hook) nil)
	((and (minibuffer-window-active-p (minibuffer-window))
	      (rassq 'minibuffer-complete (current-local-map)))
	 (let* ((table (if (eq minibuffer-completion-table
			       'read-file-name-internal)
			   'lc-read-file-name-internal
			 minibuffer-completion-table))
		(message
		 (cond ((eq table 'lc-read-file-name-internal)
			"file names")
		       ((and (listp table) (bufferp (cdr (car table))))
			"buffers")
		       ((eq obarray table)
			(cond ((not
				(and (boundp
				      'minibuffer-completion-predicate)
				     minibuffer-completion-predicate))
			       "lisp objects")
			      ((eq 'fboundp minibuffer-completion-predicate)
			       "functions")
			      ((eq 'commandp minibuffer-completion-predicate)
			       "commands")
			      ((eq 'boundp minibuffer-completion-predicate)
			       "variables")
			      ((eq 'user-variable-p
				   minibuffer-completion-predicate)
			       "user variables")))
		       (t "something")))
		(display (and (eq table 'lc-read-file-name-internal)
			      'lc-file-display-filter)))
	   (or (completing-insert table minibuffer-completion-predicate
				  -1)
	       (completing-insert table minibuffer-completion-predicate
				  (progn (goto-char (point-max))
					 (- (point) (point-min)))
				  nil message display))))
	;; I moved this here to make existing minibuffer
	;; completion info take precedence over stopped completion.
	;; -- Nick Reingold 5/24/92
	((completing-insert lc-table lc-predicate -1
			    lc-hook lc-display-filter) nil)
	(completing-insert-function
	 (call-interactively completing-insert-function))
	(t (completing-insert-lisp-object))))

;;lightnification
(defun eval-unquotes (arg)
  (cond ((atom arg)
	 arg)
	((eq (car arg) 'unquote)
	 (eval (car (cdr arg))))
	(t
	 (mapcar 'eval-unquotes arg))))

(defvar lightnify-saved-ctl-at (aref (car (cdr global-map)) 0)
  " Variable to hold the previous binding for \C-@.")

(make-variable-buffer-local 'lightnify-saved-ctl-at)

(defun lightnify (new old &optional table pred message hook display)
  "Make symbol NEW a function that calls function OLD interactively,
but set up so it reads its input in lightning completion mode.
Optional args TABLE PRED MESSAGE HOOK DISPLAY will be eval'led at the
time NEW is invoked to determine the completion environment.  Without
them completion is according to mode."
  (fset new 
	(eval-unquotes
	 '(lambda (pre)
	    (unquote (concat "Lightnified version of " (symbol-name old) "."))
	    (interactive "P")
	    (setq lightnify-saved-ctl-at (aref (car (cdr global-map)) 0))
	    (let ((prefix-arg pre)
		  err-info (lightnify-table (unquote table))
		  (lightnify-predicate (unquote pred))
		  (lightnify-message (unquote message))
		  (lightnify-hook (unquote hook))
		  (lightnify-display-filter (unquote display)))
	      (define-key global-map "\C-@" (quote lightnify-internal))
	      (unwind-protect
		  (progn 
		    (mapcar 
		     (quote 
		      (lambda (b)
			(save-window-excursion
			  (set-buffer b)
			  (make-local-variable (quote unread-command-char)))))
		     (buffer-list))
		    (set-default (quote unread-command-char) 0)
		    (condition-case x
			(unwind-protect
			    (call-interactively (quote (unquote old)))
			  (define-key global-map "\C-@" lightnify-saved-ctl-at)
			  (set-default (quote unread-command-char) -1)
			  (mapcar 
			   (quote 
			    (lambda (b)
			      (save-window-excursion
				(set-buffer b)
				(kill-local-variable 
				 (quote unread-command-char)))))
			   (buffer-list)))
		      (error (setq err-info x))
		      (quit (setq err-info x)))
		    (and err-info (signal (car err-info)
					  (cdr err-info))))
		(define-key global-map "\C-@" lightnify-saved-ctl-at)))))))

(defun lightnify-internal (pre)
  (interactive "P")
  (setq prefix-arg pre)
  (define-key global-map "\C-@" lightnify-saved-ctl-at)
  (if lightnify-table
      (completing-insert
       lightnify-table lightnify-predicate 0
       lightnify-hook lightnify-message lightnify-display-filter)
    (setq lc-stack nil)
    (completing-insert-according-to-mode)))
