;;  Buffer completion
;;  
;;  Written by Mark Haiman and Nick Reingold.
;;  Comments, suggestions, etc. should be sent to 
;;  mhaiman@macaulay.ucsd.edu

;;  Mechanism for adapting lightning completion to complete on reasonably
;;  balanced substrings of a buffer.
;;  Main entry point is
;;
;;	(completing-insert-buffer-contents BUF)
;;
;;  where BUF is interactively the current buffer or, with arg, a buffer
;;  specified by the user.

(require 'light "PATHNAME")

(defun buffer-sub-hunk (start end)
  "Return substring of current buffer from START at least up to END, extended
sufficiently to be balanced if possible, but in any case not to include
more than one non-blank line past END."
  (save-excursion
    (goto-char end)
    (skip-chars-forward "\n")
    (skip-chars-forward "^\n")
    (save-restriction
      (narrow-to-region start (point))
      (goto-char start)
      (let (n)
	(while (< (point) end)
	  (condition-case what (goto-char (setq n (scan-sexps (point) 1)))
	    (error (if (or (null n) (= ?U (aref (car (cdr what)) 0)))
		       (goto-char (point-max))
		     (forward-char 1))))))
      (buffer-substring start (point)))))

(defvar buf-comp-internal-last nil)	; last return of a try-type call

(defun buffer-completion-internal (str buf action)
  "Internal subroutine for completing-insert-buffer-contents.  Do not
call this.
  Used like read-file-name-internal but for completing STR as a substring
of buffer BUF.  Completing with space as last char matches anything,
as long as the match is unique.  ACTION nil means common part of 
proper extensions of STR, up to next sexp boundary, t means list of
some of these extensions.  Other means return nil (no substring is ever
considered complete)."
  (and
   (memq action '(nil t))		; never complete so keep is disabled
   (save-window-excursion
     (let* ((obuf (prog1 (current-buffer) (set-buffer buf)))
	    inhibit-quit case-fold-search find (l (length str)))
       (prog2
	(if (eq buf obuf)		; hide completion in progress
	    (progn (setq inhibit-quit t)
		   (delete-backward-char (length (car lc-stack)))))
	(if action
	    (let ((oball (make-vector 37 0)) (n 700))
	      (save-excursion
		(goto-char (point-min))
		(while (and (< 0 (setq n (1- n)))
			    (search-forward str nil t))
		  (intern (buffer-sub-hunk (match-beginning 0)
					   (min (point-max) (1+ (point))))
			  oball))
		(if (< 0 n) (all-completions "" oball)
		  '("Completions too numerous to mention!"))))
	  (setq				; this arranges that identical repeats
	   buf-comp-internal-last	; of a try call do no work, speeding
	   (if (eq str buf-comp-internal-last) str ; up lc-complete-stack-top.
	     (save-excursion
	       (goto-char (point-min))
	       (or
		(and
		 (search-forward str nil t)
		 (setq find (buffer-sub-hunk (match-beginning 0) (point)))
		 (progn
		   (while (and (> (length find) l) (search-forward str nil t))
		     (setq find (try-completion
				 ""
				 (list (list find)
				       (list (buffer-substring
					      (match-beginning 0)
					      (min (point-max)
						   (+ (match-beginning 0)
						      (length find)))))))))
		   find))
		(and (string-match "\\s-" (substring str -1))
		     (search-forward (setq str (substring str 0 -1)) nil t)
		     (setq find (buffer-sub-hunk (match-beginning 0)
						 (min (point-max)
						      (1+ (point)))))
		     (progn
		       (setq l (1- l))
		       (while (and (> (length find) l)
				   (search-forward str nil t))
			 (setq find (try-completion
				     ""
				     (list (list find)
					   (list (buffer-substring
						  (match-beginning 0)
						  (min (point-max)
						       (+ (match-beginning 0)
							  (length find)))))))))
		       (and (> (length find) l) find))))))))
	;; unhide:
	(if (eq buf obuf) (insert (car lc-stack))))))))

(defun completing-insert-buffer-contents  (&optional buf)
  "Complete on substrings of BUF extending to sexp boundaries.  String is
never complete, so exit with C-c.  Once unique, space means match more.
Interactively, with arg, ask for the buffer, else current buffer."
  (interactive "P")
  (if (and (interactive-p) buf)
      (setq buf (read-buffer "Complete from buffer: ")))
  (setq buf (or buf (current-buffer)))
  (completing-insert 'buffer-completion-internal buf 0 nil "buffer contents"))
