/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini, www.lorenzobettini.it
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "startapp.h"

#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <string.h>

#include "colors.h"
#include "cmdline.h"
#include "genfun.h"
#include "fileutil.h"
#include "messages.h"

#include "copyright.h"
#include "reportbugs.h"
#include "parsetags.h"

#include "htmlgeneratorfactory.h"
#include "xhtmlgeneratorfactory.h"
#include "cssgeneratorfactory.h"
#include "escgeneratorfactory.h"
#include "srcuntabifier.h"
#include "chartranslator.h"

// for globals
#include "linenumdigit.h"
#include "globalostream.h"
#include "cmdlineargs.h"
#include "generators.h"
#include "globalcolormap.h"
#include "maingeneratorfactory.h"
#include "mainoutputbuffer.h"
#include "mainoutputgenerator.h"

using std::cout;
using std::endl;

// global output stream
ostream* sout;

#ifdef BUILD_AS_CGI
#include "envmapper.h"
#endif // BUILD_AS_CGI


unsigned int line_num_digit = 0; // num of digits to represent line number

gengetopt_args_info args_info ;     // command line structure

ColorMap *MapOfColors;

static void print_cgi_header();

StartApp::StartApp() :
  entire_doc (0), verbose (0), cssUrl (0),
  use_css (0), is_cgi (0)
{
}

int
StartApp::start(int argc, char * argv[])
{
  char *docTitle;
  char *docHeader; // the buffer with the header
  char *docFooter; // the buffer with the footer
  const char *header_fileName = 0;
  const char *footer_fileName = 0;
  unsigned i;
  int v;
  int tabSpaces = 0;

#ifdef BUILD_AS_CGI
  // map environment to parameters if used as CGI
  char **temp_argv;
  temp_argv = map_environment(&argc, argv);
  is_cgi = temp_argv != argv;
  argv = temp_argv;
#endif // BUILD_AS_CGI

  if((v = cmdline_parser(argc, argv, &args_info)) != 0)
    // calls cmdline parser. The user gived bag args if it doesn't return -1
    return 1;

  if (args_info.version_given)
    {
      print_version ();
      print_copyright ();
      exit (0);
    }

  if (args_info.help_given)
    {
      cout << "GNU ";
      cmdline_parser_print_help ();
      print_reportbugs ();
      exit (0);
    }

  bool format_html = (strcmp (args_info.out_format_arg, "html") == 0);
  bool format_xhtml = (strcmp (args_info.out_format_arg, "xhtml") == 0);
  bool format_esc = (strcmp (args_info.out_format_arg, "esc") == 0);
  bool gen_version = (args_info.gen_version_flag != 0);

  const char *ext = 0;

  if (! format_html && ! format_xhtml && ! format_esc)
    {
      cerr << PACKAGE << ": output format " << args_info.out_format_arg
           << " not recognized" << endl;
      exit (1);
    }

  if (format_html || format_xhtml)
    ext = ".html";
  else
    ext = ".txt";

  /* initialization of global symbols */
  inputFileName = outputFileName = 0 ;
  sout = 0 ;
  docTitle = 0 ;
  docHeader = 0 ;
  docFooter = 0 ;

  docTitle = args_info.title_arg ;
  header_fileName = args_info.header_arg ;
  footer_fileName = args_info.footer_arg ;
  verbose = args_info.verbose_given ;
  const string tags_file = args_info.tags_file_arg;

  if ( args_info.tab_given > 0 )
    tabSpaces = args_info.tab_arg ;

  if (header_fileName)
    docHeader = read_file (header_fileName);

  if (footer_fileName)
    docFooter = read_file (footer_fileName);

  cssUrl = args_info.css_arg ;
  use_css = ( cssUrl != 0 ) ;

  entire_doc =
    (! args_info.no_doc_given) &&
    ( args_info.doc_given || (docTitle != 0) || use_css ||
      docHeader || docFooter ) ;

  inputFileName = args_info.input_arg ;
  if ( inputFileName && ! is_cgi )
    outputFileName = args_info.output_arg ;

  bool generate_to_stdout =
    (args_info.output_arg &&
     strcmp (args_info.output_arg, "STDOUT") == 0);

  if ( verbose )
    setMessager( new DefaultMessages ) ;

  printMessage( PACKAGE ) ;
  printMessage( VERSION ) ;

  string data_dir;
  if (args_info.data_dir_given)
    data_dir = args_info.data_dir_arg;
  
  string lang_map = args_info.lang_map_arg;

  if (args_info.lang_list_given) {
    istream *langs = open_data_file_istream(data_dir, lang_map);
    cout << "Supported languages (file extensions) and associated language definition files\n\n";
    cout << langs->rdbuf() << endl;
    delete langs;
    exit(EXIT_SUCCESS);
  }
  
  parseTags(data_dir, tags_file) ;

  outputbuffer = new OutputBuffer;

  if (format_esc)
    {
      if (use_css)
        {
          cerr << PACKAGE << ": cannot use css with esc output format" << endl;
          exit (1);
        }

      if (args_info.line_number_ref_given)
        {
          cerr << PACKAGE << ": cannot generate line references in esc output format" << endl;
          exit (1);
        }

      generator_factory = new EscGeneratorFactory;
    }
  else
    {
     string title;
     string input_name;
     string doc_header;
     string doc_footer;
     string css_url;

     if (docTitle)
       title = docTitle;
     if (inputFileName)
       input_name = inputFileName;
     if ((! docTitle) && inputFileName)
        title = inputFileName;
     if (docHeader)
       doc_header = docHeader;
     if (docFooter)
       doc_footer = docFooter;
     if (cssUrl)
       css_url = cssUrl;

     if (args_info.line_number_ref_given)
       args_info.line_number_given = args_info.line_number_ref_given;

     if( use_css )
       {
         generator_factory =
           new CssGeneratorFactory (title,
                                    input_name, doc_header,
                                    doc_footer, css_url);
       }
     else
       {
         if (format_xhtml && ! args_info.css_given)
           generator_factory =
             new XHtmlGeneratorFactory
              (title,
               input_name, doc_header,
               doc_footer, css_url);
         else
           generator_factory =
            new HtmlGeneratorFactory
             (title,
              input_name, doc_header,
              doc_footer, css_url);
       }
     }

  generator_factory->createGenerators ();

  if ( is_cgi )
    print_cgi_header() ;

  // let's start the translation :-)

  bool generate_line_num =
    (args_info.line_number_given || args_info.line_number_ref_given);
  
  TextFormatter *formatter = 0;
  if (tabSpaces)
    formatter = new Untabifier (tabSpaces);
  else if (args_info.line_number_given)
    formatter = new Untabifier(8);
  else
    formatter = new TextFormatter();
  
  CharTranslator *chartranslator = generator_factory->getCharTranslator();
  
  formatter->setFormatter(chartranslator);
  
  // first the --input file
  if ( ! args_info.inputs_num ) {
    processFile( args_info.src_lang_arg, args_info.out_format_arg,
                     inputFileName,
                 (generate_to_stdout ? 0 : outputFileName),
                 entire_doc, gen_version,
                 generate_line_num, args_info.line_number_ref_given,
                 args_info.lang_def_arg, lang_map, data_dir, formatter) ;
    outputbuffer->reset();
  }

  // let's process other files, if there are any
  if ( args_info.inputs_num && !is_cgi ) {
    for ( i = 0 ; i < (args_info.inputs_num) ; ++i ) {
      processFile( args_info.src_lang_arg, args_info.out_format_arg,
                         args_info.inputs[i],
                   (generate_to_stdout ? 0 :
                    (char *) (createOutputFileName (args_info.inputs[i],
                                                    args_info.output_dir_arg, ext)).c_str ()),
                   entire_doc, gen_version,
                   generate_line_num, args_info.line_number_ref_given,
                   args_info.lang_def_arg, lang_map, data_dir, formatter) ;
      cerr << "Processed " << args_info.inputs[i] << endl ;
      outputbuffer->reset();
    }
  }

  delete outputbuffer;
  outputbuffer = 0;
  delete formatter;

  return (0 );
}

void
StartApp::print_copyright()
{
  int i;

  for (i = 1; i <= copyright_text_length; ++i)
    cout << copyright_text[i] << endl;;
}

void
StartApp::print_reportbugs()
{
  int i;

  for (i = 1; i <= reportbugs_text_length; ++i)
    cout << reportbugs_text[i] << endl;
}

void
StartApp::print_version()
{
  cout << "GNU " << PACKAGE << " " << VERSION << endl;
}

void
print_cgi_header()
{
  printf( "Content-type: text/html\n" ) ;
  printf( "\n" ) ;
}

