/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <stdlib.h>

#include "genfun.h"

#include "messages.h"
#include "tags.h"
#include "keys.h"
#include "fileutil.h"
#include "docgenerator.h"
#include "lineoutputgenerator.h"
#include "regexpengine.h"
#include "langmap.h"

// globals
#include "linenumdigit.h"
#include "generators.h"
#include "mainoutputbuffer.h"
#include "mainoutputgenerator.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

using namespace std;;

void
processFile( const char *source_language, const char *output_format,
             const char *inputFileName, const char *outputFileName,
             short entire_doc, bool gen_version, bool gen_line, bool gen_ref,
             const char *lang_file, const string &lang_map, const string &data_dir,
             TextFormatter *formatter )
{
  FILE *in = 0;
  bool deleteOStream = false ;

  if ( outputFileName ) {
    sout = new ofstream(outputFileName) ;
    if ( ! (*sout) ) {
      cerr << "Error in creating " << outputFileName << " for output" << endl ;
      exit(1) ;
    }
    deleteOStream = true;
  }

  if (inputFileName)
    {
      unsigned int lines = get_line_count (inputFileName);

      line_num_digit = 0;
      while (lines)
        {
          ++line_num_digit;
          lines /= 10;
        }
    }
  else
    line_num_digit = 5;
  // if we read from stdin, we can't read the file in advance and
  // check how many lines of code it contains.  In this case set
  // the number of digit for the line number to 5.

  /*
   * Use default values for any options not provided
   */
  if (sout == 0) {
    sout = &cout;
  }

  if (in == 0) {
    ; /* Well stdin already points to stdin so, .... */
  }

  if (gen_line)
    outputgenerator = new LineOutputGenerator(*outputbuffer, *sout, gen_ref);
  else
    outputgenerator = new OutputGenerator(*outputbuffer, *sout);

  if ( entire_doc ) {
    GlobalDocGenerator->set_gen_version (gen_version);
    GlobalDocGenerator->generate_top ();
  }

  printMessage( "translating source code... ", cerr ) ;

  GlobalDocGenerator->generate_start_doc ();

  string langfile;
  if (lang_file)
    langfile = lang_file;
  
  if (!langfile.size()) {
    // find the language definition file associated to a language
    LangMap langmap(data_dir, lang_map);      
    if (source_language) {
      langfile = langmap.get_file(source_language);
      if (! langfile.size())
        {
          cerr << PACKAGE << ": ";
          cerr << "source language " << source_language
              << " not handled" << endl;
          exit (1);
        }
    } else {
      if (! inputFileName)
        {
          cerr << PACKAGE << ": ";
          cerr << "when using stdin, please specify a source language"
                << endl;
          exit (1);
        }
  
      string file_ext = get_file_extension (inputFileName);
  
      if (file_ext == "")
        {
          cerr << PACKAGE << ": ";
          cerr << "no file extension; please specify a source language"
                << endl;
          exit (1);
        }
    
      langfile = langmap.get_file(file_ext);
      if (! langfile.size())
        {
          cerr << PACKAGE << ": ";
          cerr << "unknown file extension " << file_ext << endl;
          exit (1);
        }
    }
  }

  const string &i_file_name = get_input_file_name(inputFileName);
  const char *input_file_name = (i_file_name.size() ? i_file_name.c_str() : 0);
  
  process_file(input_file_name, formatter, data_dir, langfile);
  
  outputgenerator->generate();

  GlobalDocGenerator->generate_end_doc ();

  printMessage( "done !", cerr ) ;

  if ( entire_doc ) {
    GlobalDocGenerator->generate_bottom ();
  }

  sout->flush ();

  if ( deleteOStream )
    delete sout ;

  delete outputgenerator;
}

