# GNU Solfege - eartraining for GNOME
# Copyright (C) 2001, 2002  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys, os, os.path
if __name__ == "__main__":
    sys.path.append(".")
import gettext, i18n
import re, string

tokens = ('NAME', 'STRING', 'INTEGER', 'CHAR', 'BLOCKSTART', 'FUNCTIONCALL')

for t in tokens:
    globals()[t] = t

re_FUNCTION = re.compile("(?P<func>[a-zA-Z]+\w*?)\((?P<params>.*?)\)")#FIXME

NEW_re = re.compile("""(?:
                        (\s+)|
                        (\#.*?$)|
                        (-?\d+)|
                        (\"\"\"(.*?)\"\"\")|
                        ("(.*?)")|
                        ([_\w]+[a-zA-Z-_1-9]*)
                )""",
                      re.VERBOSE|re.MULTILINE|re.DOTALL)

LI_INTEGER = NEW_re.match("-3").lastindex
LI_MSTRING = NEW_re.match('"""string"""').lastindex
LI_STRING = NEW_re.match('"string"').lastindex
LI_NAME = NEW_re.match("name").lastindex
LI_COMMENT = NEW_re.match("# comment").lastindex

lastindex_to_ID = {LI_INTEGER: INTEGER,
                     LI_STRING: STRING,
                     LI_MSTRING: STRING,
                     LI_NAME: NAME,
                    }

lastindex_to_group = {LI_INTEGER: 3,
                     LI_STRING: 7,
                     LI_MSTRING: 5,
                     LI_NAME: 8,
                    }

class DataparserException(Exception):
    def __init__(self, filename, lineno, badcode):
        self.filename = filename
        self.lineno = lineno
        self.badcode = badcode
    def __str__(self):
        return "Failed to parse lessonfile '%s'.\nError at line %s." % (self.filename, self.lineno)


class Dataparser:
    def __init__(self, globals={}, function_dict={}, gd=[]):
        self.m_globals = globals
        self.m_function_dict = function_dict
        self.m_gd = gd
    def parse_file(self, filename, only_header=0):
        self.m_filename = [filename]
        v = []
        self.m_blocks = []
        context = self.m_globals
        v = self.get_tokens(filename, only_header)
        v = self.do_functioncall(v)
        for op in ('%', '+', '/', ',', '='):
            v = self.do_simple_op(v, op)
        v = self.do_blockstart(v)
        for ex in v:
            if ex[0] == '=_EXP':
                context[ex[1][1]] = self.evaluate(ex[2])
            elif ex[0] == BLOCKSTART:
                assert context == self.m_globals
                self.m_blocks.append({})
                context = self.m_blocks[-1]
                context['blocktype'] = ex[1][1]
            elif ex[1] == '}':
                for n in self.m_gd:
                    if not context.has_key(n):
                        context[n] = self.m_globals[n]
                context = self.m_globals
            elif ex[0] == STRING:
                # the shortcut that interprets a single string
                # "this is a string" as
                # music = "this is a string"
                context['music'] = ex[1]
            elif ex[0] == '%_EXP':
                # the shortcut that interprets a single string
                # "this is a string" as
                # music = "this is a string"
                context['music'] = self.evaluate(ex)
            elif ex[0] == '+_EXP':
                # the shortcut that interprets a single string
                # "this is a string" as
                # music = "this is a string"
                context['music'] = self.evaluate(ex)
            else:
                #FIXME are there other places where ex is not as simple
                # as we think.
                if len(ex) > 3:
                    raise DataparserException(ex[2], ex[3], ex[1])
                else:
                    raise DataparserException(ex[1][2], ex[1][3], ex[1][1])
    def get_tokens(self, filename, first_block_only):
        f = open(filename, "r")
        s = f.read()
        f.close()
        v = []
        pos = 0
        lineno = 1
        while 1:
            try:
                if s[pos] in " \n\t{}=%+,/()":
                    if s[pos] in ' \t':
                        pos += 1
                        continue
                    if s[pos] == '\n':
                        pos += 1
                        lineno += 1
                        continue
                    v.append((CHAR, s[pos], filename, lineno))
                    pos += 1
                    continue
            except IndexError:
                break
            m = NEW_re.match(s, pos)
            if not m:
                break
            if m.lastindex == LI_COMMENT:
                pass
            else:
                v.append((lastindex_to_ID[m.lastindex], m.group(lastindex_to_group[m.lastindex]), filename, lineno))
            pos = m.end()
        return v
    def do_functioncall(self, list):
        x = 0
        while x < len(list):
            if list[x][0] == NAME \
                        and list[x+1][1] == '(' and list[x+3][1] == ')':
                if list[x][1] == 'include':
                    fn = os.path.join(os.path.dirname(self.m_filename[-1]),
                                      list[x+2][1])
                    list[x:x+4] = self.get_tokens(fn, 0)#+['FILE_END']#well pop header stack FIXME
                else:
                    list[x:x+4] = [(FUNCTIONCALL, list[x][1], list[x+2][1])]
            x = x + 1
        return list
    def do_simple_op(self, list, op):
        x = 1
        s = '%s_EXP' % op
        while x + 1 < len(list):
            if list[x][1] == op:
                list[x-1:x+2]=[(s, list[x-1], list[x+1])]
            else:
                x += 1
        return list
    def do_blockstart(self, list):
        x = 0
        while x < len(list):
            if list[x][0] is NAME and list[x+1][1] == '{':
                list[x:x+2]=[(BLOCKSTART, list[x])]
            else:
                x += 1
        return list
    def evaluate(self, exp):
        if exp[0] == '/_EXP':
            return int(exp[1][1]), int(exp[2][1])
        elif exp[0] == STRING:
            return exp[1]
        elif exp[0] == INTEGER:
            return int(exp[1])
        elif exp[0] == '+_EXP':
            return self.evaluate(exp[1]) + self.evaluate(exp[2])
        elif exp[0] == '%_EXP':
            if exp[1][0] is NAME:
                A = self.evaluate(exp[1])
            else:
                A = exp[1][1]
            return A % exp[2][1]
        elif exp[0] == ',_EXP':
            A = self.evaluate(exp[1])
            B = self.evaluate(exp[2])
            if type(A) == type([]):
                return A + [B]
            else:
                return [A, B]
        elif exp[0] == NAME:
            return self.m_globals[exp[1]]
        else:# exp[0] == 'FUNCTIONCALL':
            assert exp[0] == 'FUNCTIONCALL'
            return self.m_function_dict[exp[1]](exp[2])

def get_translated_string(dict, name):
    for n in i18n.langs():
        if dict.has_key("%s(%s)" % (name, n)):
            return dict["%s(%s)" % (name, n)]
    return dict[name]

def __f():
    if len(sys.argv) == 1:
        print "Give the file to parse as command line argument."
        sys.exit(-1)
    import time
    t1 = time.clock()
    for fn in sys.argv[1:]:
        if (not os.path.isfile(fn)) or (fn == 'lesson-files/Makefile'):
            continue
        p = Dataparser({'dictation': 'dictation',
                      'progression': 'progression',
                      'harmony': 'harmony',
                      'sing-chord': 'sing-chord',
                      'chord-voicing': 'chord-voicing',
                      'chord': 'chord',
                      'id-by-name': 'id-by-name',
                      'satb': 'satb',
                      'horiz': 'horiz',
                      'vertic': 'vertic',
                      'yes': 1,
                      'no': 0,
                      'accidentals': 'accidentals',
                      'tempo': (160, 4)},
                      {'_': _})
        for x in range(30):
            p.parse_file(fn)
            #p.get_tokens(fn, 0)
    print time.clock() - t1
    for i in p.get_tokens(fn, 0):
        print i


if __name__ == "__main__":
    import i18n
    __f()
    sys.exit()
    import profile, pstats
    profile.run("__f()", "profile.txt")
    s = pstats.Stats("profile.txt")
    s.strip_dirs().sort_stats('cumulative').print_stats(100)
