/******************************** -*- C -*- ****************************
 *
 *	Functions for byte code optimization & analysis
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2000, 2001, 2002, 2003 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/


#include "gstpriv.h"

/* Define this to have verbose messages from the bytecode verifier.  */
/* #define DEBUG_VERIFIER */

/* Define this to disable the peephole bytecode optimizer.  It works
   well and increases a bit performance, so there's no reason to do that
   unless you're debugging the compiler.  */
/* #define NO_OPTIMIZE */

/* The JIT compiler prefers optimized bytecodes, because they are
   more regular.  */
#ifdef ENABLE_JIT_TRANSLATION
#undef NO_OPTIMIZE
#endif

/* This specifies which bytecodes are pushes */
const char _gst_is_push_table[256] = {
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 16 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 32 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 48 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 64 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 80 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 96 */
  1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,	/* 112 */
  1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,	/* 128 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 144 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 160 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 176 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 192 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 208 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 224 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};				/* 240 */

/* This specifies which bytecodes are message sends */
const char _gst_is_send_table[256] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 16 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 32 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 48 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 64 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 80 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 96 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 112 */
  0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 128 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 144 */
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 160 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 176 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 192 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 208 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 224 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
};				/* 240 */

/* This specifies the stack balance of bytecodes  */
static const char special_message_num_args[16] = {
  1, 2, 0, 0, 1, 0, 1, 0, 0, 0, 1, 1, 0, 1, 0, 0,
};


/* This structure and the following one are used by the bytecode
   peephole optimizer.
   
   This one, in particular, defines where basic blocks start in the
   non- optimized bytecodes. byte is nothing more than an offset in
   those bytecodes; id is used to pair jump bytecodes with their
   destinations: in the initial scan, when we encounter a jump
   bytecode we fill two block_boundaries -- one has positive id and
   represents the destination of the jump, one has negative id (but
   the same absolute value) and represents the jump bytecode
   itself.  */
typedef struct block_boundary
{
  short byte;
  short id;
}
block_boundary;

/* This structure defines how to fix the jumps after the optimized
   basic blocks are put together.  Everything is done after the
   peephole pass because this allows us to handle forward jumps and
   backward jumps in the same way.

   When single blocks are optimized, the sorted block_boundaries are
   examined one at a time.  As we process blocks, we fill an array of
   jump structures with offsets in the optimized bytecode.  We fill a
   single field at a time -- the id's sign in the block_boundary says
   which field is to be filled, the absolute value gives which jump
   structure is to be filled.  In the end, block_boundaries whose id's
   absolute value is the same are all paired.  */
typedef struct jump
{
  int from;			/* where the jump bytecode lies */
  int dest;			/* where the jump bytecode lands */
}
jump;



/* Scan the bytecodes between FROM and TO. As they are
   scanned, they are overwritten with an optimized version; in
   the end, _gst_compile_bytecodes() is used to append them to the
   stream of optimized bytecodes.  The return value indicates
   whether at least a bytecode was generated.  */
static mst_Boolean optimize_basic_block (gst_uchar * from,
					 gst_uchar * to);


/* This compares two block_boundary structures according to their
   bytecode position.  */
static int compare_blocks (const PTR a, const PTR b) ATTRIBUTE_PURE;

/* This answers how the dirtyness of BLOCKOOP affects
   the block that encloses it.  */
static inline int check_inner_block (OOP blockOOP);

/* This fills a table that says to which bytecodes a jump lands.
   Starting from BP, and for a total of SIZE bytes, bytecodes are
   analyzed and on output DEST[i] is non-zero if and
   only if BP[i] is the destination of a jump. It is positive
   for a forward jump and negative for a backward jump.  The number
   of jumps is returned.  */
static int make_destination_table (gst_uchar * bp,
				   int size,
				   char *dest);

/* Helper function to compute the bytecode verifier's `in'
   sets from the `out' sets.  */
static mst_Boolean merge_stacks (OOP *dest,
				 int dest_sp,
				 OOP *src,
				 int src_sp);


int
_gst_is_simple_return (bc_vector bytecodes)
{
  gst_uchar *bytes;
  size_t byteCodeLen;

  if (bytecodes == NULL)
    return (MTH_NORMAL);

  byteCodeLen = _gst_bytecode_length (bytecodes);
  bytes = bytecodes->base;

  if (bytes[0] == LINE_NUMBER_BYTECODE)
    {
      byteCodeLen -= 3;
      bytes += 3;
    }

  if (byteCodeLen == 2)
    {
      if (bytes[1] != RETURN_CONTEXT_STACK_TOP)
	return (MTH_NORMAL);

      /* check for ^INSTANCE_VARIABLE */
      if ((bytes[0] & ~15) == PUSH_RECEIVER_VARIABLE)
	return (((bytes[0] & 0x0F) << 8) | MTH_RETURN_INSTVAR);

      /* check for ^LITERAL */
      else if (bytes[0] == PUSH_LIT_CONSTANT)
	return (MTH_RETURN_LITERAL);

      else if (bytes[0] == PUSH_ZERO)
	{
	  _gst_add_forced_object (FROM_INT (0));
	  return (MTH_RETURN_LITERAL);
	}
      else if (bytes[0] == PUSH_ONE)
	{
	  _gst_add_forced_object (FROM_INT (1));
	  return (MTH_RETURN_LITERAL);
	}
      else
	return (MTH_NORMAL);

    }

  else if (byteCodeLen == 3)
    {
      if (bytes[2] != RETURN_CONTEXT_STACK_TOP)
	return (MTH_NORMAL);

      /* check for ^INSTANCE_VARIABLE */
      if (bytes[0] == PUSH_INDEXED
	  && (bytes[1] & LOCATION_MASK) == RECEIVER_LOCATION)
	return (((bytes[1] & ~LOCATION_MASK) << 8) | MTH_RETURN_INSTVAR);

      /* check for ^LITERAL */
      else if (bytes[0] == PUSH_SIGNED_8)
	{
	  _gst_add_forced_object (FROM_INT ((signed char) bytes[1]));
	  return (MTH_RETURN_LITERAL);
	}

      else if (bytes[0] == PUSH_UNSIGNED_8)
	{
	  _gst_add_forced_object (FROM_INT ((unsigned char) bytes[1]));
	  return (MTH_RETURN_LITERAL);
	}
      else
	return (MTH_NORMAL);
    }

  /* check for ^LITERAL */
  if (byteCodeLen == 1)
    {
      /* check for ^self */
      if (bytes[0] == (RETURN_INDEXED | RECEIVER_INDEX))
        return (MTH_RETURN_SELF);

      if (bytes[0] == (RETURN_INDEXED | TRUE_INDEX))
	{
	  _gst_add_forced_object (_gst_true_oop);
	  return (MTH_RETURN_LITERAL);
	}
      else if (bytes[0] == (RETURN_INDEXED | FALSE_INDEX))
	{
	  _gst_add_forced_object (_gst_false_oop);
	  return (MTH_RETURN_LITERAL);
	}
      else if (bytes[0] == (RETURN_INDEXED | NIL_INDEX))
	{
	  _gst_add_forced_object (_gst_nil_oop);
	  return (MTH_RETURN_LITERAL);
	}
      else
	return (MTH_NORMAL);
    }

  return (MTH_NORMAL);
}

int
_gst_check_kind_of_block (bc_vector bc,
			  OOP * literals)
{
  int status, newStatus;
  gst_uchar *bp, *end;
  OOP blockClosureOOP;

  status = 0;			/* clean block */
  for (bp = bc->base, end = bc->ptr; bp != end;
       bp += BYTECODE_SIZE (*bp))
    {
      MATCH_BYTECODES (CHECK_KIND_OF_BLOCK, bp, (
        PUSH_SELF, REPLACE_SELF, RETURN_SELF,
	PUSH_RECEIVER_VARIABLE, STORE_RECEIVER_VARIABLE,
	REPLACE_RECEIVER_VARIABLE, POP_RECEIVER_VARIABLE {
	  if (status == 0)
	    status = 1;
	}

        PUSH_LIT_CONSTANT {
	  newStatus = check_inner_block (literals[n]);
	  if (newStatus > status)
	    {
	      if (newStatus == 31)
		return (31);
	      status = newStatus;
	    }
        }

        PUSH_OUTER_TEMP, POP_OUTER_TEMP, STORE_OUTER_TEMP  {
	  if (status < 1 + scopes) status = 1 + scopes;
	  if (status > 31)
	    /* ouch! how deep!! */
	    return (31);
	}

        JUMP,
        POP_JUMP_TRUE,
        POP_JUMP_FALSE,
        PUSH_TEMPORARY_VARIABLE,
        PUSH_LIT_VARIABLE,
        POP_TEMPORARY_VARIABLE,
        POP_LIT_VARIABLE,
        PUSH_SPECIAL,
        PUSH_SIGNED_8,
        PUSH_UNSIGNED_8,
        RETURN_SPECIAL,
        RETURN_CONTEXT_STACK_TOP,
        LINE_NUMBER_BYTECODE,
        STORE_TEMPORARY_VARIABLE,
        STORE_LIT_VARIABLE,
        SEND,
        POP_INTO_NEW_STACKTOP,
        POP_STACK_TOP,
        DUP_STACK_TOP,
        NOP_BYTECODE,
        REPLACE_ONE,
        REPLACE_TEMPORARY_VARIABLE,
        REPLACE_LIT_CONSTANT,
        REPLACE_LIT_VARIABLE,
        EXIT_INTERPRETER,
        SEND_ARITH,
        SEND_SPECIAL,
        MAKE_DIRTY_BLOCK { }

        RETURN_METHOD_STACK_TOP { return (31); }
        INVALID { abort(); }
      ));
    }
  return (status);
}

int
check_inner_block (OOP blockOOP)
{
  int newStatus;
  gst_compiled_block block;

  if (!IS_CLASS (blockOOP, _gst_compiled_block_class))
    return (0);

  /* Check the cleanness of the inner block and adequately change the status. 
     full block: no way dude -- exit immediately
     clean block: same for us 
     receiver access: same for us
     access to temps in the Xth context: from the perspective of the block
     being checked here, it is like an access to temps in the (X-1)th
     context access to this block's temps: our outerContext can be nil
     either, but for safety we won't be a clean block.  */
  block = (gst_compiled_block) OOP_TO_OBJ (blockOOP);
  newStatus = block->header.clean;
  switch (newStatus)
    {
    case 31:
    case 0:
    case 1:
      return (newStatus);
    default:
      return (newStatus - 1);
    }
}


int
compare_blocks (const PTR a, const PTR b)
{
  const block_boundary *ba = (const block_boundary *) a;
  const block_boundary *bb = (const block_boundary *) b;

  return (ba->byte - bb->byte);
}

bc_vector
_gst_optimize_bytecodes (bc_vector bytecodes)
{
  bc_vector old_bytecodes;
  block_boundary *blocks, *current;
  jump *jumps;
  gst_uchar *bp;
  gst_uchar *end, *first;
  int num;

#ifdef NO_OPTIMIZE
  return (bytecodes);
#endif

  bp = bytecodes->base;
  end = bytecodes->ptr;
  blocks = alloca (sizeof (block_boundary) * (end - bp + 1));
  memset (blocks, 0, sizeof (block_boundary) * (end - bp + 1));

  /* 1) Split into basic blocks.  This part cheats so that the final
     fixup also performs jump optimization.  */
  for (current = blocks, num = 0; bp != end; bp += BYTECODE_SIZE (*bp))
    {
      gst_uchar *dest = bp;
      mst_Boolean canOptimizeJump;
      do
	{
	  canOptimizeJump = false;
	  MATCH_BYTECODES (THREAD_JUMPS, dest, (
	    MAKE_DIRTY_BLOCK,
	    SEND_SPECIAL,
	    SEND_ARITH,
	    PUSH_RECEIVER_VARIABLE,
	    PUSH_TEMPORARY_VARIABLE,
	    PUSH_LIT_CONSTANT,
	    PUSH_LIT_VARIABLE,
	    POP_RECEIVER_VARIABLE,
	    POP_TEMPORARY_VARIABLE,
	    POP_LIT_VARIABLE,
	    PUSH_SELF,
	    PUSH_SPECIAL,
	    PUSH_SIGNED_8,
	    PUSH_UNSIGNED_8,
	    LINE_NUMBER_BYTECODE,
	    STORE_RECEIVER_VARIABLE,
	    STORE_TEMPORARY_VARIABLE,
	    STORE_LIT_VARIABLE,
	    SEND,
	    POP_INTO_NEW_STACKTOP,
	    POP_STACK_TOP,
	    DUP_STACK_TOP,
	    PUSH_OUTER_TEMP,
	    POP_OUTER_TEMP,
	    STORE_OUTER_TEMP,
	    NOP_BYTECODE,
	    REPLACE_SELF,
	    REPLACE_ONE,
	    REPLACE_RECEIVER_VARIABLE,
	    REPLACE_TEMPORARY_VARIABLE,
	    REPLACE_LIT_CONSTANT,
	    REPLACE_LIT_VARIABLE,
	    EXIT_INTERPRETER { }

	    JUMP {
	      if (ofs == 2 && dest[2] == POP_STACK_TOP)
		{
		  /* The bytecodes can only be those produced by
		     #ifTrue:/#ifFalse: 
			     0: jump to 2 
			     1: push nil 
			     2: pop stack top 

		     This could not be optimized to a single
		     pop, cause bytecodes 1 and 2 lie in different
		     basic blocks! So we rewrite it to a functionally
		     equivalent but optimizable bytecode sequence.  */
		  *dest = POP_STACK_TOP;
		}
	      else
		{
	          /* If bp == dest, we could end up writing a 2-byte jump
	             bytecode where space was only reserved for a 1-byte
	             jump bytecode! But if we jump to a return, we can
	             safely optimize -- returns are always one byte */
	          canOptimizeJump = (bp != dest) || (*bp >= JUMP_LONG);
	          dest += ofs;
	          canOptimizeJump |= IS_RETURN_BYTECODE (*dest);
		}
	    }

	    POP_JUMP_TRUE, POP_JUMP_FALSE {
	      /* UNCONDITIONAL jumps to CONDITIONAL jumps must not be
	         touched! */
	      if (bp == dest)
		dest += ofs;
	    }

	    RETURN_SELF, RETURN_SPECIAL, RETURN_METHOD_STACK_TOP,
	    RETURN_CONTEXT_STACK_TOP {
	      /* Return bytecodes - patch the original jump to return
	         directly */
	      if (*bp >= JUMP_LONG)
		{
		  bp[0] = NOP_BYTECODE;
		  bp[1] = *dest;	/* fill both bytes */
		}
	      else
		*bp = *dest;

	      /* This in fact eliminated the jump, don't split in basic 
	         blocks */
	      dest = bp;
	    }

	    INVALID { abort (); }
	  ));
	}
      while (canOptimizeJump);
      if (bp != dest)
	{
	  current->byte = dest - bytecodes->base;
	  current->id = ++num;
	  current++;
	  current->byte = bp - bytecodes->base;
	  current->id = -num;
	  current++;
	}
    }

  /* 2) Get the "real" block boundaries by sorting them according to
     where they happen in the original bytecode.  Note that a simple
     bucket sort is not enough because multiple jumps could end on the
     same bytecode, and the same bytecode could be both the start and
     the destination of a jump! */
  qsort (blocks, current - blocks, sizeof (block_boundary), compare_blocks);

  /* 3) Optimize the single basic blocks, and reorganize into `jumps'
     the data that was put in blocks */
  jumps = alloca (sizeof (jump) * num);

  old_bytecodes = _gst_save_bytecode_array ();

  for (bp = bytecodes->base; blocks != current; blocks++)
    {
      first = bp;
      bp = bytecodes->base + blocks->byte;
      optimize_basic_block (first, bp);
      if (blocks->id > 0)
	jumps[blocks->id - 1].dest = _gst_current_bytecode_length ();

      else
	jumps[-blocks->id - 1].from = _gst_current_bytecode_length ();
    }
  optimize_basic_block (bp, end);

  _gst_free_bytecodes (bytecodes);
  bytecodes = _gst_get_bytecodes ();

  /* 4) Fix the jumps so that they correctly point to the start of the
     same basic block */
  for (; num--; jumps++)
    {
      short offset;

      bp = bytecodes->base + jumps->from;
      offset = jumps->dest - jumps->from - 2;
      if (offset == -1)
	{			/* jump to following bytecode do */
	  if (*bp >= JUMP_LONG)	/* NOT exist - use other bytecodes */
	    bp[1] = NOP_BYTECODE;	/* 2 byte jumps = nop+nop or
					   pop+nop */

	  if (*bp & 8)
	    *bp = POP_STACK_TOP;	/* pop stack top for
					   conditionals */
	  else
	    *bp = NOP_BYTECODE;	/* nop for unconditional jumps */

	  continue;
	}
      switch (*bp & ~7)
	{
	  /* short jumps */
	case JUMP_SHORT:
	  *bp = JUMP_SHORT | offset;
	  continue;
	case POP_JUMP_FALSE_SHORT:
	  *bp = POP_JUMP_FALSE_SHORT | offset;
	  continue;

	  /* long jumps */
	case JUMP_LONG:
	  *bp = JUMP_LONG | 4;
	  break;
	case POP_JUMP_TRUE:
	  *bp &= ~3;
	  break;
	}
      *bp++ += offset >> 8;
      *bp = offset & 255;
    }

  _gst_restore_bytecode_array (old_bytecodes);

  return (bytecodes);
}

mst_Boolean
optimize_basic_block (gst_uchar * from,
		      gst_uchar * to)
{
#define NEXT(size) BEGIN_MACRO {	\
  n = size;				\
  opt += size;				\
  continue;				\
} END_MACRO

#define REPLACE(size) BEGIN_MACRO {	\
  n = size;				\
  continue;				\
} END_MACRO

#define COPY BEGIN_MACRO {		\
  n = BYTECODE_SIZE(byte);		\
  opt++;				\
  if(n != 1) *opt++ = *bp++;		\
  if(n == 3) *opt++ = *bp++;		\
  continue;				\
} END_MACRO

#define BEGIN	     if(0) {
#define BYTECODE(n)  } else if(byte == (n)) {
#define RANGE(a, b)  } else if((unsigned char)(byte - (a)) <= ((b) - (a))) {
#define EITHER(a, b) } else if(byte == (a) b) {
#define OR(b)  		       || byte == (b)
#define CONDITION(c) } else if(c) {
#define NO_MATCH     } else {
#define END	     }


  gst_uchar byte, *bp, *opt;
  int n;

  bp = opt = from;
  n = 0;
  while (bp != to)
    {
      byte = *opt = *bp++;
      BEGIN
	RANGE (RETURN_INDEXED, RETURN_CONTEXT_STACK_TOP) 
	  opt++;
          break;		/* this `break' performs unreachable
				   code elimination! */

	BYTECODE (NOP_BYTECODE) 
	  REPLACE (n);

        BYTECODE (NOT_EQUAL_SPECIAL) 
	  if (!n)
	    NEXT (1);
	  if (opt[-n] == (PUSH_SPECIAL | NIL_INDEX))
	    {			/* x ~= nil */
	      opt[-n] = NOT_NIL_SPECIAL;
	      REPLACE (1);
	    }
	  NEXT (1);

	EITHER (SAME_OBJECT_SPECIAL, OR (EQUAL_SPECIAL)) 
	  if (!n)
	    NEXT (1);
	  if (opt[-n] == (PUSH_SPECIAL | NIL_INDEX))
	    {			/* x = nil, x == nil */
	      opt[-n] = IS_NIL_SPECIAL;
	      REPLACE (1);
	    }
	  NEXT (1);

        BYTECODE (POP_STACK_TOP) 
	  if (n)
	    {
	      byte = opt[-n];
	      BEGIN 
		CONDITION (IS_PUSH_BYTECODE (byte))	/* push/pop */
		  opt -= n;
	          NEXT (0);

		BYTECODE (STORE_INDEXED)	/* store/pop */
		  byte = *--opt;	/* get data byte */
		  opt--;		/* move to opcode */
		  if (byte < 8)
		    {
		      *opt = POP_RECEIVER_VARIABLE | byte;
		      NEXT (1);
		    }
		  else if (byte >= 64 && byte < 72)
		    {
		      *opt = POP_TEMPORARY_VARIABLE | (byte & 63);
		      NEXT (1);
		    }
		  else
		    {
		      *opt = POP_STORE_INDEXED;
		      NEXT (2);
		    }
		  
		EITHER (BIG_LITERALS_BYTECODE, 
			OR (BIG_INSTANCE_BYTECODE) 
			OR (OUTER_TEMP_BYTECODE))

		  byte = opt[-2];	/* get second byte */
		  if (byte < POP_STORE_VARIABLE)
		    {
		      opt -= n;		/* push/pop */
		      NEXT (0);
		    }
		  else if (byte >= STORE_VARIABLE)
		    {
		      opt[-2] ^= (POP_STORE_VARIABLE ^ STORE_VARIABLE);
		      REPLACE (3);
		    }
	      END;
	    }

	  if (bp != to && (*bp & ~3) == RETURN_INDEXED)
	    {
	      *opt++ = *bp++;	/* pop/return */
	      break;		/* kill unreachable code */
	    }
	  NEXT (1);

	CONDITION (IS_PUSH_BYTECODE (byte))	/* push/push -> dup */
	  if (!n)
	    COPY;
	  if (opt[-n] == *opt)
	    {
	      if (n == 1)
		{
		  *opt = DUP_STACK_TOP;
		  NEXT (1);
		}
	      else if (opt[-1] == *bp)
		{
		  *opt = DUP_STACK_TOP;
		  bp++;
		  NEXT (1);
		}
	    }

	  BEGIN			/* pop-store/push -> store */
	    RANGE (PUSH_RECEIVER_VARIABLE, PUSH_RECEIVER_VARIABLE | 7)
	      if (opt[-n] == (POP_RECEIVER_VARIABLE | (byte & 15)))
		{
		  opt[-1] = STORE_INDEXED;
		  *opt++ = RECEIVER_LOCATION | (byte & 15);
		  REPLACE (2);
		}
	      if (opt[-n] == POP_STACK_TOP)
	        {			/* pop/push -> replace */
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = RECEIVER_LOCATION | (byte & 15);
		  REPLACE (2);
		}

	    RANGE (PUSH_RECEIVER_VARIABLE | 8, PUSH_RECEIVER_VARIABLE | 15)
	      if (opt[-n] == POP_STORE_INDEXED
		  && opt[-1] == (RECEIVER_LOCATION | (byte & 15)))
		{
		  opt[-2] = STORE_INDEXED;
		  REPLACE (2);
		}
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = RECEIVER_LOCATION | (byte & 15);
		  REPLACE (2);
		}

	    RANGE (PUSH_TEMPORARY_VARIABLE, PUSH_TEMPORARY_VARIABLE | 7)
	      if (opt[-n] == (POP_TEMPORARY_VARIABLE | (byte & 15)))
		{
		  opt[-1] = STORE_INDEXED;
		  *opt++ = TEMPORARY_LOCATION | (byte & 15);
		  REPLACE (2);
		}
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = TEMPORARY_LOCATION | (byte & 15);
		  REPLACE (2);
		}

	    RANGE (PUSH_TEMPORARY_VARIABLE | 8, PUSH_TEMPORARY_VARIABLE | 15)
	      if (opt[-n] == POP_STORE_INDEXED
		  && opt[-1] == (TEMPORARY_LOCATION | (byte & 15)))
		{
		  opt[-2] = STORE_INDEXED;
		  REPLACE (2);
		}
	      if (opt[-n] == POP_STACK_TOP)
	        {
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = TEMPORARY_LOCATION | (byte & 15);
		  REPLACE (2);
		}

	    RANGE (PUSH_LIT_VARIABLE, PUSH_LIT_VARIABLE | 31)
	      if (opt[-n] == POP_STORE_INDEXED
		  && opt[-1] == (LIT_VAR_LOCATION | (byte & 31)))
		{
		  opt[-2] = STORE_INDEXED;
		  REPLACE (2);
		}
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = LIT_VAR_LOCATION | (byte & 31);
		  REPLACE (2);
		}

	    RANGE (PUSH_LIT_CONSTANT, PUSH_LIT_CONSTANT | 31)
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt--;
		  *opt++ = REPLACE_INDEXED;
		  *opt++ = LIT_CONST_LOCATION | (byte & 31);
		  REPLACE (2);
		}

	    BYTECODE (PUSH_SPECIAL | RECEIVER_INDEX)
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt[-1] = REPLACE_SELF;
		  REPLACE (1);
		}

	    BYTECODE (PUSH_ONE)
	      if (opt[-n] == POP_STACK_TOP)
		{
		  opt[-1] = REPLACE_ONE;
		  REPLACE (1);
		}

	    BYTECODE (PUSH_INDEXED)
	      byte = *bp++;
	      if (opt[-n] == POP_STORE_INDEXED)
		{
		  opt[-2] = STORE_INDEXED;
		  if (opt[-1] == byte)
		    REPLACE (2);
		  else
		    *opt = REPLACE_INDEXED;
		}
	      else if (opt[-n] == POP_STACK_TOP)
		*--opt = REPLACE_INDEXED;

	      opt[1] = byte;
	      NEXT (2);

	  END;
	  COPY;

        BYTECODE (BIG_INSTANCE_BYTECODE) 
	  if (!n || opt[-n] != byte)
	    COPY;

          byte = opt[-2];		/* get second byte */
	  if (byte < PUSH_VARIABLE)
	    ;	  /* do nothing */
	  else if (byte < POP_STORE_VARIABLE)
	    {
	      if (byte == *bp && opt[-1] == bp[1])
		{			/* push/push -> dup */
		  *opt = DUP_STACK_TOP;
		  bp += 2;
		  NEXT (1);
		}
	    }
	  else if (byte < STORE_VARIABLE)
	    {			/* pop-store/push -> store */
	      if ((byte & 63) == (*bp & 63) && opt[-1] == bp[1])
		{
		  opt[-2] ^= (POP_STORE_VARIABLE ^ STORE_VARIABLE);
		  bp += 2;
		  REPLACE (3);
		}
	    }
	  
	  opt++;
	  *opt++ = *bp++;
	  *opt++ = *bp++;
	  REPLACE (3);

        EITHER (BIG_LITERALS_BYTECODE,
	        OR (OUTER_TEMP_BYTECODE)) 
	  if (!n || opt[-n] != byte)
	    COPY;

          byte = opt[-2];		/* get second byte */
	  if (byte < POP_STORE_VARIABLE)
	    {
	      if (byte == *bp && opt[-1] == bp[1])
		{			/* push/push -> dup */
		  *opt = DUP_STACK_TOP;
		  bp += 2;
		  NEXT (1);
		}
	    }
	  else if (byte < STORE_VARIABLE)
	    {			/* pop-store/push -> store */
	      if ((byte & 63) == (*bp & 63) && opt[-1] == bp[1])
		{
		  opt[-2] ^= (POP_STORE_VARIABLE ^ STORE_VARIABLE);
		  bp += 2;
		  REPLACE (3);
		}
	    }
	  
	  opt++;
	  *opt++ = *bp++;
	  *opt++ = *bp++;
	  REPLACE (3);

	NO_MATCH
	  COPY;
      END;
    }

  _gst_compile_bytecodes (from, opt);
  return (opt != from);
}

void
_gst_compute_stack_positions (gst_uchar * bp,
			      int size,
			      PTR * base,
			      PTR ** pos)
{
  gst_uchar *end;
  int balance, ofs, bc_size;

  memzero (pos, sizeof (PTR *) * (1 + size));

  pos[0] = base;
  for (end = bp + size; bp != end;
       pos[bc_size] = (pos[bc_size] ? pos[bc_size] : pos[0] + balance),
       pos += bc_size, bp += bc_size)
    {
      bc_size = BYTECODE_SIZE (*bp);
      MATCH_BYTECODES (COMPUTE_BALANCE, bp, (
        RETURN_METHOD_STACK_TOP,
        RETURN_CONTEXT_STACK_TOP,
        RETURN_SELF,
	RETURN_SPECIAL { continue; }

        POP_RECEIVER_VARIABLE,
        POP_TEMPORARY_VARIABLE,
        POP_LIT_VARIABLE,
        POP_INTO_NEW_STACKTOP,
        POP_STACK_TOP,
        POP_OUTER_TEMP,
        SEND_ARITH { balance = -1; }

        LINE_NUMBER_BYTECODE,
        STORE_RECEIVER_VARIABLE,
        STORE_TEMPORARY_VARIABLE,
        STORE_LIT_VARIABLE,
        STORE_OUTER_TEMP,
        NOP_BYTECODE,
        REPLACE_SELF,
        REPLACE_ONE,
        REPLACE_RECEIVER_VARIABLE,
        REPLACE_TEMPORARY_VARIABLE,
        REPLACE_LIT_CONSTANT,
        REPLACE_LIT_VARIABLE,
        EXIT_INTERPRETER,
        MAKE_DIRTY_BLOCK { balance = 0; }

        PUSH_RECEIVER_VARIABLE,
        PUSH_TEMPORARY_VARIABLE,
        PUSH_LIT_CONSTANT,
        PUSH_LIT_VARIABLE,
        PUSH_SELF,
        PUSH_SPECIAL,
        PUSH_SIGNED_8,
        PUSH_UNSIGNED_8,
        DUP_STACK_TOP,
        PUSH_OUTER_TEMP { balance = 1; }

        SEND { balance = -num_args; }
        SEND_SPECIAL { balance = -special_message_num_args[n]; }
        INVALID { abort(); }

        JUMP {
          balance = 0;
          if (ofs > 0)
            pos[ofs] = pos[0] + balance;
          else
	    assert (pos[ofs] == pos[0] + balance);

	  continue;
        }
        POP_JUMP_TRUE, POP_JUMP_FALSE {
          balance = -1;
          if (ofs > 0)
            pos[ofs] = pos[0] + balance;
          else
	    assert (pos[ofs] == pos[0] + balance);
        }
      ));

      assert (!pos[bc_size] || pos[bc_size] == pos[0] + balance);
    }
}


void
_gst_analyze_bytecodes (OOP methodOOP,
			 int size,
			 char *dest)
{
  gst_uchar *bp;

  bp = GET_METHOD_BYTECODES (methodOOP);
  make_destination_table (bp, size, dest);

  /* Nothing more for now */
}

int
make_destination_table (gst_uchar * bp,
			int size,
			char *dest)
{
  gst_uchar *end;
  int count;

  memzero (dest, sizeof (char) * size);

  for (count = 0, end = bp + size; bp != end;
       dest += BYTECODE_SIZE (*bp), bp += BYTECODE_SIZE (*bp))
    {
      MATCH_BYTECODES (MAKE_DEST_TABLE, bp, (
        PUSH_RECEIVER_VARIABLE,
        PUSH_TEMPORARY_VARIABLE,
        PUSH_LIT_CONSTANT,
        PUSH_LIT_VARIABLE,
        POP_RECEIVER_VARIABLE,
        POP_TEMPORARY_VARIABLE,
        POP_LIT_VARIABLE,
        PUSH_SELF,
        PUSH_SPECIAL,
        PUSH_SIGNED_8,
        PUSH_UNSIGNED_8,
        RETURN_SELF,
	RETURN_SPECIAL,
        RETURN_METHOD_STACK_TOP,
        RETURN_CONTEXT_STACK_TOP,
        LINE_NUMBER_BYTECODE,
        STORE_RECEIVER_VARIABLE,
        STORE_TEMPORARY_VARIABLE,
        STORE_LIT_VARIABLE,
        SEND,
        POP_INTO_NEW_STACKTOP,
        POP_STACK_TOP,
        DUP_STACK_TOP,
        PUSH_OUTER_TEMP,
        POP_OUTER_TEMP,
        STORE_OUTER_TEMP,
        NOP_BYTECODE,
        REPLACE_SELF,
        REPLACE_ONE,
        REPLACE_RECEIVER_VARIABLE,
        REPLACE_TEMPORARY_VARIABLE,
        REPLACE_LIT_CONSTANT,
        REPLACE_LIT_VARIABLE,
        EXIT_INTERPRETER,
        SEND_ARITH,
        SEND_SPECIAL,
        MAKE_DIRTY_BLOCK { }
        INVALID { abort(); }

        JUMP, POP_JUMP_TRUE, POP_JUMP_FALSE {
          dest[ofs] = ofs > 0 ? 1 : -1;
	  count++;
        }
      ));
    }

  return (count);
}



#define SELF 0
#define VARYING 1
#define UNDEFINED 2

typedef struct partially_constructed_array {
  struct partially_constructed_array *next;
  int sp;
  int size;
} partially_constructed_array;

typedef struct basic_block_item {
  struct basic_block_item *next;
  struct basic_block_item **bb;
  gst_uchar *bp;
  int sp;
  OOP stack[1];
} basic_block_item;

#define ALLOCA_BASIC_BLOCK() alloca (sizeof (basic_block_item) + \
				     sizeof (OOP) * (stack_depth - 1))

#define CHECK_LITERAL(n) \
  /* Todo: recurse into BlockClosures! */ \
  last_used_literal = literals[n]; \
  if ((n) >= num_literals) \
    return ("literal out of range");

#define CHECK_TEMP(n) \
  last_used_literal = NULL; \
  if ((n) >= sp - stack) \
    return ("temporary out of range");

#define CHECK_REC_VAR(first, n) \
  last_used_literal = NULL; \
  if ((n) < (first) || (n) >= num_rec_vars) \
    return ("receiver variable out of range");

#define CHECK_LIT_VARIABLE(store, n) \
  CHECK_LITERAL (n); \
  if (IS_INT (literals[(n)]) || \
      !is_a_kind_of (OOP_CLASS (literals[(n)]), _gst_association_class)) \
    return ("Association expected"); \
  else if (store \
	   && untrusted \
	   && !IS_OOP_UNTRUSTED (literals[(n)])) \
    return ("Invalid global variable access");

#define LIT_VARIABLE_CLASS(n) \
  /* Special case Array because it is used to compile {...} */ \
  (ASSOCIATION_VALUE (literals[(n)]) == _gst_array_class \
    ? OOP_CLASS (_gst_array_class) \
    : FROM_INT (VARYING))

#define LITERAL_CLASS(n) \
  OOP_INT_CLASS (literals[(n)])

/* Bytecode verification is a dataflow analysis on types.  We perform it
   on basic blocks: `in' is the stack when entering the basic block and
   `out' is the stack upon exit.

   Each member of the stack can be UNDEFINED, a type, or VARYING.  When
   merging two `out' stacks to compute an `in' stack, we have these
   possible situations:
   - the stacks are not the same height, and bytecode verification fails
   - a slot is the same in both stacks, so it has this type in the output too
   - a slot is different in the two stacks, so it is VARYING in the output.

   Bytecode verification proceeds forwards, so the worklist is added all the
   successors of the basic block whenever merging results in a difference.  */

mst_Boolean
merge_stacks (OOP *dest, int dest_sp,
	      OOP *src, int src_sp)
{
  mst_Boolean varied = false;
  assert (dest_sp == src_sp);

  for (; src_sp--; dest++, src++)
    {
      OOP newDest = *src;
      if (newDest != *src)
	{
	  if (*dest != FROM_INT (UNDEFINED))
	    /* If different, mark as overdefined.  */
	    newDest = FROM_INT (VARYING);

	  if (newDest != *dest)
	    {
	      *dest = newDest;
	      varied = true;
	    }
	}
    }

  return (varied);
}

void
_gst_verify_sent_method (OOP methodOOP)
{
  const char *error;
  error = _gst_verify_method (methodOOP, NULL, 0);

  if (error)
    {
      _gst_errorf ("Bytecode verification failed: %s", error);
      if (OOP_CLASS (methodOOP) == _gst_compiled_block_class)
        methodOOP = GET_BLOCK_METHOD (methodOOP);

      _gst_errorf ("Method verification failed for %O>>%O",
                   GET_METHOD_CLASS (methodOOP),
                   GET_METHOD_SELECTOR (methodOOP));

      abort ();
    }
}

const char *
_gst_verify_method (OOP methodOOP, int *num_outer_temps, int depth)
{
  int size, num_temps, stack_depth,
    num_literals, num_rec_vars, num_ro_rec_vars;

  mst_Boolean untrusted;
  const char *error;
  gst_uchar *bp;
  OOP *literals, methodClass, last_used_literal;
  basic_block_item **bb_start, *bb_first, *worklist;
  partially_constructed_array *arrays = NULL, *arrays_pool = NULL;

  if (IS_OOP_VERIFIED (methodOOP))
    return (NULL);

  size = NUM_INDEXABLE_FIELDS (methodOOP);
  bp = GET_METHOD_BYTECODES (methodOOP);
  literals = GET_METHOD_LITERALS (methodOOP);
  methodClass = GET_METHOD_CLASS (methodOOP);
  num_literals = NUM_METHOD_LITERALS (methodOOP);
  num_rec_vars = CLASS_FIXED_FIELDS (methodClass);
  untrusted = IS_OOP_UNTRUSTED (methodOOP);

  if (is_a_kind_of (OOP_CLASS (methodOOP), _gst_compiled_method_class))
    {
      method_header header;
      header = GET_METHOD_HEADER (methodOOP);
      num_temps = header.numArgs + header.numTemps;
      stack_depth = header.stack_depth << DEPTH_SCALE;
      switch (header.headerFlag)
        {
	case MTH_NORMAL:
	case MTH_PRIMITIVE:
	case MTH_ANNOTATED:
	case MTH_UNUSED:
	  break;

	case MTH_USER_DEFINED:
	case MTH_RETURN_SELF:
	  methodOOP->flags |= F_VERIFIED;
	  return (NULL);

	case MTH_RETURN_INSTVAR:
	  CHECK_REC_VAR (0, header.primitiveIndex);
	  methodOOP->flags |= F_VERIFIED;
	  return (NULL);

	case MTH_RETURN_LITERAL:
	  CHECK_LITERAL (0);
	  methodOOP->flags |= F_VERIFIED;
	  return (NULL);
        }
    }
  else if (OOP_CLASS (methodOOP) == _gst_compiled_block_class)
    {
      block_header header;
      header = GET_BLOCK_HEADER (methodOOP);

      /* If we're verifying a block but not from a nested call,
	 restart from the top-level method.  */
      if (header.clean != 0 && depth == 0)
	return _gst_verify_method (GET_BLOCK_METHOD (methodOOP), NULL, 0);

      num_temps = header.numArgs + header.numTemps;
      stack_depth = header.depth << DEPTH_SCALE;
    }
  else
    return "invalid class";

  if (untrusted)
    {
       OOP class_oop, arrayOOP;
       for (class_oop = methodClass; IS_OOP_UNTRUSTED (class_oop);
            class_oop = SUPERCLASS (class_oop))
         ;

       num_ro_rec_vars = CLASS_FIXED_FIELDS (class_oop);
    }
  else
    num_ro_rec_vars = 0;

  /* Prepare the NUM_OUTER_TEMPS array for the inner blocks.  */
  if (depth)
    {
      int *new_num_outer_temps = alloca (sizeof (int) * (depth + 1));
      memcpy (new_num_outer_temps + 1, num_outer_temps, sizeof (int) * depth);
      num_outer_temps = new_num_outer_temps;
    }
  else
    num_outer_temps = &num_temps;

  depth++;

  bb_start = alloca ((1 + size) * sizeof (basic_block_item *));
  memzero (bb_start, (1 + size) * sizeof (basic_block_item *));

#ifdef DEBUG_VERIFIER
  printf ("Verifying %O (max. stack depth = %d):\n", methodOOP, stack_depth);
#endif

  /* Allocate the first and last basic block specially */
  bb_first = ALLOCA_BASIC_BLOCK ();
  bb_first->next = NULL;
  bb_first->bp = bp;
  bb_first->bb = bb_start;
  bb_first->sp = num_temps;
  bb_start[0] = bb_first;

  bb_start[size] = ALLOCA_BASIC_BLOCK ();
  bb_start[size]->next = NULL;
  bb_start[size]->bp = bp + size;
  bb_start[size]->bb = bb_start + size;
  bb_start[size]->sp = num_temps;

  /* First build the pointers to the basic blocks into BB_START.  The use
     of a worklist here is only to find a correct order for visiting the
     basic blocks, not because they're visited multiple times.  This
     works transparently when we have a return in the middle of the method.
     Then the basic block is ending, yet it might be that the stack height
     for the next bytecode is already known!!! */
  for (worklist = bb_first; worklist; )
    {
      int curr_sp = worklist->sp;
      bp = worklist->bp;
      bb_start = worklist->bb;
      worklist = worklist->next;

#ifdef DEBUG_VERIFIER
      printf ("Tracing basic block at %d:\n", bb_start - bb_first->bb);
#endif

      do
        {
	  int deepest_access = 0, balance;
          int bc_len = BYTECODE_SIZE (*bp);

#ifdef DEBUG_VERIFIER
          printf ("\t[SP=%3d]  %5d: ", curr_sp, bb_start - bb_first->bb);
          _gst_print_bytecode_name (bp, bb_start - bb_first->bb, literals);
          printf ("\n");
#endif

	  MATCH_BYTECODES (CREATE_BASIC_BLOCKS, bp, (
            RETURN_SELF,
	    RETURN_SPECIAL,
	    RETURN_METHOD_STACK_TOP,
	    RETURN_CONTEXT_STACK_TOP { break; }

            POP_RECEIVER_VARIABLE,
            POP_TEMPORARY_VARIABLE,
            POP_LIT_VARIABLE,
            POP_STACK_TOP,
            POP_OUTER_TEMP { balance = -1; }

            PUSH_RECEIVER_VARIABLE,
            PUSH_TEMPORARY_VARIABLE,
            PUSH_LIT_CONSTANT,
            PUSH_LIT_VARIABLE,
            PUSH_SELF,
            PUSH_SPECIAL,
            PUSH_SIGNED_8,
            PUSH_UNSIGNED_8,
            PUSH_OUTER_TEMP { balance = 1; }

            LINE_NUMBER_BYTECODE,
            STORE_RECEIVER_VARIABLE,
            STORE_TEMPORARY_VARIABLE,
            STORE_LIT_VARIABLE,
            STORE_OUTER_TEMP,
            NOP_BYTECODE,
            REPLACE_SELF,
            REPLACE_ONE,
            REPLACE_RECEIVER_VARIABLE,
            REPLACE_TEMPORARY_VARIABLE,
            REPLACE_LIT_CONSTANT,
            REPLACE_LIT_VARIABLE,
            EXIT_INTERPRETER,
            MAKE_DIRTY_BLOCK { balance = 0; }

            SEND {
	      balance = -num_args;
	      deepest_access = 1;
	    }

            SEND_ARITH {
	      balance = -1;
	      deepest_access = 1;
	    }

            SEND_SPECIAL {
	      balance = -special_message_num_args[n];
	      deepest_access = 1;
	    }

            POP_INTO_NEW_STACKTOP {
	      balance = -1;
	      deepest_access = 1;
	    }

            DUP_STACK_TOP {
	      balance = 1;
	      deepest_access = 1;
	    }

            INVALID {
	      return ("invalid bytecode");
	    }

            JUMP {
	      balance = 0;
	      if (!bb_start[ofs])
		{
		  int i;
		  bb_start[ofs] = ALLOCA_BASIC_BLOCK ();
		  bb_start[ofs]->next = worklist;
		  bb_start[ofs]->bp = bp + ofs;
		  bb_start[ofs]->bb = bb_start + ofs;
		  bb_start[ofs]->sp = curr_sp + balance;
		  for (i = 0; i < num_temps; i++)
		    bb_start[ofs]->stack[i] = FROM_INT (VARYING);
		  for (; i < bb_start[ofs]->sp; i++)
		    bb_start[ofs]->stack[i] = FROM_INT (UNDEFINED);

		  worklist = bb_start[ofs];
		}
	      else if (curr_sp + balance != bb_start[ofs]->sp)
		return ("stack height mismatch");
	    }

	    POP_JUMP_TRUE, POP_JUMP_FALSE {
	      balance = -1;
	      if (!bb_start[bc_len])
		{
		  int i;
		  bb_start[bc_len] = ALLOCA_BASIC_BLOCK ();
		  bb_start[bc_len]->next = worklist;
		  bb_start[bc_len]->bp = bp + bc_len;
		  bb_start[bc_len]->bb = bb_start + bc_len;
		  bb_start[bc_len]->sp = curr_sp + balance;
		  for (i = 0; i < num_temps; i++)
		    bb_start[bc_len]->stack[i] = FROM_INT (VARYING);
		  for (i = 0; i < bb_start[bc_len]->sp; i++)
		    bb_start[bc_len]->stack[i] = FROM_INT (UNDEFINED);

		  worklist = bb_start[bc_len];
		}
	      else if (curr_sp + balance != bb_start[bc_len]->sp)
		return ("stack height mismatch");

	      if (!bb_start[ofs])
		{
		  int i;
		  bb_start[ofs] = ALLOCA_BASIC_BLOCK ();
		  bb_start[ofs]->next = worklist;
		  bb_start[ofs]->bp = bp + ofs;
		  bb_start[ofs]->bb = bb_start + ofs;
		  bb_start[ofs]->sp = curr_sp + balance;
		  for (i = 0; i < num_temps; i++)
		    bb_start[ofs]->stack[i] = FROM_INT (VARYING);
		  for (i = 0; i < bb_start[ofs]->sp; i++)
		    bb_start[ofs]->stack[i] = FROM_INT (UNDEFINED);

		  worklist = bb_start[ofs];
		}
	      else if (curr_sp + balance != bb_start[ofs]->sp)
		return ("stack height mismatch");
            }
          ));

	  curr_sp += balance;
	  if (curr_sp >= stack_depth)
	    return ("stack overflow");

	  /* Sends touch the new stack top, so they require an extra slot.  */
	  if (curr_sp - deepest_access < 0)
	    return ("stack underflow");

	  bb_start += bc_len, bp += bc_len;
	}
      while (!*bb_start);
    }

  for (worklist = bb_first; worklist; )
    {
      OOP *stack = worklist->stack;
      OOP *sp = stack + worklist->sp;
      bp = worklist->bp;
      bb_start = worklist->bb;
      worklist = worklist->next;

#ifdef DEBUG_VERIFIER
      printf ("Executing basic block at %d:\n", bb_start - bb_first->bb);
#endif
      last_used_literal = NULL;
      do
	{
	  int bc_len = BYTECODE_SIZE (*bp);

#ifdef DEBUG_VERIFIER
          printf ("\t[SP=%3d]  %5d: ", sp - stack, bb_start - bb_first->bb);
          _gst_print_bytecode_name (bp, bb_start - bb_first->bb, literals);
          printf ("\n");
#endif

	  MATCH_BYTECODES (EXEC_BASIC_BLOCK, bp, (
	    PUSH_RECEIVER_VARIABLE {
	      CHECK_REC_VAR (0, n);
	      *sp++ = FROM_INT (VARYING);
	    }

	    PUSH_TEMPORARY_VARIABLE {
	      CHECK_TEMP (n);
	      *sp++ = stack[n];
	    }

	    PUSH_LIT_CONSTANT {
	      CHECK_LITERAL (n);
	      *sp++ = LITERAL_CLASS (n);
	    }

	    PUSH_LIT_VARIABLE {
	      CHECK_LIT_VARIABLE (false, n);
	      *sp++ = LIT_VARIABLE_CLASS (n);
	    }

	    POP_RECEIVER_VARIABLE {
	      CHECK_REC_VAR (num_ro_rec_vars, n);
	      sp--;
	    }

	    POP_TEMPORARY_VARIABLE {
	      CHECK_TEMP (n);
	      sp--;
	    }

	    POP_LIT_VARIABLE {
	      CHECK_LIT_VARIABLE (true, n);
	      sp--;
	    }

	    PUSH_SELF {
	      last_used_literal = NULL;
	      *sp++ = FROM_INT (SELF);
	    }
	    PUSH_SPECIAL {
	      static OOP *specialOOPs[] = {
		&_gst_true_oop, &_gst_false_oop,
		&_gst_nil_oop
	      };

	      last_used_literal = *specialOOPs[n];
	      *sp++ = OOP_CLASS (last_used_literal);
	    }
	    PUSH_SIGNED_8, PUSH_UNSIGNED_8 {
	      last_used_literal = FROM_INT (n);
	      *sp++ = _gst_small_integer_class;
	    }

	    RETURN_SELF,
	    RETURN_SPECIAL,
	    RETURN_METHOD_STACK_TOP,
	    RETURN_CONTEXT_STACK_TOP,
	    NOP_BYTECODE,
	    LINE_NUMBER_BYTECODE { }

	    STORE_RECEIVER_VARIABLE {
	      CHECK_REC_VAR (num_ro_rec_vars, n);
	    }
	    STORE_TEMPORARY_VARIABLE {
	      CHECK_TEMP (n);
	    }
	    STORE_LIT_VARIABLE {
	      CHECK_LIT_VARIABLE (true, n);
	    }

	    SEND {
	      last_used_literal = NULL;
	      sp -= num_args;
	      if (super && sp[-1] != FROM_INT (SELF))
		return ("Invalid send to super");

	      sp[-1] = FROM_INT (VARYING);
	    }

	    POP_INTO_NEW_STACKTOP {
	      if (sp[-2] != _gst_array_class)
	        return ("Array expected");

	      if (!arrays || &sp[-2] - stack != arrays->sp)
	        return ("Invalid Array constructor");

	      if (n >= arrays->size)
	        return ("Out of bounds Array access");

	      last_used_literal = NULL;
	      sp--;
	    }

	    POP_STACK_TOP {
	      last_used_literal = NULL;
	      sp--;
	    }
	    DUP_STACK_TOP {
	      sp++;
	      sp[-1] = sp[-2];
	    }

	    PUSH_OUTER_TEMP {
	      if (scopes > depth || n >= num_outer_temps[scopes])
	        return ("temporary out of range");

	      last_used_literal = NULL;
	      *sp++ = FROM_INT (VARYING);
	    }
	    POP_OUTER_TEMP {
	      if (scopes > depth || n >= num_outer_temps[scopes])
	        return ("temporary out of range");

	      last_used_literal = NULL;
	      sp--;
	    }
	    STORE_OUTER_TEMP {
	      if (scopes > depth || n >= num_outer_temps[scopes])
	        return ("temporary out of range");
	    }

	    REPLACE_SELF {
	      sp[-1] = FROM_INT (SELF);
	      last_used_literal = NULL;
	    }
	    REPLACE_ONE {
	      sp[-1] = _gst_small_integer_class;
	      last_used_literal = FROM_INT (1);
	    }

	    REPLACE_RECEIVER_VARIABLE {
	      CHECK_REC_VAR (0, n);
	      sp[-1] = FROM_INT (VARYING);
	    }
	    REPLACE_TEMPORARY_VARIABLE {
	      CHECK_TEMP (n);
	      sp[-1] = stack[n];
	    }
	    REPLACE_LIT_CONSTANT {
	      CHECK_LITERAL (n);
	      sp[-1] = LITERAL_CLASS (n);
	    }
	    REPLACE_LIT_VARIABLE {
	      CHECK_LIT_VARIABLE (false, n);
	      sp[-1] = LIT_VARIABLE_CLASS (n);
	    }

	    EXIT_INTERPRETER {
	      if (size != 2
		  || bp != GET_METHOD_BYTECODES (methodOOP)
		  || bp[1] != RETURN_CONTEXT_STACK_TOP)
		return ("bad termination method");
	    }

	    JUMP {
	      if (merge_stacks (stack, sp - stack, bb_start[ofs]->stack,
				bb_start[ofs]->sp))
		bb_start[ofs]->next = worklist, worklist = bb_start[ofs];
	    }

	    POP_JUMP_TRUE, POP_JUMP_FALSE {
	      sp--;
	      if (merge_stacks (stack, sp - stack, bb_start[bc_len]->stack,
				bb_start[bc_len]->sp))
		bb_start[bc_len]->next = worklist, worklist = bb_start[bc_len];

	      if (merge_stacks (stack, sp - stack, bb_start[ofs]->stack,
				bb_start[ofs]->sp))
		bb_start[ofs]->next = worklist, worklist = bb_start[ofs];
	    }

	    SEND_ARITH {
	      sp--;
	      sp[-1] = FROM_INT (VARYING);
	    }
	    SEND_SPECIAL {
	      if (*bp == NEW_COLON_SPECIAL
		  && IS_INT (last_used_literal)
		  && sp[-2] == OOP_CLASS (_gst_array_class))
		{
		  partially_constructed_array *a;
		  sp--;

		  /* If possible, reuse an existing struct, else allocate a new one.  */
		  if (arrays_pool)
		    {
		      a = arrays_pool;
		      arrays_pool = arrays_pool->next;
		    }
		  else
		    a = alloca (sizeof (partially_constructed_array));

		  a->size = TO_INT (last_used_literal);
		  a->sp = &sp[-1] - stack;
		  a->next = arrays;
		  arrays = a;

		  sp[-1] = _gst_array_class;
		}
	      else
	        {
		  sp -= special_message_num_args[n];
	          sp[-1] = FROM_INT (VARYING);
		}
	    }

	    MAKE_DIRTY_BLOCK {
	      if (sp[-1] != _gst_compiled_block_class
		  || !last_used_literal)
		return ("CompiledBlock expected");

	      error = _gst_verify_method (last_used_literal, num_outer_temps, depth);
	      if (error)
	        return (error);
	    }

	    INVALID {
	      abort ();
	    }
	  ));

	  /* Discard arrays whose SP is higher than ours.  */
	  while (arrays && &sp[-1] - stack < arrays->sp)
	    {
	      partially_constructed_array *next = arrays->next;
	      arrays->next = arrays_pool;
	      arrays_pool = arrays;
	      arrays = next;
	    }

	  bb_start += bc_len, bp += bc_len;
	}
      while (!*bb_start);
    }

  methodOOP->flags |= F_VERIFIED;
  return (NULL);
}

