/******************************** -*- C -*- ****************************
 *
 *	GNU Smalltalk genbc tool - parser for definitions
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2003 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/

/* This parser builds the decision tree for matching the bytecodes.
   This file knows also how to output C code to visit that tree.  */

%{
#include "genbc.h"
#include "avltrees.h"

#define yyparse decl_yyparse
#define yydebug decl_yydebug
#define YYERROR_VERBOSE
#define YYPRINT yyprint

typedef struct var_info {
  avl_node_t avl;
  char *name;
} var_info;

typedef struct opcode {
  struct opcode *parent, *next, *children;
  int first;
  int last;
  int children_max_bits;
  char *name;
  char *setup_code;
} opcode;

static void define_decl (char *id, int code, int bits);
static void push_tree_decl (char *id, int code);
static void pop_tree_decl ();
static void define_var (char *id);
static void define_field (char *id, int bits);
static void define_computed_field (char *id, char *expr);
static char *extraction_code (int bits);
static void emit_var_names (var_info *node, const char *prefix);
static void emit_opcode_switch (opcode *op, int operand_bits, int depth);

int curr_bit_offset = 0, depth = 0;
var_info *var_root;
opcode *top = NULL, *opcodes = NULL, *curr = NULL, **p_next = &top;
Filament *curr_fil;

%}

%debug
%defines

%union {
  char *text;
  int num;
}

%type<num> bitfields
%type<num> opt_size

%token MATCH_BYTECODES "MATCH_BYTECODES"
%token OPCODE "opcode"
%token<num> NUMBER "number"
%token<text> ID "identifier"
%token<text> EXPR "expression"

%%

decls:
	decls decl
	| decl
	;

decl:
	ID ':' "opcode" NUMBER opt_size
	  { define_decl ($1, $4, $5); curr_fil = filnew (NULL, 0); }
	  bitfields 
	  { if ($7) curr->setup_code = fildelete (curr_fil); }
	  ';'
	  { if (curr_bit_offset % 8)
              yyerror ("instruction is not an integral number of bytes");
	  }
	| ID ':' "opcode" NUMBER 
	  { push_tree_decl ($1, $4); }
	  sub_decl
	  { pop_tree_decl (); }
	;

sub_decl:
	'{' decls '}'
	;

opt_size:
	'(' NUMBER ')'		{ $$ = $2; }
	| /* empty */		{ $$ = 8; }
	;

bitfields:
	bitfields ',' bitfield	{ $$ = $1 + 1; }
	| /* empty */		{ $$ = 0; }
	;

bitfield:
	ID opt_size
	  { define_field ($1, $2); }
	| ID EXPR
	  { define_computed_field ($1, $2); }
	;

%%

/* Advance the pointer by BITS bits and return the code to 
   extract those bits.  */
char *
extraction_code (int bits)
{
  char *s;
  int n_bit = curr_bit_offset % 8;
  int n_byte = curr_bit_offset / 8;

  if (n_bit + bits <= 8)
    {
      int rshift = 8 - (n_bit + bits);
      int mask = (1 << bits) - 1;
      curr_bit_offset += bits;

      if (n_bit && rshift)
        return my_asprintf ("(bp_[%d] >> %d) & %d", n_byte, rshift, mask);

      if (rshift)
        return my_asprintf ("bp_[%d] >> %d", n_byte, rshift);

      if (n_bit)
        return my_asprintf ("bp_[%d] & %d", n_byte, mask);
      else
        return my_asprintf ("bp_[%d]", n_byte);
    }

  /* Else, multi-byte extraction.  */
  if (curr_bit_offset % 8)
    /* Complete the current byte...  */
    {
      int n = 8 - (curr_bit_offset % 8);
      s = extraction_code (n);
      bits -= n;
      n_bit = 0;
      n_byte++;
    }
  else
    /* ... or do a new one.  */
    {
      s = my_asprintf ("bp_[%d]", n_byte++);
      curr_bit_offset += 8;
      bits -= 8;
    }

  /* Add entire bytes as long as possible.  */
  while (bits >= 8)
    {
      char *new_s = my_asprintf ("((%s) << 8) | bp_[%d]", s, n_byte++);
      free (s);
      s = new_s;
      curr_bit_offset += 8;
      bits -= 8;
    }

  /* And finally any spare bits.  */
  if (bits)
    {
      char *new_s = my_asprintf ("((%s) << 8) | %s", s, extraction_code (bits));
      free (s);
      s = new_s;
    }

  return (s);
}

/* Define a variable that is used to pass the operands of the
   bytecode.  We use an AVL tree to store them so that we can
   output them nicely sorted, and we can eliminate duplicates.  */
void
define_var (char *name)
{
  avl_node_t **p = (avl_node_t **) &var_root;
  var_info *node;
  var_info *var = NULL;

  while (*p)
    {
      int cmp;
      var = (var_info *) *p;

      cmp = strcmp(name, var->name);
      if (cmp < 0)
        p = &(*p)->avl_left;
      else if (cmp > 0)
        p = &(*p)->avl_right;
      else
        return;
    }

  node = (var_info *) calloc(1, sizeof(struct var_info));
  node->avl.avl_parent = (avl_node_t *) var;
  node->avl.avl_left = node->avl.avl_right = NULL;
  node->name = name;
  *p = &(node->avl);

  avl_rebalance(&node->avl, (avl_node_t **) &var_root);
}

/* Define an operation that is BITS bits wide and whose opcodes start
   at OPCODE) and does the ID operation.  */
void
define_decl (char *id, int opcode, int bits)
{
  curr = calloc (1, sizeof (struct opcode));
  curr->name = id;
  curr->parent = opcodes;
  curr->first = opcode & ~(255 >> bits);
  curr->last = opcode | (255 >> bits);

  if (opcodes && bits > opcodes->children_max_bits)
    opcodes->children_max_bits = bits;

  *p_next = curr;
  p_next = &(curr->next);

  if (bits > 8 || (opcode & (255 >> bits)))
    yyerror ("Invalid opcode specification");

  curr_bit_offset = depth * 8 + bits;
}

/* Define an operation called ID and whose opcode starts
   with a byte equal to OPCODE.  */
void
push_tree_decl (char *id, int opcode)
{
  define_decl (id, opcode, 8);
  depth++;
  opcodes = curr;
  curr = NULL;
  p_next = &(opcodes->children);
}

/* Return to the upper level of the bytecode decoding tree.  */
void
pop_tree_decl ()
{
  --depth;
  curr = curr->parent;
  p_next = &(curr->next);
  opcodes = curr->parent;
}

/* Define a BITS bits-wide operand named NAME of the current bytecode.  */
void
define_field (char *name, int bits)
{
  char *s = extraction_code (bits);
  define_computed_field (name, s);
  free (s);
}

/* Define a synthetic operand named NAME, and computed with expression
   VALUE, of the current bytecode.  */
void
define_computed_field (char *name, char *value)
{
  define_var (name);
  filcat (curr_fil, name);
  filcat (curr_fil, " =");
  filcat (curr_fil, value);
  filcat (curr_fil, "; ");
}

/* Emit the declarations for the variable names.  NODE is the root of
   the tree, PREFIX ("int" or ",") is emitted before the variable.  */
void
emit_var_names (var_info *node, const char *prefix)
{
  if (node->avl.avl_left)
    {
      emit_var_names ((var_info *) node->avl.avl_left, prefix);
      prefix = ",";
    }

  printf ("%s %s", prefix, node->name);

  if (node->avl.avl_right)
    emit_var_names ((var_info *) node->avl.avl_right, ",");
}

/* Emit the switch()-based decision tree for the tree starting at OP.
   DEPTH is used to indent the code properly.  OPERAND_BITS is used
   to shift right the opcode and limit the gigantic-ness of the switches
   (for instance if all opcodes are 2-bit and followed by 6 operand bits,
   we can switch on (bp_[0] >> 6). */
void
emit_opcode_switch (opcode *op, int operand_bits, int depth)
{
  int indent = depth * 2 + 2;
  int n;

  if (operand_bits)
    printf ("%*sswitch (bp_[%d] >> %d) { \\\n", indent, "", depth, operand_bits);
  else
    printf ("%*sswitch (bp_[%d]) { \\\n", indent, "", depth);

  for (; op; op = op->next)
    {
      int first_val = op->first >> operand_bits;
      int last_val = op->last >> operand_bits;

      for (n = 0; first_val <= last_val; first_val++, n++)
	{
	  if (!(n & 3))
	    printf ("%s%*s", n ? "\\\n" : "", indent, "");

	  printf ("case %d: ", first_val);
	}

      printf ("\\\n");
      if (op->children)
	emit_opcode_switch (op->children, 8 - op->children_max_bits, depth + 1);
      else
        {
	  if (op->setup_code)
	    printf ("%*s  %s\\\n", indent, "", op->setup_code);
	  printf ("%*s  goto MATCH_BYTECODES_##name_##_%s; \\\n", indent, "", op->name);
        }
    }

  printf ("%*s} \\\n", indent, "", depth);
}

/* Emit the decision tree for the bytecodes.  */
void
emit_opcodes ()
{
  if (var_root)
    {
      printf ("  ");
      emit_var_names (var_root, "int");
      printf ("; \\\n");
    }

  emit_opcode_switch (top, 0, 0);
}
