/******************************** -*- C -*- ****************************
 *
 *	Byte code array utility routines.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/

#include "gstpriv.h"

#define BYTECODE_CHUNK_SIZE  64

/* The table of bytecode sizes.  */
const int _gst_bytecode_size_table[256] = {
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 16 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 32 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 48 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 64 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 80 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 96 */
  1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3,	/* 112 */
  2, 2, 2, 2, 3, 2, 3, 1, 1, 1, 3, 1, 1, 1, 2, 1,	/* 128 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 144 */
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 160 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 176 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 192 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 208 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 224 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1	/* 240 */
};

/* Where the compiled bytecodes go.  */
bc_vector _gst_cur_bytecodes;

/* Reallocate an array of bytecodes, leaving space for DELTA more
   bytes.  */
static void realloc_bytecodes (bc_vector bytecodes,
			       int delta);


/* These are the selectors for special case send bytecodes that the compiler
 * generates.  These are used only to print out the generated byte codes.  */
static const char *special_message_name[] = {
  "+",
  "-",
  "<",
  ">",
  "<=",
  ">=",
  "=",
  "~=",
  "*",
  "/",
  "\\",
  "@",
  "bitShift:",
  "//",
  "bitAnd:",
  "bitOr:",
  "at:",
  "at:put:",
  "size",
  "next",
  "nextPut:",
  "atEnd",
  "==",
  "class",
  NULL,
  "value",
  "value:",
  "do:",
  "new",
  "new:",
  "isNil",
  "notNil"
};



bc_vector
_gst_extract_bytecodes (OOP byteArrayOOP)
{
  bc_vector result;
  int len;
  gst_byte_array byteArray;

  byteArray = (gst_byte_array) OOP_TO_OBJ (byteArrayOOP);
  len = oop_num_fields (byteArrayOOP);
  result = (bc_vector) xmalloc (sizeof (struct bytecode_array));

  result->base = (gst_uchar *) xmalloc (len);
  result->ptr = result->base + len;
  result->maxLen = len;
  memcpy (result->base, byteArray->bytes, len);
  return (result);
}

void
_gst_line_number (int n, mst_Boolean force)
{
  static int line;
  if (n > 65535) n = 65535;

  if (n == line && !force)
    return;

  line = n;
  if (n < 0)
    return;

  _gst_compile_byte (LINE_NUMBER_BYTECODE);
  _gst_compile_byte (n >> 8);
  _gst_compile_byte (n & 255);
}

void
_gst_compile_byte (gst_uchar byte)
{
  assert (_gst_cur_bytecodes);

  if ((_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base) >=
	   _gst_cur_bytecodes->maxLen)
    realloc_bytecodes (_gst_cur_bytecodes, BYTECODE_CHUNK_SIZE);

  *_gst_cur_bytecodes->ptr++ = byte;
}

void
_gst_free_bytecodes (bc_vector bytecodes)
{
  if (bytecodes != NULL)
    {
      xfree (bytecodes->base);
      xfree (bytecodes);
    }
}

void
_gst_compile_and_free_bytecodes (bc_vector bytecodes)
{
  _gst_compile_bytecodes (bytecodes->base, bytecodes->ptr);

  /* First add the worst case, then leave the net effect.  */
  ADD_STACK_DEPTH (bytecodes->max_stack_depth);
  SUB_STACK_DEPTH (bytecodes->max_stack_depth - bytecodes->stack_depth);

  _gst_free_bytecodes (bytecodes);
}

bc_vector
_gst_get_bytecodes (void)
{
  bc_vector curBytecodes;

  curBytecodes = _gst_cur_bytecodes;
  _gst_cur_bytecodes = NULL;

  return (curBytecodes);
}


bc_vector
_gst_save_bytecode_array ()
{
  bc_vector curBytecodes;

  curBytecodes = _gst_cur_bytecodes;
  _gst_alloc_bytecodes ();

  return (curBytecodes);
}


void
_gst_restore_bytecode_array (bc_vector bytecodes)
{
  _gst_cur_bytecodes = bytecodes;
}

int
_gst_bytecode_length (bc_vector bytecodes)
{
  if (bytecodes == NULL)
    return (0);

  return (bytecodes->ptr - bytecodes->base);
}


int
_gst_current_bytecode_length (void)
{
  if (_gst_cur_bytecodes == NULL)
    return (0);

  return (_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base);
}

void
_gst_copy_bytecodes (gst_uchar * dest,
		     bc_vector bytecodes)
{
  memcpy (dest, bytecodes->base, _gst_bytecode_length (bytecodes));
}

void
_gst_truncate_bytecodes (gst_uchar * here,
			 bc_vector bytecodes)
{
  bytecodes->ptr = here;
}



void
_gst_print_bytecodes (bc_vector bytecodes,
		      OOP * literal_vec)
{
  gst_uchar *b;
  int ip;

  if (bytecodes == NULL)
    return;

  for (b = bytecodes->base; b < bytecodes->ptr; b += BYTECODE_SIZE (*b))
    {
      ip = b - bytecodes->base;
      printf ("%5d:\t", ip);
      _gst_print_bytecode_name (b, ip, literal_vec);
      printf ("\n");
    }
  printf ("\n");
}

void
_gst_print_bytecode_name (gst_uchar * bp,
			  int ip,
			  OOP * literal_vec)
{
  MATCH_BYTECODES (PRINT_BYTECODE_NAME, bp, (
    PUSH_RECEIVER_VARIABLE {
      printf ("push Instance Variable[%d]", n);
    }

    PUSH_TEMPORARY_VARIABLE {
      printf ("push Temporary Variable[%d]", n);
    }

    PUSH_LIT_CONSTANT {
      printf ("push Literal[%d]", n);
      if (literal_vec)
	printf (" = %O", literal_vec[n]);
    }

    PUSH_LIT_VARIABLE {
      printf ("push Global Variable[%d]", n);
      if (literal_vec)
	printf (" = %+O", literal_vec[n]);
    }

    POP_RECEIVER_VARIABLE {
      printf ("pop and store into Instance Variable[%d]", n);
    }
    POP_TEMPORARY_VARIABLE {
      printf ("pop and store into Temporary Variable[%d]", n);
    }
    POP_LIT_VARIABLE {
      printf ("pop and store into Global Variable[%d]", n);
      if (literal_vec)
	printf (" = %+O", literal_vec[n]);
    }

    PUSH_SELF {
      printf ("push self");
    }
    PUSH_SPECIAL {
      switch (n) {
	case 0: printf ("push true"); break;
	case 1: printf ("push false"); break;
	case 2: printf ("push nil"); break;
      }
    }
    PUSH_SIGNED_8 {
      printf ("push %d", n);
    }
    PUSH_UNSIGNED_8 {
      printf ("push %d", n);
    }

    RETURN_SELF {
      printf ("return self");
    }
    RETURN_SPECIAL {
      switch (n) {
	case 0: printf ("return true"); break;
	case 1: printf ("return false"); break;
	case 2: printf ("return nil"); break;
      }
    }
    RETURN_METHOD_STACK_TOP {
      printf ("return explicitly from method");
    }
    RETURN_CONTEXT_STACK_TOP {
      printf ("return stack top");
    }

    LINE_NUMBER_BYTECODE {
      printf ("source line %d", n);
    }

    STORE_RECEIVER_VARIABLE {
      printf ("store into Instance Variable[%d]", n);
    }
    STORE_TEMPORARY_VARIABLE {
      printf ("store into Temporary Variable[%d]", n);
    }
    STORE_LIT_VARIABLE {
      printf ("store into Global Variable[%d]", n);
      if (literal_vec)
	printf (" = %+O", literal_vec[n]);
    }

    SEND {
      printf ("send selector %d%s, %d args", n, super ? " to super" : "", num_args);
      if (literal_vec)
	printf (" = %O", literal_vec[n]);
    }

    POP_INTO_NEW_STACKTOP {
      printf ("pop and store into Instance Variable[%d] of new stack top", n);
    }

    POP_STACK_TOP {
      printf ("pop stack top");
    }
    DUP_STACK_TOP {
      printf ("duplicate stack top");
    }

    PUSH_OUTER_TEMP {
      printf ("push outer var scopes = %d varIndex = %d", scopes, n);
    }
    POP_OUTER_TEMP {
      printf ("pop outer var scopes = %d varIndex = %d", scopes, n);
    }
    STORE_OUTER_TEMP {
      printf ("pop and store outer var scopes = %d varIndex = %d", scopes, n);
    }

    NOP_BYTECODE {
      printf ("nop bytecode");
    }
    REPLACE_SELF {
      printf ("replace stack top with self");
    }
    REPLACE_ONE {
      printf ("replace stack top with 1");
    }

    REPLACE_RECEIVER_VARIABLE {
      printf ("replace stack top with Instance Variable[%d]", n);
    }
    REPLACE_TEMPORARY_VARIABLE {
      printf ("replace stack top with Temporary Variable[%d]", n);
    }
    REPLACE_LIT_CONSTANT {
      printf ("replace stack top with Literal[%d]", n);
      if (literal_vec)
	printf (" = %O", literal_vec[n]);
    }
    REPLACE_LIT_VARIABLE {
      printf ("replace stack top with Global Variable[%d]", n);
      if (literal_vec)
	printf (" = %+O", literal_vec[n]);
    }

    EXIT_INTERPRETER {
      printf ("terminate interpreter");
    }

    JUMP {
      printf ("jump to %d", ip + ofs);
    }
    POP_JUMP_TRUE {
      printf ("pop and jump to %d if true", ip + ofs);
    }
    POP_JUMP_FALSE {
      printf ("pop and jump to %d if false", ip + ofs);
    }

    SEND_ARITH {
      printf ("send arithmetic message #%s",
	      special_message_name[n]);
    }
    SEND_SPECIAL {
      printf ("send special message #%s",
	      special_message_name[n + 16]);
    }

    MAKE_DIRTY_BLOCK {
      printf ("make dirty block");
    }

    INVALID {
      printf ("INVALID BYTECODE %d", bp[0]);
    }
  ));
}

void
_gst_compile_bytecodes (gst_uchar * from,
			gst_uchar * to)
{
  int free;
  assert (_gst_cur_bytecodes);

  free = _gst_cur_bytecodes->maxLen -
    (_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base);

  if (free < (to - from))
    {
      memcpy (_gst_cur_bytecodes->ptr, from, free);
      _gst_cur_bytecodes->ptr += free;
      from += free;
      realloc_bytecodes (_gst_cur_bytecodes,
		         BYTECODE_CHUNK_SIZE + (to - from));
    }

  memcpy (_gst_cur_bytecodes->ptr, from, to - from);
  _gst_cur_bytecodes->ptr += to - from;
}

void
_gst_alloc_bytecodes ()
{
  bc_vector newBytecodes;

  newBytecodes = (bc_vector) xmalloc (sizeof (struct bytecode_array));
  newBytecodes->base = (gst_uchar *) xmalloc (BYTECODE_CHUNK_SIZE);
  newBytecodes->ptr = newBytecodes->base;
  newBytecodes->maxLen = BYTECODE_CHUNK_SIZE;

  newBytecodes->stack_depth = 0;
  newBytecodes->max_stack_depth = 0;

  _gst_cur_bytecodes = newBytecodes;
}

void
realloc_bytecodes (bc_vector bytecodes,
		   int delta)
{
#ifndef OPTIMIZE
  if (bytecodes->maxLen != (bytecodes->ptr - bytecodes->base))
    _gst_errorf
      ("realloc_bytecodes called with maxLen != byteCode len");
#endif

  bytecodes->base =
    (gst_uchar *) xrealloc (bytecodes->base, bytecodes->maxLen + delta);
  bytecodes->ptr = bytecodes->base + bytecodes->maxLen;
  bytecodes->maxLen += delta;
}
