/* gst-gtk module */
/* Functions to be called from Smalltalk */

#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <glib-object.h>
#include "gstpub.h"
#include "global.h"
#include "basic.h"
#include "callbacks.h"
#include "widgets.h"

void _widget_destroyed(GtkWidget *widget, gpointer user_data);

void *
gst_gtk_create_widget(OOP st_obj, char *aName, char *aType, void *aParent)
{
  GType typ;
  gpointer obj;
  GtkWidget *wgt;

  typ = gtk_type_from_name(aType);
  if (typ == 0) {
    fprintf(stderr, "Gtk type '%s' has not been registered.", aType);
    wgt = NULL;
  } else {
    obj = g_object_newv(typ, 0, NULL);
    wgt = GTK_WIDGET(obj);
    gtk_widget_set_name (wgt, aName);
    gtk_widget_ref (wgt);
    gtk_object_sink (GTK_OBJECT(wgt));

    /* Get a reference to the ST object, to prevent it being garbage collected */ 
    smalltalk_vm_proxy->registerOOP(st_obj);

    /* Connect the destroy signal of the widget to #widgetDestroyed in the ST object */
    gtk_signal_connect (
        GTK_OBJECT (wgt),
        "destroy",
	GTK_SIGNAL_FUNC (_widget_destroyed),
	GINT_TO_POINTER ( (gint)(smalltalk_vm_proxy->OOPToId(st_obj)) )
    );

    /* At some stage, we need to introduce a parameter to govern whether the widget is created visible */
    gtk_widget_show (wgt);
  }
  return(wgt);
}

void
gst_gtk_destroy_widget(void *aWidget)
{
  fprintf(stderr, "Unreffing widget"); /* DEBUG */
  gtk_widget_unref(GTK_WIDGET(aWidget));
}

void
gst_gtk_destroy_window(void *aWidget)
{
  /* A top level window has an extra reference from Gtk itself, so you have to 
     call destroy explicitly. Technically, this fn is misnamed, because it would 
     be valid for any widget, but we're sticking to reference counting for the 
     most part */
  gtk_object_destroy(GTK_OBJECT(aWidget));
}

void _widget_destroyed(GtkWidget *widget, gpointer user_data)
{
  OOP oop;

  /* Inform the ST object that the widget has gone */
  oop = smalltalk_vm_proxy->idToOOP(GPOINTER_TO_INT(user_data));
  smalltalk_vm_proxy->strMsgSend(oop, "widgetDestroyed", NULL);

  /* Release our reference to the ST object - may result in it being GC'd */
  smalltalk_vm_proxy->unregisterOOP (oop);
}

void _unary_callback(gpointer widget, gpointer callback);
void
_unary_callback(gpointer widget, gpointer callback)
{
  GstGtkCallback *cb;

  cb = (GstGtkCallback *) callback;
  smalltalk_vm_proxy->strMsgSend(
    smalltalk_vm_proxy->idToOOP(cb->st_signal),
    "callback",
    NULL
  );
}

gchar *_signal_name(void *widget, char *signal);
gchar *
_signal_name(void *widget, char *signal)
{
  return(g_strdup_printf("%p:%s", widget, signal));
}

void
_signal_connect(void *widget, char *signal, OOP cb_obj, GCallback cb_func)
{
  GstGtkCallback *cb;
  gchar *key;

  gst_gtk_signal_disconnect(widget, signal);

  smalltalk_vm_proxy->registerOOP (cb_obj);

  cb = malloc (sizeof (GstGtkCallback));
  cb->st_signal = smalltalk_vm_proxy->OOPToId(cb_obj);

  /* fprintf(stderr, "oop = %d, id = %d", (int)cb_obj, (int)cb->st_signal); */

  /* A bit inefficient to build the key twice (it was built in
     blox_signal_disconnect), but we probably won't be connecting
     and disconnecting signals all the time, so I reckon it's OK */
  key = _signal_name(widget, signal);

  _add_callback(key, cb);

  cb->handler_id = g_signal_connect(
    G_OBJECT(widget),
    signal,
    cb_func,
    cb
  );
}

void
gst_gtk_signal_connect(void *widget, char *signal, OOP cb_obj)
{
  _signal_connect(widget, signal, cb_obj, G_CALLBACK(_unary_callback));
}

void
gst_gtk_signal_disconnect(void *widget, char *signal)
{
  gchar *key;
  GstGtkCallback *cb;

  key = _signal_name(widget, signal);

  if ( (cb = _get_callback(key)) )
  {
    g_signal_handler_disconnect(widget, cb->handler_id);
    smalltalk_vm_proxy->unregisterOOP(smalltalk_vm_proxy->idToOOP(cb->st_signal));

    /* Second lookup is unavoidable? */
    _remove_callback(key);
  }
  free(key);
}

void
gst_gtk_signal_abort(void *widget, char *signal)
{
  g_signal_stop_emission_by_name (G_OBJECT(widget), signal);
}

void
gst_gtk_signal_emit(void *widget, char *signal)
{
  g_signal_emit_by_name (G_OBJECT(widget), signal);
}

void
gst_gtk_container_add(void *aContainer, void *aWidget)
{
  gtk_container_add (
    GTK_CONTAINER (aContainer),
    GTK_WIDGET (aWidget)
  );
}

void
gst_gtk_container_remove(void *aContainer, void *aWidget)
{
  gtk_container_remove (
    GTK_CONTAINER (aContainer),
    GTK_WIDGET (aWidget)
  );
}



void
gst_gtk_set_property(void *aObject, char *aProperty, OOP aValue)
{
  GParamSpec *spec;
  GObject *obj;
  GValue value = { 0, };

  obj = G_OBJECT(aObject);
  spec = g_object_class_find_property(G_OBJECT_GET_CLASS(obj), aProperty);
  _value_init_from_oop(&value, aValue, spec->value_type);
  g_object_set_property(obj, aProperty, &value);
}

OOP
gst_gtk_get_property(void *aObject, char *aProperty)
{
  GParamSpec *spec;
  GValue result = { 0, };
  GObject *obj;

  obj = G_OBJECT(aObject);
  spec = g_object_class_find_property(G_OBJECT_GET_CLASS(obj), aProperty);
  g_value_init(&result, spec->value_type);
  g_object_get_property(obj, aProperty, &result);
  return(_oop_from_value(&result));
}

/* I've not followed the name of the Gtk function because it isn't
   actually a property of the child that is being set */
void
gst_gtk_set_child_property(void *aContainer, void *aChild, char *aProperty, OOP
 aValue)
{
  GParamSpec *spec;
  GValue value = { 0, };

  spec = gtk_container_class_find_child_property(
    G_OBJECT_GET_CLASS(G_OBJECT(aContainer)), aProperty
  );
  _value_init_from_oop(&value, aValue, spec->value_type);
  gtk_container_child_set_property(
    GTK_CONTAINER(aContainer), GTK_WIDGET(aChild), aProperty, &value
  );
}

OOP
gst_gtk_get_child_property(void *aContainer, void *aChild, char *aProperty)
{
  GParamSpec *spec;
  GValue result = { 0, };

  spec = gtk_container_class_find_child_property(
    G_OBJECT_GET_CLASS(G_OBJECT(aContainer)), aProperty
  );
  g_value_init(&result, spec->value_type);
  gtk_container_child_get_property(
    GTK_CONTAINER(aContainer), GTK_WIDGET(aChild), aProperty, &result
  );
  return(_oop_from_value(&result));
}

void
gst_gtk_size_request(void *aWidget)
{
  GtkRequisition req;
  guint path_length;
  gchar *path;

  gtk_widget_size_request(GTK_WIDGET(aWidget), &req);
  gtk_widget_path(aWidget, &path_length, &path, NULL);
  fprintf(stderr, "Size req for %s is w%d h%d\n", path, req.width, req.height);
  free(path);
}

/*void
*blox_create_treemodel(char *aType, int nColumns, ...)
{
  GtkTreeModel *result;
  va_list cols;
  va_start (cols, nColumns);
  if (strcmp(aType, "GtkTreeStore"))
    result = gtk_tree_store_new(nColumns, cols);
  else if (strcmp(aType, "GtkListStore"))
    result = gtk_list_store_new(nColumns, cols);
  else
    fprintf(stderr, "Unknown type passed to blox_create_treemodel: '%s'", aType);
  va_end (cols);
  return(result);
}*/

int gst_gtk_test_oop(OOP aOop)
{
  OOP oop_result;
  int bool_result;

  oop_result = smalltalk_vm_proxy->strMsgSend(
    aOop,
    "isNil",
    NULL
  ); 

  bool_result = smalltalk_vm_proxy->OOPToBool(oop_result);

  fprintf(stderr, "OOP %d tested OK (%d)", (int) aOop, bool_result);

  /* Actually, we are looking for whether the oop is invalid, so 
     we would have got an error, not a true/false return */
  return(bool_result);
}

void
gst_gtk_test(void)
{
  fprintf(stderr, "Some random tests...\n");
  fprintf(stderr, "...end of tests\n");
}

void
gst_gtk_main(void)
{
  gtk_main();
}

