/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef INCLUDED_USRP_STANDARD_H
#define INCLUDED_USRP_STANDARD_H

#include <usrp_basic.h>

/*!
 * \brief standard usrp RX class.  
 *
 * Assumes digital down converter in FPGA
 */
class usrp_standard_rx : public usrp_basic_rx
{
 private:
  static const int	MAX_CHAN = 4;
  unsigned int		d_decim_rate;
  double		d_rx_freq[MAX_CHAN];


 protected:
  usrp_standard_rx (int which_board, unsigned int decim_rate);	// throws if trouble

  // called after construction in base class to derived class order
  bool initialize ();


 public:
  ~usrp_standard_rx ();

  /*!
   * \brief invokes constructor, returns instance or 0 if trouble
   */
  static usrp_standard_rx *make (int which_board, unsigned int decim_rate);

  /*!
   * \brief Set decimator rate.  \p rate must be in [1, 256]
   *
   * The final complex sample rate across the USB is
   *   adc_freq () / decim_rate ()
   */
  bool set_decim_rate  (unsigned int rate);

  /*!
   * \brief set the frequency of the digital down converter.
   *
   * \p channel must be in the range [0,3].  \p freq is the center
   * frequency in Hz.  \p freq may be either negative or postive.
   * The frequency specified is quantized.  Use rx_freq to retrieve
   * the actual value used.
   */
  bool set_rx_freq (int channel, double freq);  

  // ACCESSORS
  unsigned int decim_rate () const;
  double rx_freq (int channel) const;
};

// ----------------------------------------------------------------

/*!
 * \brief standard usrp TX class.
 *
 * Uses digital upconverter (coarse & fine modulators) in AD9862...
 */
class usrp_standard_tx : public usrp_basic_tx
{
 public:
  enum coarse_mod_t {
    CM_NEG_FDAC_OVER_4,		// -32 MHz
    CM_NEG_FDAC_OVER_8,		// -16 MHz
    CM_OFF,
    CM_POS_FDAC_OVER_8,		// +16 MHz
    CM_POS_FDAC_OVER_4		// +32 MHz
  };

 protected:
  static const int	MAX_CHAN = 2;
  unsigned int		d_interp_rate;
  double		d_tx_freq[MAX_CHAN];
  coarse_mod_t		d_coarse_mod[MAX_CHAN];
  unsigned char		d_tx_modulator_shadow[MAX_CHAN];

  virtual bool set_coarse_modulator (int channel, coarse_mod_t cm);
  usrp_standard_tx::coarse_mod_t coarse_modulator (int channel) const;

 protected:
  usrp_standard_tx (int which_board, unsigned int interp_rate);	// throws if trouble

  // called after construction in base class to derived class order
  bool initialize ();

 public:
  ~usrp_standard_tx ();

  /*!
   * \brief invokes constructor, returns instance or 0 if trouble
   */
  static usrp_standard_tx *make (int which_board, unsigned int interp_rate);

  /*!
   * \brief Set interpolator rate.  \p rate must be in [4, 1024] and a multiple of 4.
   *
   * The final complex sample rate across the USB is
   *   dac_freq () / interp_rate ()
   */
  virtual bool set_interp_rate (unsigned int rate);

  /*!
   * \brief set the frequency of the digital up converter.
   *
   * \p channel must be in the range [0,1].  \p freq is the center
   * frequency in Hz.  It must be in the range [-44M, 44M].
   * The frequency specified is quantized.  Use tx_freq to retrieve
   * the actual value used.
   */
  virtual bool set_tx_freq (int channel, double freq);  // chan: [0,1]

  // ACCESSORS
  unsigned int interp_rate () const;
  double tx_freq (int channel) const;

};

#endif /* INCLUDED_USRP_STANDARD_H */
