/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <usrp_standard.h>

#include "usrp_prims.h"
#include "fpga_regs.h"
#include <stdexcept>
#include <assert.h>
#include <math.h>
#include <ad9862.h>


// #define USE_FPGA_TX_CORDIC


using namespace ad9862;

#define NELEM(x) (sizeof (x) / sizeof (x[0]))

// The FPGA NCOs use a signed 32-bit frequency tuning word,
// but we keep only the most significant 14 bits.

static unsigned int
compute_freq_control_word_fpga (double master_freq, double target_freq,
				double *actual_freq)
{
  static const int NBITS = 14;
  
  int	v = (int) rint (target_freq / master_freq * pow (2.0, 32.0));
  // v += 1 << (32 - NBITS - 1);		// add 1/2
  v = (v >> (32 - NBITS)) << (32 - NBITS);	// keep only top NBITS

  *actual_freq = v * master_freq / pow (2.0, 32.0);

  if (1)
    fprintf (stderr,
	     "compute_freq_control_word_fpga: target = %g  actual = %g  delta = %g\n",
	     target_freq, *actual_freq, *actual_freq - target_freq);

  return (unsigned int) v;
}

// The 9862 uses an unsigned 24-bit frequency tuning word and 
// a separate register to control the sign.

static unsigned int
compute_freq_control_word_9862 (double master_freq, double target_freq,
				double *actual_freq)
{
  double sign = 1.0;

  if (target_freq < 0)
    sign = -1.0;

  int	v = (int) rint (fabs (target_freq) / master_freq * pow (2.0, 24.0));
  *actual_freq = v * master_freq / pow (2.0, 24.0) * sign;

  if (1)
    fprintf (stderr,
     "compute_freq_control_word_9862: target = %g  actual = %g  delta = %g  v = %8d\n",
     target_freq, *actual_freq, *actual_freq - target_freq, v);

  return (unsigned int) v;
}


// ----------------------------------------------------------------

usrp_standard_rx::usrp_standard_rx (int which_board, unsigned int decim_rate)
  : usrp_basic_rx (which_board)
{
  set_decim_rate (decim_rate);

  for (int i = 0; i < MAX_CHAN; i++)
    set_rx_freq (i, 0);
}

usrp_standard_rx::~usrp_standard_rx ()
{
  // nop
}

bool
usrp_standard_rx::initialize ()
{
  if (!usrp_basic_rx::initialize ())
    return false;

  // add our code here

  return true;
}

usrp_standard_rx *
usrp_standard_rx::make (int which_board, unsigned int decim_rate)
{
  usrp_standard_rx *u = 0;
  
  try {
    u = new usrp_standard_rx (which_board, decim_rate);
    if (!u->initialize ()){
      fprintf (stderr, "usrp_standard_rx::make failed to initialize\n");
      throw std::runtime_error ("usrp_standard_rx::make");
    }
    return u;
  }
  catch (...){
    delete u;
    return 0;
  }

  return u;
}

bool
usrp_standard_rx::set_decim_rate  (unsigned int rate)
{
  d_decim_rate = rate;
  // setup_status_polling ();

  bool ok;

  if (rx_enable ()){
    set_rx_enable (false);
    ok = _write_fpga_reg (FR_DECIM_RATE, d_decim_rate - 1);
    set_rx_enable (true);
  }
  else
    ok = _write_fpga_reg (FR_DECIM_RATE, d_decim_rate - 1);

  return ok;
}

bool
usrp_standard_rx::set_rx_freq (int channel, double freq)
{
  if (channel < 0 || channel > MAX_CHAN)
    return false;

  unsigned int v =
    compute_freq_control_word_fpga (fpga_master_clock_freq () / 2,
				    freq, &d_rx_freq[channel]);

  return _write_fpga_reg (FR_RX_FREQ_0 + channel, v);
}

unsigned int
usrp_standard_rx::decim_rate () const { return d_decim_rate; }


double 
usrp_standard_rx::rx_freq (int channel) const
{
  if (channel < 0 || channel >= MAX_CHAN)
    return 0;

  return d_rx_freq[channel];
}

//////////////////////////////////////////////////////////////////


// tx data is timed to CLKOUT1 (64 MHz)
// interpolate 4x
// fine modulator enabled


static unsigned char tx_regs_use_nco[] = {
  REG_TX_IF,		(TX_IF_USE_CLKOUT1
			 | TX_IF_I_FIRST
			 | TX_IF_2S_COMP
			 | TX_IF_INTERLEAVED),
  REG_TX_DIGITAL,	(TX_DIGITAL_2_DATA_PATHS
			 | TX_DIGITAL_INTERPOLATE_4X)
};


usrp_standard_tx::usrp_standard_tx (int which_board, unsigned int interp_rate)
  : usrp_basic_tx (which_board)
{
  if (!usrp_9862_write_many_all (d_udh, tx_regs_use_nco, sizeof (tx_regs_use_nco))){
    fprintf (stderr, "usrp_standard_tx: failed to init AD9862 TX regs\n");
    throw std::runtime_error ("usrp_standard_tx/init_9862");
  }

  set_interp_rate (interp_rate);

  for (int i = 0; i < MAX_CHAN; i++){
    d_tx_modulator_shadow[i] = (TX_MODULATOR_DISABLE_NCO
				| TX_MODULATOR_COARSE_MODULATION_NONE);
    d_coarse_mod[i] = CM_OFF;
    set_tx_freq (i, 0);
  }
}

usrp_standard_tx::~usrp_standard_tx ()
{
  // nop
}

bool
usrp_standard_tx::initialize ()
{
  if (!usrp_basic_tx::initialize ())
    return false;

  // add our code here

  return true;
}

usrp_standard_tx *
usrp_standard_tx::make (int which_board, unsigned int interp_rate)
{
  usrp_standard_tx *u = 0;
  
  try {
    u = new usrp_standard_tx (which_board, interp_rate);
    if (!u->initialize ()){
      fprintf (stderr, "usrp_standard_tx::make failed to initialize\n");
      throw std::runtime_error ("usrp_standard_tx::make");
    }
    return u;
  }
  catch (...){
    delete u;
    return 0;
  }

  return u;
}

bool
usrp_standard_tx::set_interp_rate (unsigned int rate)
{
  // fprintf (stderr, "usrp_standard_tx::set_interp_rate\n");

  if (rate & 0x3){
    fprintf (stderr, "usrp_standard_tx::set_interp_rate: rate must be a multiple of 4!\n");
    return false;
  }

  d_interp_rate = rate;
  // setup_status_polling ();

  // We're using the interp by 4 feature of the 9862 so that we can
  // use its fine modulator.  Thus, we reduce the FPGA's interpolation rate
  // by a factor of 4.

  bool ok;

  if (tx_enable ()){
    set_tx_enable (false);
    ok = _write_fpga_reg (FR_INTERP_RATE, d_interp_rate/4 - 1);
    set_tx_enable (true);
  }
  else
    ok = _write_fpga_reg (FR_INTERP_RATE, d_interp_rate/4 - 1);    

  return ok;
}

#ifdef USE_FPGA_TX_CORDIC

bool
usrp_standard_tx::set_tx_freq (int channel, double freq)
{
  if (channel < 0 || channel >= MAX_CHAN)
    return false;

  // This assumes we're running the 4x on-chip interpolator.

  unsigned int v =
    compute_freq_control_word_fpga (dac_freq () / 4,
				    freq, &d_tx_freq[channel]);

  return _write_fpga_reg (FR_TX_FREQ_0 + channel, v);
}


#else

bool
usrp_standard_tx::set_tx_freq (int channel, double freq)
{
  if (channel < 0 || channel >= MAX_CHAN)
    return false;

  // split freq into fine and coarse components

  coarse_mod_t	cm;
  double	coarse;

  assert (dac_freq () == 128000000);

  if (freq < -44e6)		// too low
    return false;
  else if (freq < -24e6){	// [-44, -24)
    cm = CM_NEG_FDAC_OVER_4;
    coarse = -dac_freq () / 4;
  }
  else if (freq < -8e6){	// [-24, -8)
    cm = CM_NEG_FDAC_OVER_8;
    coarse = -dac_freq () / 8;
  }
  else if (freq < 8e6){		// [-8, 8)
    cm = CM_OFF;
    coarse = 0;
  }
  else if (freq < 24e6){	// [8, 24)
    cm = CM_POS_FDAC_OVER_8;
    coarse = dac_freq () / 8;
  }
  else if (freq <= 44e6){	// [24, 44]
    cm = CM_POS_FDAC_OVER_4;
    coarse = dac_freq () / 4;
  }
  else				// too high
    return false;


  set_coarse_modulator (channel, cm);	// set bits in d_tx_modulator_shadow

  double fine = freq - coarse;


  // Compute fine tuning word...
  // This assumes we're running the 4x on-chip interpolator.
  // (This is required to use the fine modulator.)

  unsigned int v =
    compute_freq_control_word_9862 (dac_freq () / 4,
				    fine, &d_tx_freq[channel]);

  d_tx_freq[channel] += coarse;		// adjust actual
  
  unsigned char high, mid, low;

  high = (v >> 16) & 0xff;
  mid =  (v >>  8) & 0xff;
  low =  (v >>  0) & 0xff;

  bool ok = true;

  // write the fine tuning word
  ok &= _write_9862 (channel, REG_TX_NCO_FTW_23_16, high);
  ok &= _write_9862 (channel, REG_TX_NCO_FTW_15_8,  mid);
  ok &= _write_9862 (channel, REG_TX_NCO_FTW_7_0,   low);


  d_tx_modulator_shadow[channel] |= TX_MODULATOR_ENABLE_NCO;

  if (fine < 0)
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_NEG_FINE_TUNE;
  else
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_NEG_FINE_TUNE;

  ok &=_write_9862 (channel, REG_TX_MODULATOR, d_tx_modulator_shadow[channel]);

  return ok;
}
#endif

bool
usrp_standard_tx::set_coarse_modulator (int channel, coarse_mod_t cm)
{
  if (channel < 0 || channel >= MAX_CHAN)
    return false;

  switch (cm){
  case CM_NEG_FDAC_OVER_4:
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_CM_MASK;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_COARSE_MODULATION_F_OVER_4;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_NEG_COARSE_TUNE;
    break;

  case CM_NEG_FDAC_OVER_8:
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_CM_MASK;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_COARSE_MODULATION_F_OVER_8;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_NEG_COARSE_TUNE;
    break;

  case CM_OFF:
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_CM_MASK;
    break;

  case CM_POS_FDAC_OVER_8:
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_CM_MASK;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_COARSE_MODULATION_F_OVER_8;
    break;

  case CM_POS_FDAC_OVER_4:
    d_tx_modulator_shadow[channel] &= ~TX_MODULATOR_CM_MASK;
    d_tx_modulator_shadow[channel] |= TX_MODULATOR_COARSE_MODULATION_F_OVER_4;
    break;

  default:
    return false;
  }

  d_coarse_mod[channel] = cm;
  // return _write_9862 (channel, REG_TX_MODULATOR, d_tx_modulator_shadow[channel]);
  return true;
}

unsigned int
usrp_standard_tx::interp_rate () const { return d_interp_rate; }

double
usrp_standard_tx::tx_freq (int channel) const
{
  if (channel < 0 || channel >= MAX_CHAN)
    return 0;

  return d_tx_freq[channel];
}

usrp_standard_tx::coarse_mod_t
usrp_standard_tx::coarse_modulator (int channel) const
{
  if (channel < 0 || channel >= MAX_CHAN)
    return CM_OFF;

  return d_coarse_mod[channel];
}
