/* GNU Mailutils -- a suite of utilities for electronic mail
   Copyright (C) 1999, 2000, 2001 Free Software Foundation, Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA  */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/time.h>
#include <unistd.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <mailutils/sys/pop3.h>

static int pop3_sleep (int seconds);

/* Open the connection to the server. The server sends an affirmative greeting
   that may contain a timestamp for APOP.  */
int
pop3_connect (pop3_t pop3, const char *host, unsigned int port)
{
  int status;

  /* Sanity checks.  */
  if (pop3 == NULL || host == NULL)
    return MU_ERROR_INVALID_PARAMETER;

  /* Default is 110.  */
  if (!port)
    port = 110;

  /* Enter the pop state machine, and boogy: AUTHORISATION State.  */
  switch (pop3->state)
    {
    default:
      /* FALLTHROUGH */
      /* If pop3 was in an error state going through here should clear it.  */

    case POP3_NO_STATE:
      /* Create the networking stack.  */
      if (pop3->carrier == NULL)
	{
	  stream_t carrier;
	  status = pop3_get_carrier (pop3, &carrier);
	  POP3_CHECK_ERROR (pop3, status);
	  /* An stream_ref was done in po3_get_carrier ().  */
	  stream_destroy (&carrier);
	}
      else
	{
          /* This is sudden death: for many pop servers, it is important to
             let them time to remove locks or move the .user.pop files.  This
             happen when we do close() and immediately open().  For example,
	     the user does not want to read the entire file, and wants to start
	     to read a new message, closing the connection and immediately
	     contacting the server again, and he'll end up having
	     "-ERR Mail Lock busy" or something similar. To prevent this race
	     condition we sleep 2 seconds.  This is really obvious for in
	     environment where QPopper is use and the user as a big mailbox. */
	  pop3_disconnect (pop3);
          pop3_sleep (2);
	}
      pop3->state = POP3_CONNECT;

    case POP3_CONNECT:
      /* Establish the connection.  */
      status = stream_open (pop3->carrier, host, port,
			    MU_STREAM_READ|MU_STREAM_WRITE);
      POP3_CHECK_EAGAIN (pop3, status);
      pop3->acknowledge = 0;
      pop3->state = POP3_GREETINGS;

    case POP3_GREETINGS:
      /* Get the greetings.  */
      {
	size_t len = 0;
	char *right, *left;
	status = pop3_response (pop3, NULL, 0, &len);
	POP3_CHECK_EAGAIN (pop3, status);
	pop3_debug_ack (pop3);
	if (strncasecmp (pop3->ack.buf, "+OK", 3) != 0)
	  {
	    stream_close (pop3->carrier);
	    pop3->state = POP3_NO_STATE;
	    return MU_ERROR_OPERATION_DENIED;
	  }

	/* Get the timestamp.  */
	right = memchr (pop3->ack.buf, '<', len);
	if (right)
	  {
	    len = len - (right - pop3->ack.buf);
	    left = memchr (right, '>', len);
	    if (left)
	      {
		len = left - right;
		pop3->timestamp = calloc (len + 1, 1);
		if (pop3->timestamp == NULL)
		  {
		    stream_close (pop3->carrier);
		    POP3_CHECK_ERROR (pop3, MU_ERROR_NO_MEMORY);
		  }
		/* Do not copy the surrounding '<>'.  */
		memcpy (pop3->timestamp, right + 1, len - 1);
	      }
	  }
	pop3->state = POP3_NO_STATE;
      }
    } /* End AUTHORISATION state. */

  return status;
}

/* GRRRRR!!  We can not use sleep in the library since this we'll
   muck up any alarm() done by the user.  */
static int
pop3_sleep (int seconds)
{
  struct timeval tval;
  tval.tv_sec = seconds;
  tval.tv_usec = 0;
  return select (1, NULL, NULL, NULL, &tval);
}
