/* GNU Mailutils -- a suite of utilities for electronic mail
   Copyright (C) 1999, 2000, 2001 Free Software Foundation, Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA  */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <errno.h>
#include <stdlib.h>
#include <mailutils/iterator.h>
#include <mailutils/error.h>
#include <mailutils/sys/observable.h>

struct observer_info
{
  int type;
  observer_t observer;
};

int
observable_create (observable_t *pobservable)
{
  observable_t observable;
  int status;
  if (pobservable == NULL)
    return MU_ERROR_INVALID_PARAMETER;
  observable = calloc (sizeof (*observable), 1);
  if (observable == NULL)
    return MU_ERROR_NO_MEMORY;
  status = mu_list_create (&(observable->list));
  if (status != 0 )
    {
      free (observable);
      return status;
    }
  *pobservable = observable;
  return 0;
}

void
observable_destroy (observable_t *pobservable)
{
  if (pobservable && *pobservable)
    {
      observable_t observable = *pobservable;
      iterator_t iterator = NULL;
      int status = mu_list_get_iterator (observable->list, &iterator);
      if (status == 0)
	{
	  struct observer_info *info;
	  for (iterator_first (iterator); !iterator_is_done (iterator);
	       iterator_next (iterator))
	    {
	      info = NULL;
	      iterator_current (iterator, (void **)&info);
	      if (info)
		{
		  observer_destroy (&info->observer);
		  free (info);
		}
	    }
	  iterator_destroy (&iterator);
	}
      mu_list_destroy (&observable->list);
      free (observable);
      *pobservable = NULL;
    }
}

int
observable_attach (observable_t observable, int type,  observer_t observer)
{
  struct observer_info *info;
  if (observable == NULL || observer == NULL)
    return MU_ERROR_INVALID_PARAMETER;
  info = calloc (1, sizeof *info);
  if (info == NULL)
    return MU_ERROR_NO_MEMORY;
  info->type = type;
  info->observer = observer;
  return mu_list_append (observable->list, info);
}

int
observable_detach (observable_t observable, observer_t observer)
{
  iterator_t iterator = NULL;
  int status;
  int found = 0;
  struct observer_info *info;
  if (observable == NULL ||observer == NULL)
    return EINVAL;
  status = mu_list_get_iterator (observable->list, &iterator);
  if (status != 0)
    return status;
  for (iterator_first (iterator); !iterator_is_done (iterator);
       iterator_next (iterator))
    {
      info = NULL;
      iterator_current (iterator, (void **)&info);
      if (info && (int)(info->observer) == (int)observer)
        {
          found = 1;
          break;
        }
    }
  iterator_destroy (&iterator);
  if (found)
    {
      status = mu_list_remove (observable->list, info);
      free (info);
    }
  return status;
}

int
observable_notify_all (observable_t observable, struct event evt)
{
  iterator_t iterator;
  struct observer_info *info;
  int status = 0;

  if (observable == NULL)
    return MU_ERROR_INVALID_PARAMETER;
  status = mu_list_get_iterator (observable->list, &iterator);
  if (status != 0)
    return status;
  for (iterator_first (iterator); !iterator_is_done (iterator);
       iterator_next (iterator))
    {
      info = NULL;
      iterator_current (iterator, (void **)&info);
      if (info && info->type & evt.type)
        {
	  status |= observer_action (info->observer, evt);
        }
    }
  iterator_destroy (&iterator);
  return status;
}
