#include <stdio.h>
#include <string.h>
#include <interface/Bitmap.h>
#include <app/Application.h>
#include <support/Debug.h>

#include "BTSNetDrawWindow.h"
#include "BTSDrawView.h"
#include "BTSNetDrawClient.h"
#include "BTSNetDrawMsgDefs.h"


// =============================================================================
//     BTSNetDrawWindow
// =============================================================================
BTSNetDrawWindow::BTSNetDrawWindow(BRect size, char* inName) :
	 BWindow(size, inName, B_TITLED_WINDOW, B_NOT_RESIZABLE)
{
	Lock();
	// Align all the children in the window.
	size.OffsetTo(B_ORIGIN);

	// Create and add the children.
	fOffView = new BView(size, "OffView", 0,
		B_NOT_RESIZABLE);
	fOffBitMap = new BBitmap(size, B_MONOCHROME_1_BIT, TRUE);

	fDrawView = new BTSDrawView(size, fOffBitMap);
	
	fOffBitMap->AddChild(fOffView);
	AddChild(fDrawView);
	Unlock();
	

	return;
}
	
	
// =============================================================================
//     QuitRequested
// =============================================================================
bool	
BTSNetDrawWindow::QuitRequested()
{
	// Tell the application to quit!
	be_app->PostMessage(B_QUIT_REQUESTED);
	return(TRUE);
}

// =============================================================================
//     MessageReceived
// =============================================================================
void 
BTSNetDrawWindow::MessageReceived(BMessage* inMessage)
{
	PRINT(("BTSNetDrawWindow::MessageReceived - ENTER\n"));
	switch (inMessage->what)
	{
		case NETDRAWLINE_MSG:
		
			PRINT(("NET DRAW\n"));
			// Draw the line in this client's window.
			BPoint 	start = inMessage->FindPoint("LineStart");
			if (inMessage->Error() == B_NO_ERROR)
			{
				BPoint 	end = inMessage->FindPoint("LineEnd");
				if (inMessage->Error() == B_NO_ERROR)
				{
					PRINT(("Drawing...\n"));
					DrawLine(start, end);
				}
			}
						
			PRINT(("NET DRAW END\n"));		
		break;
		
		case BITMAP_DATA_MSG:
			long bitmapSize;
			if (inMessage->HasData("Bitmap", B_RAW_TYPE))
			{
				void* bitmapData = inMessage->FindData("Bitmap", B_RAW_TYPE, &bitmapSize);
				SetBitmap(bitmapData, bitmapSize);
			}
		break;
		
		case QUIT_DRAW_WINDOW_MSG:	
			PRINT(("Window quitting...\n"));
			fDrawView->SetExiting(TRUE);
			BWindow::QuitRequested();
		break;
		
		case CLEAR_MSG:
			Clear();
		break;
		
		default:
			BWindow::MessageReceived(inMessage);
		break;				
	}
	PRINT(("BTSNetDrawWindow::MessageReceived - EXIT\n"));
}

// =============================================================================
//     DrawLine
// =============================================================================
void 
BTSNetDrawWindow::DrawLine(BPoint start, BPoint end)
{
	PRINT(("BTSNetDrawWindow::DrawLine - ENTER\n"));	
	BRect	boundsRect;
	SetRect(&boundsRect, start, end);
	fOffBitMap->Lock();
	fOffView->MovePenTo(start.x, start.y);
	fOffView->StrokeLine(end);	
	fOffView->Sync();
	fOffBitMap->Unlock();
	
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, boundsRect, boundsRect);
	fDrawView->Sync();
	Unlock();
	PRINT(("BTSNetDrawWindow::DrawLine - EXIT\n"));
	return;	
}
	
// =============================================================================
//     Clear
// =============================================================================
void 
BTSNetDrawWindow::Clear()
{
	void* buf = fOffBitMap->Bits();
	long numBytes = fOffBitMap->BitsLength();
	memset(buf, 0, numBytes);
	fOffBitMap->Lock();
	fOffBitMap->SetBits(buf, numBytes, 0, B_MONOCHROME_1_BIT);
	fOffView->Sync();
	fOffBitMap->Unlock();
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, Bounds(), Bounds());
	Unlock();
}

// =============================================================================
//     SetBitmap
// =============================================================================	
void BTSNetDrawWindow::SetBitmap(void* bitmapData, long bitmapSize)
{
	fOffBitMap->Lock();
	fOffBitMap->SetBits(bitmapData, bitmapSize, 0, B_MONOCHROME_1_BIT);
	fOffView->Sync();
	fOffBitMap->Unlock();
	Lock();
	fDrawView->DrawBitmap(fOffBitMap, Bounds(), Bounds());
	Unlock();
}

// =============================================================================
//     WindowActivated
// =============================================================================	
void BTSNetDrawWindow::WindowActivated(bool active)
{
	BPoint mousePoint;
	ulong 	buttons;
	fDrawView->GetMouse(&mousePoint, &buttons);
	fDrawView->SetLineStart(mousePoint);
	return;
}

// =============================================================================
//     SetRect
// =============================================================================	
void
BTSNetDrawWindow::SetRect(BRect* boundsRect, BPoint lineStart, BPoint lineEnd)
{
	float top, left, bot, right;
	if (lineStart.x < lineEnd.x)
	{
		left = lineStart.x;
		right = lineEnd.x;
	}
	else
	{
		left = lineEnd.x;
		right = lineStart.x;
	}
	if (lineStart.y < lineEnd.y)
	{
		top = lineStart.y;
		bot = lineEnd.y;
	}
	else
	{
		top = lineEnd.y;
		bot = lineStart.y;
	}
	boundsRect->Set(left, top, right, bot);
	return;
}