/*
[.......*1......*..2....*....3..*......4*.......*5......*..6....*....7..*......]
	SWindow.cpp
the status window itself
(c) 1996 Benoit Triquet
*/


#include "SWindow.h"
#include "SysCon.h"


//*********************** management of the little icons * Icons & BBitmaps ****
//******************************************************************************


//	Builtin icons
//------------------------------------------------------------------------------
//	16x16 pixels, 256 colors icons compiled in the application (yes, i could
//	have used resources, yes i'm lazy :-)
//
extern uchar info_8bit_image[ ], warning_8bit_image[ ], error_8bit_image[ ];
#define ICON_W 16
#define ICON_H 16


//	create_info_bitmap, create_warning_bitmap, create_error_bitmap
//------------------------------------------------------------------------------
//	Return a new BBitmap with one of the icons'bits.
//
static BBitmap *create_info_bitmap( )
{
	BRect r( 0, 0, ICON_W-1, ICON_H-1 );
	// pray for 4 bytes alignment, btw
	BBitmap *bm = new BBitmap( r, B_COLOR_8_BIT );
	bm->SetBits( info_8bit_image, 256, 0, B_COLOR_8_BIT );
	return bm;
}


static BBitmap *create_warning_bitmap( )
{
	BRect r( 0, 0, ICON_W-1, ICON_H-1 );
	BBitmap *bm = new BBitmap( r, B_COLOR_8_BIT );
	bm->SetBits( warning_8bit_image, 256, 0, B_COLOR_8_BIT );
	return bm;
}


static BBitmap *create_error_bitmap( )
{
	BRect r( 0, 0, ICON_W-1, ICON_H-1 );
	BBitmap *bm = new BBitmap( r, B_COLOR_8_BIT );
	bm->SetBits( error_8bit_image, 256, 0, B_COLOR_8_BIT );
	return bm;
}




//************************************ list of the messages * class MsgList ****
//******************************************************************************


//	struct ml_node
//------------------------------------------------------------------------------
//	nodes of the BList in class MsgList.
//
struct ml_node {
	float y, h, w; // y, height and width of this msg in the view's bounds
	BBitmap *icn; // the icon (cache for the ptr obtained with s)
	long s; // the status value
	char **txt; // a null-terminated list of lines of text for this msg
};


//	Class MsgList
//------------------------------------------------------------------------------
//	.
//
class MsgList : public BView {
public:
	MsgList( BRect, char *, ulong );
	virtual ~MsgList( );
	void AttachedToWindow( );
	void FrameResized( float, float ); // no FrameMoved, it never moves
	void Draw( BRect );
	void MouseDown( BPoint );
	void MouseMoved( BPoint, ulong, BMessage * );
	void MessageReceived( BMessage * );
	void AddMsg( long, char * );
	void Activated( bool );
	void Clear( );
private:
	void Recalc( );
	void RecalcOne( ml_node * );
	void RecalcScrollBars( );
	void ClearList( );
	char *GetText( long *s = NULL );
	BBitmap *info_bitmap, *warning_bitmap, *error_bitmap;
	BList ml;
	float width, height, text_line_height, text_ascent;
	int selected;
	bool activated;
};


//	MsgList::MsgList
//------------------------------------------------------------------------------
//	Of course all MsgList instances could have shared the same bitmaps...
//	They could have been class members... No they couldn't ! (ever heard of
//	static initialization ? can't be done before "new BApplication(..);" !)
//	They could have been globals... I HATE globals.
//	Oh well, there is only one instance of MsgList in this program :-)
//
MsgList::MsgList( BRect r, char *n, ulong m )
	: BView( r, n, m, B_WILL_DRAW|B_FRAME_EVENTS )
{
	info_bitmap = create_info_bitmap( );
	warning_bitmap = create_warning_bitmap( );
	error_bitmap = create_error_bitmap( );
	selected = -1;
}


//	MsgList::~MsgList
//------------------------------------------------------------------------------
//	great !
//
MsgList::~MsgList( )
{
	ClearList( );
	delete info_bitmap;
	delete warning_bitmap;
	delete error_bitmap;
}


//	MsgList::ClearList
//------------------------------------------------------------------------------
//	Frees the text lines and the nodes, then empties the BList & clear selected.
//
void MsgList::ClearList( )
{
	for( int i = 0; i<ml.CountItems( ); i ++ ) {
		ml_node *n = ( ml_node * )ml.ItemAt( i );
		char **l = n->txt;
		while( *l )
			delete [ ] *( l ++ );
		delete [ ] n->txt;
		delete n;
	}
	ml.MakeEmpty( );
	selected = -1;
}


//	MsgList::Clear
//------------------------------------------------------------------------------
//	Clears the list and refresh the view.
//
void MsgList::Clear( )
{
	Window( )->Lock( );
	ClearList( );
	Recalc( );
	Invalidate( );
	Window( )->Unlock( );
}


//	MsgList::AttachedToWindow
//------------------------------------------------------------------------------
//	Font measurements.
//
void MsgList::AttachedToWindow( )
{
	SetFontName( "Kate" );
	font_info fi;
	GetFontInfo( &fi );
	text_ascent = fi.ascent;
	text_line_height = fi.ascent+fi.descent+fi.leading;
	Recalc( );
}


//	MsgList::FrameResized
//------------------------------------------------------------------------------
//	Update scroll bars.
//
void MsgList::FrameResized( float w, float h )
{
	RecalcScrollBars( );
}


//	MsgList::RecalcScrollBars
//------------------------------------------------------------------------------
//	proportional bars, please.
//
void MsgList::RecalcScrollBars( )
{
	BRect bounds = Bounds( );
	BScrollBar *v = ScrollBar( B_VERTICAL ), *h = ScrollBar( B_HORIZONTAL );
	v->SetRange( 0, max( 0, height-bounds.Height( ) ) );
	v->SetProportion( bounds.Height( )/height );
	v->SetSteps( text_line_height, bounds.Height( ) );
	h->SetRange( 0, max( 0, width-bounds.Width( ) ) );
	h->SetProportion( bounds.Width( )/width );
	h->SetSteps( StringWidth( "M" ), bounds.Width( ) );
}


//	MsgList::Draw
//------------------------------------------------------------------------------
//	We redraw on a message basis (ie. all the lines and the icon of a message,
//	but not the whole list).
//
void MsgList::Draw( BRect upd )
{
	for( int i = 0; i<ml.CountItems( ); i ++ ) {
		ml_node *n = ( ml_node * )ml.ItemAt( i );
		if( n->y+n->h<=upd.top )
			continue;
		if( n->y>upd.bottom )
			break;
		DrawBitmapAsync( n->icn, BPoint( 0, n->y ) );
		char **cl = n->txt;
		float y = n->y+text_ascent;
		while( *cl ) {
			DrawString( *cl, BPoint( ICON_W, y ) );
			y += text_line_height;
			cl ++;
		}
	}
	if( activated && selected>=0 ) {
		ml_node *n = ( ml_node * )ml.ItemAt( selected );
		InvertRect( BRect( ICON_W-1, n->y, Bounds( ).right, n->y+n->h-1 ) );
	}
}


//	MsgList::Recalc
//------------------------------------------------------------------------------
//	Compute the width and heigth of the view's Frame.
//
void MsgList::Recalc( )
{
	width = 0;
	height = 0;
	for( int i = 0; i<ml.CountItems( ); i ++ ) {
		ml_node *n = ( ml_node * )ml.ItemAt( i );
		height += n->h;
		if( n->w>width )
			width = n->w;
	}
	RecalcScrollBars( );
}


//	MsgList::RecalcOne
//------------------------------------------------------------------------------
//	Update the width and height of the view's frame and compute the y, w, h of
//	the new node.
//
void MsgList::RecalcOne( ml_node *n )
{
	n->y = height;
	char **cl = n->txt;
	int nl = 0;
	n->w = 0;
	while( *cl ) {
		float sw = StringWidth( *cl )+ICON_W;
		if( sw>n->w )
			n->w = sw;
		nl ++;
		cl ++;
	}
	n->h = max( ICON_H, nl*text_line_height );
	height += n->h;
	width = max( width, n->w );
}


//	MsgList::Activated
//------------------------------------------------------------------------------
//	The guide lines say "hide the selection when the window is not activated".
//	I know, it's not a very good idea to redraw everything, but it's way
//	simpler to do it that way. Simply Inverting the rectangle from here does
//	_not_ work (in the case where you have a window partially covering the
//	selection and you activate SysCon).
//
void MsgList::Activated( bool a )
{
	activated = a;
	Invalidate( );
}


//	MsgList::MouseDown
//------------------------------------------------------------------------------
//	Left button : select and deselect a message.
//	Other buttons : begin a drag'n'drop session (too bad the bitmap transparency
//	is not taken into account...)
//
void MsgList::MouseDown( BPoint p )
{
	BRect bounds = Bounds( );
	ulong buttons = Window( )->CurrentMessage( )->FindLong( "buttons" );

	int i;
	ml_node *n, *prev_n;
	for( i = 0; i<ml.CountItems( ); i ++ ) {
		n = ( ml_node * )ml.ItemAt( i );
		if( n->y<=p.y && p.y<n->y+n->h )
			break;
	}
	if( i!=ml.CountItems( ) ) {
		if( selected==i ) {
			if( buttons==B_PRIMARY_MOUSE_BUTTON ) {
				InvertRect( BRect( ICON_W-1, n->y,
				 bounds.right, n->y+n->h-1 ) );	
				selected = -1;
			}
		} else {
			if( selected>=0 ) {
				prev_n = ( ml_node * )ml.ItemAt( selected );
				InvertRect( BRect( ICON_W-1, prev_n->y, bounds.right,
				 prev_n->y+prev_n->h-1 ) );	
			}
			selected = i;
			InvertRect( BRect( ICON_W-1, n->y, bounds.right, n->y+n->h-1 ) );
		}
	} else
		if( selected>=0 ) {
			n = ( ml_node * )ml.ItemAt( selected );
			InvertRect( BRect( ICON_W-1, n->y, bounds.right, n->y+n->h-1 ) );	
			selected = -1;
		}

	char *buf;
	long s;
	if( buttons==B_TERTIARY_MOUSE_BUTTON || buttons==B_SECONDARY_MOUSE_BUTTON )
		if( buf = GetText( &s ) ) {
			BMessage *msg = new BMessage( B_SIMPLE_DATA );
			msg->AddData( "text", B_ASCII_TYPE, buf, strlen( buf ) );
			delete [ ] buf;
			BPoint cp;
			ulong cb;
			GetMouse( &cp, &cb, FALSE );
			BBitmap *bm;
			switch( s ) {
			case SYSCON_INFO:
				bm = create_info_bitmap( );
				break;
			case SYSCON_WARNING:
				bm = create_warning_bitmap( );
				break;
			case SYSCON_ERROR:
				bm = create_error_bitmap( );
				break;
			default:
				return;
			}
			//DragMessage( msg, BRect( cp.x, cp.y, cp.x+15, cp.y+15 ) );
			DragMessage( msg, bm, BPoint( ICON_W-1, ICON_H-1 ) );
		}
}


//	MsgList::MouseMoved
//------------------------------------------------------------------------------
//	(in case i need it... Nah, it was there for bug tracking :-)
//
void MsgList::MouseMoved( BPoint p, ulong tr, BMessage *msg )
{

}


//	MsgList::GetText
//------------------------------------------------------------------------------
//	Get the text of the selected message. The caller must free the string.
//
char *MsgList::GetText( long *s )
{
	if( selected<0 )
		return NULL;
	int l = 0;
	ml_node *n = ( ml_node * )ml.ItemAt( selected );
	if( s )
		*s = n->s;
	char **cl = n->txt;
	while( *cl )
		l += strlen( *( cl ++ ) )+1;
	char *buf = new char[ l ];
	*buf = '\0';
	cl = n->txt;
	while( *cl ) {
		strcat( buf, *( cl ++ ) );
		strcat( buf, "\n" );
	}
	return buf;
}


//	MsgList::MessageReceived
//------------------------------------------------------------------------------
//	Handles copy requests.
//
void MsgList::MessageReceived( BMessage *msg )
{
	switch( msg->what ) {
	case B_COPY:
		char *buf = GetText( );
		if( !buf )
			break;
		be_clipboard->Lock( );
		be_clipboard->Clear( );
		be_clipboard->AddText( buf );
 		be_clipboard->Commit( );
		be_clipboard->Unlock( );
		delete [ ] buf;
		break;
	default:
		inherited::MessageReceived( msg );
	}
}


//	MsgList::AddMsg
//------------------------------------------------------------------------------
//	Add a message to the list : get a new node, choose an icon, split the
//	string into lines, update the view.
//	Note that the view will scroll only if the scroll bar is at the bottom.
//
void MsgList::AddMsg( long s, char *msg )
{
	BBitmap *ic;
	switch( s ) {
	case SYSCON_INFO:
		ic = info_bitmap;
		break;
	case SYSCON_WARNING:
		ic = warning_bitmap;
		break;
	case SYSCON_ERROR:
		ic = error_bitmap;
		break;
	default:
		return;
	}

	ml_node *n = new ml_node;
	char *c = msg;
	int lc = 1;
	while( *c!='\0' )
		if( *( c ++ )=='\n' )
			lc ++;
	if( c!=msg && *( c-1 )=='\n' )
		lc --;
	n->txt = new char *[ lc+1 ];
	n->txt[ lc ] = NULL;
	c = msg;
	for( int i = 0; i<lc; i ++ ) {
		char *e = strchr( c, '\n' );
		int l;
		if( e )
			l = e-c;
		else
			l = strlen( c );
		n->txt[ i ] = new char[ l+1 ];
		strncpy( n->txt[ i ], c, l );
		n->txt[ i ][ l ] = '\0';
		c = e+1;
	}
	n->icn = ic;
	n->s = s;
	ml.AddItem( n );

	BRect bounds = Bounds( );
	float prev_h = height; 
	RecalcOne( n );
	RecalcScrollBars( );
	if( prev_h==bounds.bottom )
		ScrollBy( 0, height-prev_h );
	Invalidate( );
}



//***************************** a view with a memory gauge * class MemGauge ****
//******************************************************************************


//	class MemGauge
//------------------------------------------------------------------------------
//	A gauge. Thank you GEB for having told me there was a BStatusBar class ;-)
//
class MemGauge : public BView {
public:
	MemGauge( BRect, char *, ulong );
	virtual ~MemGauge( );
	void AttachedToWindow( );
	void Draw( BRect );
	void Pulse( );
private:
	float mem;
	ulong pages;
	char txt[ 16 ];
};


//	MemGauge::MemGauge
//------------------------------------------------------------------------------
//	We must provide the first update by hand.
//
MemGauge::MemGauge( BRect r, char *n, ulong m )
	: BView( r, n, m, B_WILL_DRAW|B_PULSE_NEEDED )
{
	Pulse( );
}


//	MemGauge::~MemGauge
//------------------------------------------------------------------------------
//	not much...
//
MemGauge::~MemGauge( )
{
}


//	MemGauge::AttachedToWindow
//------------------------------------------------------------------------------
//	Can't be done in the ctor.
//
void MemGauge::AttachedToWindow( )
{
	SetViewColor( B_TRANSPARENT_32_BIT );
	SetFontName( "Erich" ); //Kate
}


//	MemGauge::Draw
//------------------------------------------------------------------------------
//	by hand. Line arrays :-)
//
void MemGauge::Draw( BRect upd )
{
static const rgb_color SEP = { 152, 152, 152 };
static const rgb_color HI = { 255, 255, 255 };
static const rgb_color MED = { 216, 216, 216 };
static const rgb_color LOW = { 192, 192, 192 };
static const rgb_color OUTERHI = { 160, 160, 160 };
static const rgb_color OUTHI = { 80, 80, 80 };
static const rgb_color OUTLO = { 128, 128, 128 };
static const rgb_color OUTERLO = { 240, 240, 240 };
static const rgb_color BG = { 160, 160, 160 }; // 255, 255, 255
static const rgb_color BHI = { 102, 203, 255 };
static const rgb_color BMED = { 51, 152, 255 };
static const rgb_color BLOW = { 51, 102, 152 };

	BRect bounds = Bounds( );
	BRect bar( bounds.left+4, bounds.top+5, bounds.right-48, bounds.bottom-4 );
	BRect gau = bar;
	gau.right = gau.left+gau.Width( )*mem;
	gau.bottom += 1;

	SetDrawingMode( B_OP_COPY );
	SetHighColor( MED ); // background
	FillRect( BRect( bounds.left+1, bounds.top+2,
	 bounds.right-1, bounds.bottom-1 ) );


	BeginLineArray( 16 );
	// top separator
	AddLine( BPoint( bounds.left, bounds.top ),
	 BPoint( bounds.right, bounds.top ), SEP );

	// left & top lighted borders
	AddLine( BPoint( bounds.left+1, bounds.top+1 ),
	 BPoint( bounds.right-1, bounds.top+1 ), HI );
	AddLine( BPoint( bounds.left, bounds.top+1 ),
	 BPoint( bounds.left, bounds.bottom-1 ), HI );

	// right & bottom shadowed borders
	AddLine( BPoint( bounds.right, bounds.top+2 ),
	 BPoint( bounds.right, bounds.bottom-1 ), LOW );
	AddLine( BPoint( bounds.left+1, bounds.bottom ),
	 BPoint( bounds.right, bounds.bottom ), LOW );

	AddLine( BPoint( bar.left-2, bar.top-2 ),
	 BPoint( bar.right+2, bar.top-2 ), OUTERHI );
	AddLine( BPoint( bar.left-2, bar.top-1 ),
	 BPoint( bar.left-2, bar.bottom+2 ), OUTERHI );

	AddLine( BPoint( bar.left-1, bar.top-1 ),
	 BPoint( bar.right+1, bar.top-1 ), OUTHI );
	AddLine( BPoint( bar.left-1, bar.top ),
	 BPoint( bar.left-1, bar.bottom+1 ), OUTHI );

	AddLine( BPoint( bar.left, bar.bottom+1 ),
	 BPoint( bar.right+1, bar.bottom+1 ), OUTLO );
	AddLine( BPoint( bar.right+1, bar.top ),
	 BPoint( bar.right+1, bar.bottom ), OUTLO );

	AddLine( BPoint( bar.left-1, bar.bottom+2 ),
	 BPoint( bar.right+2, bar.bottom+2 ), OUTERLO );
	AddLine( BPoint( bar.right+2, bar.top-1 ),
	 BPoint( bar.right+2, bar.bottom+1 ), OUTERLO );
	
	EndLineArray( );

	SetHighColor( BG );
	FillRect( bar );

	SetHighColor( BMED );
	FillRect( BRect( gau.left+1, gau.top+1, gau.right-1, gau.bottom-1 ) );

	BeginLineArray( 16 );

	AddLine( BPoint( gau.left, gau.top ),
	 BPoint( gau.right-1, gau.top ), BHI );
	AddLine( BPoint( gau.left, gau.top+1 ),
	 BPoint( gau.left, gau.bottom-1 ), BHI );

	AddLine( BPoint( gau.left, gau.bottom ),
	 BPoint( gau.right, gau.bottom ), BLOW );
	AddLine( BPoint( gau.right, gau.top ),
	 BPoint( gau.right, gau.bottom-1 ), BLOW );

	EndLineArray( );

	SetHighColor( 0, 0, 0 );
	SetDrawingMode( B_OP_OVER );
	DrawString( txt, BPoint( bounds.right-43, bounds.bottom-2 ) );
}


//	MemGauge::Pulse
//------------------------------------------------------------------------------
//	update the view if necessary (i hate those pulse updated view that redraw
//	even when they don't change the least pixel and happen to blink)
//	Of course i could have splitted Draw() into DrawBackground() and DrawBar(),
//	but... it doesn't blink :-)))
//	No, the right reason is that anyway redrawing the background rectangle is
//	needed because the text is drawn in OP_OVER mode, so in fact only the
//	outer 4 lines are useless when the view is invalidated from here. Big deal.
//
void MemGauge::Pulse( )
{
	system_info si;
	get_system_info( &si );
	ulong prec = pages;
	pages = si.used_pages;
	if( pages!=prec ) {
		sprintf( txt, "%dkB", 4096/1024*si.used_pages );
		mem = ( float )si.used_pages/( float )si.max_pages;
		Invalidate( );
	}
}




//*************************************** The status window * class SWindow ****
//******************************************************************************


//	Private messages
//------------------------------------------------------------------------------
//	.
//
enum {
	CLEAR = 'SCcl',
	SNAP = 'SCsn',
	BACK_TO_SNAPPED = 'SCbs',
	REBOOT = 'SCbo',
};


//	SWindow::SWindow
//------------------------------------------------------------------------------
//	Register our master BLooper, add the menu and the views.
//	wacky SetSizeLimits(), indeed ! note the 228 and the 13 !!!
//
SWindow::SWindow( BMessenger h )
	: BWindow( BRect( 100, 100, 499, 199 ), "System Console 1.0",
	B_DOCUMENT_WINDOW, NULL )
{
	master = h;
	SetDiscipline( TRUE );
	Lock( );
	BRect bounds = Bounds( );

// build the menu
	menu_bar = new BMenuBar( BRect( 0, 0, 0, 0 ), "menu bar" );
	menu_bar->SetBorder( B_BORDER_FRAME );
	BMenu *menu = new BMenu( "Edit" );
	BMenuItem *mi;
	menu->AddItem( mi = new BMenuItem( "Copy", new BMessage( B_COPY ), 'C' ) );
	menu->AddItem( new BMenuItem( "Clear", new BMessage( CLEAR ) ) );
	menu_bar->AddItem( menu );

	menu = new BMenu( "Window" );
	menu->AddItem( new BMenuItem( "Snap window frame",
	 new BMessage( SNAP ), 'S' ) );
	menu->AddItem( new BMenuItem( "Restore snapped frame",
	 new BMessage( BACK_TO_SNAPPED ), 'R' ) );
	menu_bar->AddItem( menu );

	menu = new BMenu( "Special" );
	menu->AddItem( new BMenuItem( "Reboot (quit zoo_keeper)",
	 new BMessage( REBOOT ) ) );
	menu_bar->AddItem( menu );

	AddChild( menu_bar );
	BRect menu_frame = menu_bar->Frame( );
	
	float y = menu_frame.bottom+1.0;
	msg_view = new MsgList( BRect( 0, y, 32, y+32 ),
	 "msg view", B_FOLLOW_ALL_SIDES );


	BScrollView *sv = new BScrollView( "scroll view", msg_view,
	 B_FOLLOW_ALL_SIDES, B_FRAME_EVENTS, TRUE, TRUE, FALSE );

	AddChild( sv );
	msg_view->MakeFocus( );

	mi->SetTarget( msg_view );

	sv->ResizeTo( bounds.right+1.0, bounds.bottom-menu_frame.bottom );
	BScrollBar *sb = sv->ScrollBar( B_HORIZONTAL );
	sb->ResizeBy( -200, 0 );
	sb->MoveBy( 200, 0 );
	
	MemGauge *gauge = new MemGauge(
	 BRect( 0, bounds.bottom+1-sb->Frame( ).Height( ), 199, bounds.bottom ),
	 "gauge", B_FOLLOW_LEFT|B_FOLLOW_BOTTOM );
	AddChild( gauge );

	Unlock( );
	SetPulseRate( 0.5e6 );
	SetSizeLimits( 228, 1e9, y+13, 1e9 );
	SetWorkspaces( B_ALL_WORKSPACES );
}


//	SWindow::MessageReceived
//------------------------------------------------------------------------------
//	handles SYSCON_PRINT_MESSAGE.
//
void SWindow::MessageReceived( BMessage *msg )
{
	switch ( msg->what ) {
	case SYSCON_PRINT_MESSAGE:
		char *l = ( char * )msg->FindString( "message" );
		long status = msg->FindLong( "status" );
		if( msg->Error( )!=B_NO_ERROR )
			return;
		msg_view->AddMsg( status, l );
		break;
	case CLEAR:
		msg_view->Clear( );
		break;
	case SNAP:
		master.SendMessage( SNAP_WINDOW );
		break;
	case BACK_TO_SNAPPED:
		master.SendMessage( BACK_TO_SNAPPED_WINDOW );
		break;
	case REBOOT:
		BMessenger zoo_keeper( 'CERB' );
		zoo_keeper.SendMessage( B_QUIT_REQUESTED );
		break;
	default:
		inherited::MessageReceived( msg );
	}
}


//	SWindow::WindowActivated
//------------------------------------------------------------------------------
//	Spread the (de-)activate word.
//
void SWindow::WindowActivated( bool a )
{
	msg_view->Activated( a );
}


//	SWindow::QuitRequested
//------------------------------------------------------------------------------
//	pretty basic :) we don't want to quit but to be hidden
//
bool SWindow::QuitRequested( )
{
	master.SendMessage( SWITCH_WINDOW );
	return FALSE;
}


//  1996 Benot Triquet
