// ============================================================
//  STEngine.private.cpp	1996 Hiroshi Lockheimer
// ============================================================
// 	STE Version 1.0a4

#include "STEngine.h"
#include <ctype.h>


// ------------------------------------------------------------
// 	 HandleBackspace
// ------------------------------------------------------------
// The Backspace key has been pressed

void
STEngine::HandleBackspace()
{
	if (mSelStart == mSelEnd) {
		if (mSelStart == 0)
			return;
		else 
			mSelStart--;
	}
	else
		DrawSelection(mSelStart, mSelEnd);

	RemoveRange(mSelStart, mSelEnd);
	mSelEnd = mSelStart;
	
	Refresh(mSelStart, mSelEnd, TRUE, TRUE);
	
	HandleModification();
}


// ------------------------------------------------------------
// 	 HandleArrowKey
// ------------------------------------------------------------
// One of the four arrow keys has been pressed

void
STEngine::HandleArrowKey(
	ulong	inArrowKey)
{
	// return if there's nowhere to go
	if (mText.Length() == 0)
		return;
	
	long	selStart = mSelStart;
	long	selEnd = mSelEnd;
	long	scrollToOffset = 0;
	bool	shiftDown = modifiers() & B_SHIFT_KEY;

	switch (inArrowKey) {
		case B_UP_ARROW:
			if ((selStart == selEnd) || (shiftDown)) {
				BPoint point = OffsetToPoint(selStart);
				point.y--;
				selStart = PointToOffset(point);
				if (!shiftDown)
					selEnd = selStart;
				scrollToOffset = selStart;
				break;
			}
			// else fall through
			
		case B_LEFT_ARROW:
			if (shiftDown) {
				if (selStart > 0)
					selStart--;
			}
			else {
				if (selStart == selEnd) {
					if (selStart > 0) 
						selEnd = --selStart;
				}
				else
					selEnd = selStart;
			}
			scrollToOffset = selStart;
			break;
			
		case B_DOWN_ARROW:
			if ((selStart == selEnd) || (shiftDown)) {
				float	height;
				BPoint	point = OffsetToPoint(selEnd, &height);
				point.y += height;
				selEnd = PointToOffset(point);
				if (!shiftDown)
					selStart = selEnd;
				scrollToOffset = selEnd;
				break;
			}
			// else fall through
			
		case B_RIGHT_ARROW:
			if (shiftDown) {
				if (selEnd < mText.Length())
					selEnd++;
			}
			else {
				if (selStart == selEnd) {
					if (selStart < mText.Length())
						selStart = ++selEnd;
				}
				else
					selStart = selEnd;
			}
			scrollToOffset = selEnd;
			break;
	}
	
	// invalidate the null style
	mStyles.InvalidateNullStyle();
	
	Select(selStart, selEnd);
	
	// scroll if needed
	ScrollToOffset(scrollToOffset);
}


// ------------------------------------------------------------
// 	 HandleDelete
// ------------------------------------------------------------
// The Delete key has been pressed

void
STEngine::HandleDelete()
{
	if (mSelStart == mSelEnd) {
		if (mSelEnd == mText.Length())
			return;
		else 
			mSelEnd++;
	}
	else
		DrawSelection(mSelStart, mSelEnd);
		
	RemoveRange(mSelStart, mSelEnd);
	
	mSelEnd = mSelStart;
	
	Refresh(mSelStart, mSelEnd, TRUE, TRUE);
	
	HandleModification();
}


// ------------------------------------------------------------
// 	 HandlePageKey
// ------------------------------------------------------------
// Home, End, Page Up, or Page Down has been pressed

void
STEngine::HandlePageKey(
	ulong	inPageKey)
{		
	switch (inPageKey) {
		case B_HOME:
		case B_END:
			ScrollToOffset((inPageKey == B_HOME) ? 0 : mText.Length());
			break;
			
		case B_PAGE_UP: 
		case B_PAGE_DOWN:
		{
			BScrollBar *vScroll = ScrollBar(B_VERTICAL);
			if (vScroll != NULL) {
				float delta = Bounds().Height();
				delta = (inPageKey == B_PAGE_UP) ? -delta : delta;
				
				vScroll->SetValue(vScroll->Value() + delta);
				Window()->UpdateIfNeeded();
			}
			break;
		}
	}
}


// ------------------------------------------------------------
// 	 HandleAlphaKey
// ------------------------------------------------------------
// A printing key has been pressed

void
STEngine::HandleAlphaKey(
	uchar	inAlphaKey)
{
	bool refresh = mSelStart != mText.Length();
	
	if (mSelStart != mSelEnd) {
		DrawSelection(mSelStart, mSelEnd);
		RemoveRange(mSelStart, mSelEnd);
		refresh = TRUE;
	}
	InsertAt((char *)&inAlphaKey, 1, mSelStart);
	
	mSelEnd = ++mSelStart;

	Refresh(mSelStart, mSelEnd, refresh, TRUE);
	
	HandleModification();
}


// ------------------------------------------------------------
// 	 InsertAt
// ------------------------------------------------------------
// Copy inLength bytes of inText to the buffer, starting at offset
//
// Optionally apply inStyles to the newly inserted text

void
STEngine::InsertAt(
	const char				*inText,
	long 					inLength,
	long 					inOffset,
	ConstSTEStyleRangePtr	inStyles)
{
	// why add nothing?
	if (inLength < 1)
		return;
	
	// add the text to the buffer
	mText.InsertText(inText, inLength, inOffset);
	
	// update the start offsets of each line below inOffset
	mLines.BumpOffset(inLength, OffsetToLine(inOffset) + 1);
	
	// update the style runs
	mStyles.BumpOffset(inLength, mStyles.OffsetToRun(inOffset - 1) + 1);
	
	if (inStyles != NULL)
		SetStyleRange(inOffset, inOffset + inLength, inStyles, FALSE);
	else {
		// apply nullStyle to inserted text
		mStyles.SyncNullStyle(inOffset);
		mStyles.SetStyleRange(inOffset, inOffset + inLength, 
							  mText.Length(), doAll, NULL, this);
	}
}


// ------------------------------------------------------------
// 	 RemoveRange
// ------------------------------------------------------------
// Remove data that lies between fromOffset and toOffset

void
STEngine::RemoveRange(
	long	fromOffset,
	long 	toOffset)
{
	// sanity checking
	if ((fromOffset >= toOffset) || (fromOffset < 0) || (toOffset < 0))
		return;
		
	// set nullStyle to style at beginning of range
	mStyles.InvalidateNullStyle();
	mStyles.SyncNullStyle(fromOffset);	
	
	// remove from the text buffer
	mText.RemoveRange(fromOffset, toOffset);
	
	// remove any lines that have been obliterated
	mLines.RemoveLineRange(fromOffset, toOffset);
	
	// remove any style runs that have been obliterated
	mStyles.RemoveStyleRange(fromOffset, toOffset);
}


// ------------------------------------------------------------
// 	 Refresh
// ------------------------------------------------------------
// Recalculate the line breaks from fromOffset to toOffset
// and redraw the text with the new line breaks
//
// If erase is true, the affected text area will be erased  
// before the text is drawn
//
// If scroll is true, the view will be scrolled so that
// the end of the selection is visible

void
STEngine::Refresh(
	long	fromOffset,
	long	toOffset,
	bool	erase,
	bool	scroll)
{
	float	saveHeight = mTextRect.Height();
	long 	fromLine = OffsetToLine(fromOffset);
	long 	toLine = OffsetToLine(toOffset);
	long	saveFromLine = fromLine;
	long	saveToLine = toLine;
	float	saveLineHeight = GetHeight(fromLine, fromLine);
	
	RecalLineBreaks(&fromLine, &toLine);

	float newHeight = mTextRect.Height();
	
	// if the line breaks have changed, force an erase
	if ( (fromLine != saveFromLine) || (toLine != saveToLine) || 
		 (newHeight != saveHeight) )
		erase = TRUE;
	
	if (newHeight != saveHeight) {
		// the text area has changed
		if (newHeight < saveHeight)
			toLine = PixelToLine(saveHeight + mTextRect.top);
		else
			toLine = PixelToLine(newHeight + mTextRect.top);
	}
	
	long drawOffset = fromOffset;
	if ( (GetHeight(fromLine, fromLine) != saveLineHeight) || 
		 (newHeight < saveHeight) || (fromLine < saveFromLine) )
		drawOffset = mLines[fromLine]->offset;
			
	DrawLines(fromLine, toLine, drawOffset, erase);
	
	// erase the area below the text
	BRect bounds = Bounds();
	BRect eraseRect = bounds;
	eraseRect.top = mTextRect.top + mLines[mLines.NumLines()]->origin;
	eraseRect.bottom = mTextRect.top + saveHeight;
	if ((eraseRect.bottom > eraseRect.top) && (eraseRect.Intersects(bounds))) {
		SetLowColor(ViewColor());
		FillRect(eraseRect, B_SOLID_LOW);
	}
	
	// update the scroll bars if the text area has changed
	if (newHeight != saveHeight)
		UpdateScrollbars();

	if (scroll)
		ScrollToOffset(mSelEnd);
}


// ------------------------------------------------------------
// 	 RecalLineBreaks
// ------------------------------------------------------------
// Recalculate the line breaks starting at startLine
// Recalculate at least up to endLine
//
// Pass back the range of affected lines in startLine and endLine

void
STEngine::RecalLineBreaks(
	long	*startLine,
	long	*endLine)
{
	// are we insane?
	*startLine = (*startLine < 0) ? 0 : *startLine;
	*endLine = (*endLine > mLines.NumLines() - 1) ? mLines.NumLines() - 1 : *endLine;
	
	long	textLength = mText.Length();
	long	lineIndex = (*startLine > 0) ? *startLine - 1 : 0;
	long	recalThreshold = mLines[*endLine + 1]->offset;
	float	width = mTextRect.Width();
	
	// cast away the const-ness
	STELinePtr curLine = (STELinePtr)mLines[lineIndex];
	STELinePtr nextLine = (STELinePtr)curLine + 1;

	do {
		long 	fromOffset = curLine->offset;
		float	ascent = 0.0;
		float	descent = 0.0;
		long 	toOffset = FindLineBreak(fromOffset, &ascent, 
										 &descent, width);

		// we want to advance at least by one character
		if ((toOffset == fromOffset) && (fromOffset < textLength))
			toOffset++;
		
		// set the ascent of this line
		curLine->ascent = ascent;
		
		lineIndex++;
		STELine saveLine = *nextLine;		
		if ( (lineIndex > mLines.NumLines()) || 
			 (toOffset < nextLine->offset) ) {
			// the new line comes before the old line start, add a line
			STELine newLine;
			newLine.offset = toOffset;
			newLine.origin = curLine->origin + ascent + descent;
			newLine.ascent = 0.0;
			mLines.InsertLine(&newLine, lineIndex);
		}
		else {
			// update the exising line
			nextLine->offset = toOffset;
			nextLine->origin = curLine->origin + ascent + descent;
			
			// remove any lines that start before the current line
			while ( (lineIndex < mLines.NumLines()) &&
					(toOffset >= (mLines[lineIndex] + 1)->offset) )
				mLines.RemoveLines(lineIndex + 1);
			
			nextLine = (STELinePtr)mLines[lineIndex];
			if (nextLine->offset == saveLine.offset) {
				if (nextLine->offset >= recalThreshold) {
					if (nextLine->origin != saveLine.origin)
						mLines.BumpOrigin(nextLine->origin - saveLine.origin, 
										  lineIndex + 1);
					break;
				}
			}
			else {
				if ((lineIndex > 0) && (lineIndex == *startLine))
					*startLine = lineIndex - 1;
			}
		}

		curLine = (STELinePtr)mLines[lineIndex];
		nextLine = (STELinePtr)curLine + 1;
	} while (curLine->offset < textLength);

	// update the text rect
	float newHeight = GetHeight(0, mLines.NumLines() - 1);
	mTextRect.bottom = mTextRect.top + newHeight;

	*endLine = lineIndex - 1;
	*startLine = (*startLine > *endLine) ? *endLine : *startLine;
}


// ------------------------------------------------------------
// 	 FindLineBreak
// ------------------------------------------------------------
// Determine where to break a line that is ioWidth wide, 
// starting at fromOffset
//
// Pass back the maximum ascent and descent for the line in
// outAscent and outDescent
// Set ioWidth to the total width of the line

long
STEngine::FindLineBreak(
	long	fromOffset,
	float	*outAscent,
	float	*outDescent,
	float	width)
{
	*outAscent = 0.0;
	*outDescent = 0.0;
	
	const long limit = mText.Length();

	// is fromOffset at the end?
	if (fromOffset >= limit) {
		// try to return valid height info anyway			
		if (mStyles.NumRuns() > 0)
			mStyles.Iterate(fromOffset, 1, NULL, outAscent, outDescent);
		else {
			if (mStyles.IsValidNullStyle()) {
				ConstSTEStylePtr style = &mStyles.GetNullStyle();
				SetFontName(style->font);
				SetFontSize(style->size);
				SetFontShear(style->shear);
				
				font_info fInfo;
				GetFontInfo(&fInfo);
				*outAscent = fInfo.ascent;
				*outDescent = fInfo.descent + fInfo.leading;
			}
		}

		return (limit);
	}
	
	bool	done = FALSE;
	float	ascent = 0.0;
	float	descent = 0.0;
	long	offset = fromOffset;
	long	delta = 0;
	float	deltaWidth = 0.0;
	float	tabWidth = 0.0;
	float	strWidth = 0.0;
	
	// maybe we don't need to wrap the text?
	if (!mWrap) {
		long length = limit - fromOffset;
		mText.FindChar('\n', fromOffset, &length);
		offset = fromOffset + (++length);
		offset = (offset > limit) ? limit : offset;

		// iterate through the style runs for the heights
		while (long numChars = mStyles.Iterate(fromOffset, length, NULL, &ascent, &descent)) {
			*outAscent = (ascent > *outAscent) ? ascent : *outAscent;
			*outDescent = (descent > *outDescent) ? descent : *outDescent;
			
			fromOffset += numChars;
			length -= numChars;
		}

		return (offset);
	}
	
	// wrap the text
	do {
		bool foundTab = FALSE;
		
		// find the next line break candidate
		for ( ; (offset + delta) < limit ; delta++) {
			if (IsLineBreakChar(mText[offset + delta]))
				break;
		}
		for ( ; (offset + delta) < limit; delta++) {
			uchar theChar = mText[offset + delta];
			if (!IsLineBreakChar(theChar))
				break;
			
			if (theChar == '\n') {
				// found a newline, we're done!
				done = TRUE;
				delta++;
				break;
			}
			else {
				// include all trailing spaces and tabs,
				// but not spaces after tabs
				if ((theChar != ' ') && (theChar != '\t'))
					break;
				else {
					if ((theChar == ' ') && (foundTab))
						break;
					else {
						if (theChar == '\t')
							foundTab = TRUE;
					}
				}
			}
		}
		delta = (delta < 1) ? 1 : delta;
	
		deltaWidth = StyledWidth(offset, delta, &ascent, &descent);
		strWidth += deltaWidth;

		if (!foundTab)
			tabWidth = 0.0;
		else {
			long tabCount = 0;
			for (long i = delta - 1; mText[offset + i] == '\t'; i--)
				tabCount++;

			tabWidth = ActualTabWidth(strWidth);
			if (tabCount > 1)
				tabWidth += ((tabCount - 1) * mTabWidth);

			strWidth += tabWidth;
		}
			
		if (strWidth >= width) {
			// we've found where the line will wrap
			bool foundNewline = done;
			done = TRUE;
			long pos = delta - 1;
			if ((mText[offset + pos] != ' ') &&
				(mText[offset + pos] != '\t') &&
				(mText[offset + pos] != '\n'))
				break;
			
			strWidth -= (deltaWidth + tabWidth);
			
			for ( ; ((offset + pos) > offset); pos--) {
				uchar theChar = mText[offset + pos];
				if ((theChar != ' ') &&
					(theChar != '\t') &&
					(theChar != '\n'))
					break;
			}

			strWidth += StyledWidth(offset, pos + 1, &ascent, &descent);
			if (strWidth >= width)
				break;

			if (!foundNewline) {
				for ( ; (offset + delta) < limit; delta++) {
					if ((mText[offset + delta] != ' ') &&
						(mText[offset + delta] != '\t'))
						break;
				}
				if ( ((offset + delta) < limit) && 
					 (mText[offset + delta] == '\n') )
					delta++;
			}
			// get the ascent and descent of the spaces/tabs
			StyledWidth(offset, delta, &ascent, &descent);
		}
		
		*outAscent = (ascent > *outAscent) ? ascent : *outAscent;
		*outDescent = (descent > *outDescent) ? descent : *outDescent;
		
		offset += delta;
		delta = 0;
	} while ((offset < limit) && (!done));
	
	if ((offset - fromOffset) < 1) {
		// there weren't any words that fit entirely in this line
		// force a break in the middle of a word
		*outAscent = 0.0;
		*outDescent = 0.0;
		strWidth = 0.0;
		
		for (offset = fromOffset; offset < limit; offset++) {
			strWidth += StyledWidth(offset, 1, &ascent, &descent);
			
			if (strWidth >= width)
				break;
				
			*outAscent = (ascent > *outAscent) ? ascent : *outAscent;
			*outDescent = (descent > *outDescent) ? descent : *outDescent;
		}
	}
	
	offset = (offset < limit) ? offset : limit;

	return (offset);
}


// ------------------------------------------------------------
// 	 IsWordBreakChar
// ------------------------------------------------------------
// Return true if inChar will break a word

bool
STEngine::IsWordBreakChar(
	uchar	inChar)
{
	if ((ispunct(inChar)) || (isspace(inChar)))
		return (TRUE);
		
	return (FALSE);
}


// ------------------------------------------------------------
// 	 IsLineBreakChar
// ------------------------------------------------------------
// Return true if inChar will break a line

bool
STEngine::IsLineBreakChar(
	uchar	inChar)
{
	switch (inChar) {
		case '\0':
		case '\t':
		case '\n':
		case ' ':
		case '&':
		case '*':
		case '+':
		case '-':
		case '/':
		case '<':
		case '=':
		case '>':
		case '\\':
		case '^':
		case '|':
			return (TRUE);
		
		default:
			return (FALSE);
	}
	
	return (FALSE);
}


// ------------------------------------------------------------
// 	 StyledWidth
// ------------------------------------------------------------
// Return the width of length bytes of styled text beginning at
// fromOffset
//
// Pass back the maximum ascent and maximum descent of the text
// in outAscent and outDescent
//
// Pass NULL for outAscent and/or outDescent if you are not
// interested in that data
//
// Tab-widths are not calculated, use OffsetToPoint() if you need
// tab-inclusive widths

float
STEngine::StyledWidth(
	long	fromOffset,
	long 	length,
	float	*outAscent,
	float	*outDescent)
{
	float result = 0.0;
	float ascent = 0.0;
	float descent = 0.0;
	float maxAscent = 0.0;
	float maxDescent = 0.0;
	
	// iterate through the style runs
	ConstSTEStylePtr style = NULL;
	while (long numChars = mStyles.Iterate(fromOffset, length, &style, &ascent, &descent)) {		
		maxAscent = (ascent > maxAscent) ? ascent : maxAscent;
		maxDescent = (descent > maxDescent) ? descent : maxDescent;

		result += sWidths.StringWidth(mText, fromOffset, numChars, style, this);

		fromOffset += numChars;
		length -= numChars;
	}

	if (outAscent != NULL)
		*outAscent = maxAscent;
	if (outDescent != NULL)
		*outDescent = maxDescent;

	return (result);
}


// ------------------------------------------------------------
// 	 ActualTabWidth
// ------------------------------------------------------------
// Return the actual tab width at location 
// 
// location should be in text rect coordinates

float
STEngine::ActualTabWidth(
	float	location)
{
	if (mTabWidth <= 0.0)
		return (0.0);
		
	return ( mTabWidth - 
			 (location - ((long)(location / mTabWidth)) * mTabWidth) );
}


// ------------------------------------------------------------
// 	 DrawLines
// ------------------------------------------------------------
// Draw the lines from startLine to endLine
// Erase the affected area before drawing if erase is true
//
// startOffset gives the offset of the first character in startLine
// that needs to be erased (avoids flickering of the entire line)
//
// Pass a value of -1 in startOffset if you want the entire 
// line of startLine to be erased 
 
void
STEngine::DrawLines(
	long	startLine,
	long	endLine,
	long	startOffset,
	bool	erase)
{	
	BRect bounds = Bounds();
	
	// clip the text	
	BRect clipRect = bounds & mTextRect;
	clipRect.InsetBy(-1.0, -1.0);
	BRegion newClip;
	newClip.Set(clipRect);
	ConstrainClippingRegion(&newClip);

	// set the low color to the view color so that 
	// drawing to a non-white background will work	
	rgb_color viewColor = ViewColor();
	SetLowColor(viewColor);

	// draw only those lines that are visible
	long startVisible = PixelToLine(bounds.top);
	long endVisible = PixelToLine(bounds.bottom);
	startLine = (startLine < startVisible) ? startVisible : startLine;
	endLine = (endLine > endVisible) ? endVisible : endLine;

	BRect 			eraseRect = clipRect;
	long			startEraseLine = startLine;
	ConstSTELinePtr	line = mLines[startLine];

	if ((mOffscreen == NULL) && (erase) && (startOffset != -1)) {
		// erase only portion of first line
		startEraseLine++;
			
		long startErase = startOffset;
		if (startErase > line->offset) {
			for ( ; (mText[startErase] != ' ') && (mText[startErase] != '\t'); startErase--) {
				if (startErase <= line->offset)
					break;	
			}
			if (startErase > line->offset)
				startErase--;
		}

		eraseRect.left = OffsetToPoint(startErase).x;
		eraseRect.top = line->origin + mTextRect.top;
		eraseRect.bottom = (line + 1)->origin + mTextRect.top;
		
		FillRect(eraseRect, B_SOLID_LOW);

		eraseRect = clipRect;
	}
	
	for (long i = startLine; i <= endLine; i++) {
		BView	*drawView = this;
		float	penDelta = 0.0;		
		BPoint	penLoc(0.0, 0.0);
		long 	length = (line + 1)->offset - line->offset;
		
		// DrawString() chokes if you draw a newline
		if (mText[(line + 1)->offset - 1] == '\n')
			length--;	
					
		if ((mOffscreen == NULL) || (i > startLine) || (startOffset == -1)) {						
			if ((erase) && (i >= startEraseLine)) {
				eraseRect.top = line->origin + mTextRect.top;
				eraseRect.bottom = (line + 1)->origin + mTextRect.top;
				
				FillRect(eraseRect, B_SOLID_LOW);
			}
			
			penDelta = mTextRect.left;
			penLoc.Set(penDelta, line->origin + line->ascent + mTextRect.top);
			MovePenTo(penLoc);			
		}
		else {
			startEraseLine++;
			
			mOffscreen->Lock();
			drawView = mOffscreen->ChildAt(0);
			
			BRect	offBounds = drawView->Bounds();
			float	lineHeight = (line + 1)->origin - line->origin;
			if (offBounds.Height() < lineHeight) {
				// bitmap isn't tall enough for the current line, resize
				color_space colors = mOffscreen->ColorSpace();
				mOffscreen->Unlock();
				delete (mOffscreen);

				offBounds.bottom = lineHeight;
				mOffscreen = new BBitmap(offBounds, colors, TRUE);
				drawView = new BView(offBounds, B_EMPTY_STRING, B_FOLLOW_NONE, 0);
				mOffscreen->AddChild(drawView);
				
				mOffscreen->Lock();
			}
			
			drawView->SetLowColor(viewColor);
			drawView->FillRect(offBounds, B_SOLID_LOW);
			
			penDelta = 0.0;
			penLoc.Set(penDelta, line->ascent);
			drawView->MovePenTo(penLoc);
		}
		
		// do we have any text to draw?
		if (length > 0) {
			// iterate through each style on this line
			BPoint				startPenLoc = penLoc;
			bool 				foundTab = FALSE;
			long				tabChars = 0;
			long 				offset = line->offset;
			ConstSTEStylePtr	style = NULL;
			while (long numChars = mStyles.Iterate(offset, length, &style)) {
				drawView->SetFontName(style->font);
				drawView->SetFontSize(style->size);
				drawView->SetFontShear(style->shear);
				drawView->SetHighColor(style->color);
				
				tabChars = numChars;
				do {
					startPenLoc = penLoc;

					foundTab = mText.FindChar('\t', offset, &tabChars);
						
					drawView->DrawString(mText.GetString(offset, tabChars), tabChars);
					penLoc.x += sWidths.StringWidth(mText, offset, tabChars, style, this);
					
					if (foundTab) {
						long numTabs = 0;
						for (numTabs = 0; (tabChars + numTabs) < numChars; numTabs++) {
							if (mText[offset + tabChars + numTabs] != '\t')
								break;
						}
												
						float tabWidth = ActualTabWidth(penLoc.x - penDelta);
						if (numTabs > 1)
							tabWidth += ((numTabs - 1) * mTabWidth);

						penLoc.x += tabWidth;
						drawView->MovePenTo(penLoc);
						
						tabChars += numTabs;
					}
					
					if (style->underline) {
						BPoint curPenLoc = penLoc;
						startPenLoc.y += 1.0;
						curPenLoc.y += 1.0;
						
						drawView->StrokeLine(startPenLoc, curPenLoc);
						
						drawView->MovePenTo(penLoc);
					}

					offset += tabChars;
					length -= tabChars;
					numChars -= tabChars;
					tabChars = numChars;
				} while ((foundTab) && (tabChars > 0));
			}
		}
		
		if (drawView != this) {
			BRect 	offBounds = mOffscreen->Bounds();
			float	lineHeight = (line + 1)->origin - line->origin;
			
			BRect srcRect = offBounds;
			srcRect.bottom = srcRect.top + lineHeight;
			
			BRect dstRect;
			dstRect.left = mTextRect.left;
			dstRect.top = line->origin + mTextRect.top;
			dstRect.right = dstRect.left + offBounds.Width();
			dstRect.bottom = dstRect.top + lineHeight;

			drawView->Sync();
			DrawBitmap(mOffscreen, srcRect, dstRect);
			
			mOffscreen->Unlock();
		}
		
		line++;
	}

	ConstrainClippingRegion(NULL);
}


// ------------------------------------------------------------
// 	 DrawSelection
// ------------------------------------------------------------
// Hilite the characters between startOffset and endOffset
//
// BeOS doesn't have a 'hilite' mode, so we invert instead
// Be Engineers: Please?

void
STEngine::DrawSelection(
	long	startOffset,
	long	endOffset)
{		
	// get real
	if (startOffset >= endOffset)
		return;
		
	BRegion selRegion;
	GetHiliteRegion(startOffset, endOffset, &selRegion);

	ConstrainClippingRegion(&selRegion);
	InvertRect(selRegion.Frame());
	ConstrainClippingRegion(NULL);
}


// ------------------------------------------------------------
// 	 DrawCaret
// ------------------------------------------------------------
// Draw the caret at offset

void
STEngine::DrawCaret(
	long	offset)
{
	float	lineHeight = 0.0;
	BPoint	caretPoint = OffsetToPoint(offset, &lineHeight);
	caretPoint.x = (caretPoint.x > mTextRect.right) ? mTextRect.right : caretPoint.x;
	
	BRect caretRect;
	caretRect.left = caretRect.right = caretPoint.x;
	caretRect.top = caretPoint.y;
	caretRect.bottom = caretPoint.y + lineHeight;
	
	InvertRect(caretRect);
}


// ------------------------------------------------------------
// 	 InvertCaret
// ------------------------------------------------------------
// Invert the caret at mSelStart

void
STEngine::InvertCaret()
{
	DrawCaret(mSelStart);
	mCaretVisible = !mCaretVisible;
	mCaretTime = system_time();
}


// ------------------------------------------------------------
// 	 DragCaret
// ------------------------------------------------------------
// Draw a temporary caret at offset 

void
STEngine::DragCaret(
	long	offset)
{
	// does the caret need to move?
	if (offset == mDragOffset)
		return;
	
	// hide the previous drag caret
	if (mDragOffset != -1)
		DrawCaret(mDragOffset);
		
	// do we have a new location?
	if (offset != -1) {
		if (mActive) {
			// ignore if offset is within active selection
			if ((offset >= mSelStart) && (offset <= mSelEnd)) {
				mDragOffset = -1;
				return;
			}
		}
		
		DrawCaret(offset);
	}
	
	mDragOffset = offset;
}


// ------------------------------------------------------------
// 	 TrackDrag
// ------------------------------------------------------------
// Track and give feedback for a drag
//
// This function gets called repeatedly while there is a drag
// that needs to be tracked
//
// When mDragOwner is TRUE, this function will get called
// even when the drag is not above this view
//
// When this view is the drag owner (mDragOwner), it is assumed
// that it can handle its own drag (CanDrop() is not consulted)

void
STEngine::TrackDrag(
	BPoint	where)
{
	BRect bounds = Bounds();

	// are we the drag owner?
	if (!mDragOwner) {
		// drag isn't ours, don't worry about auto-scrolling
		if (bounds.Contains(where))
			DragCaret(PointToOffset(where));
		return;
	}
	
	// expand the bounds
	bounds.InsetBy(-B_V_SCROLL_BAR_WIDTH, -B_H_SCROLL_BAR_HEIGHT);
	
	// is the mouse within the expanded bounds?
	if (bounds.Contains(where)) {
		long hDelta = 0;
		long vDelta = 0;
		
		BScrollBar *hScroll = ScrollBar(B_HORIZONTAL);
		if (hScroll != NULL) {
			// left edge
			if (where.x < (bounds.left + B_V_SCROLL_BAR_WIDTH)) {
				hDelta = where.x - (bounds.left + B_V_SCROLL_BAR_WIDTH);
			}
			else {
				// right edge
				if (where.x > (bounds.right - B_V_SCROLL_BAR_WIDTH))
					hDelta = where.x - (bounds.right - B_V_SCROLL_BAR_WIDTH);
			}
			
			if (hDelta != 0) {
				DragCaret(-1);
				hScroll->SetValue(hScroll->Value() + (hDelta * 5));
			}
		}
		
		BScrollBar *vScroll = ScrollBar(B_VERTICAL);
		if (vScroll != NULL) {	
			// top edge
			if (where.y < (bounds.top + B_H_SCROLL_BAR_HEIGHT)) {
				vDelta = where.y - (bounds.top + B_H_SCROLL_BAR_HEIGHT);
			}
			else {
				// bottom edge
				if (where.y > (bounds.bottom - B_H_SCROLL_BAR_HEIGHT))
					vDelta = where.y - (bounds.bottom - B_H_SCROLL_BAR_HEIGHT);
			}
			
			if (vDelta != 0) {
				DragCaret(-1);
				vScroll->SetValue(vScroll->Value() + (vDelta * 5));
			}
		}
		
		if ((hDelta != 0) || (vDelta != 0))
			Window()->UpdateIfNeeded();
		else
			DragCaret(PointToOffset(where));
	}
}


// ------------------------------------------------------------
// 	 InitiateDrag
// ------------------------------------------------------------
// Create a new drag message and pass it on to the app_server

void
STEngine::InitiateDrag()
{	
	// create a new drag message
	BMessage *drag = new BMessage(B_SIMPLE_DATA);
	
	// add the text
	drag->AddData("text", B_ASCII_TYPE, mText.Text() + mSelStart, 
			  	  mSelEnd - mSelStart);
	
	// get the corresponding styles
	long 				styleLen = 0;
	STEStyleRangePtr	styles = GetStyleRange(mSelStart, mSelEnd, &styleLen);
	
	// reset the extra fields, override if you don't like this behavior
	for (long i = 0; i < styles->count; i++)
		styles->runs[i].style.extra = 0;
	
	drag->AddData("style", STE_STYLE_TYPE, styles, styleLen);

	BRegion hiliteRgn;
	GetHiliteRegion(mSelStart, mSelEnd, &hiliteRgn);
	BRect dragRect = hiliteRgn.Frame();
	BRect bounds = Bounds();	
	if (!bounds.Contains(dragRect))
		dragRect = bounds & dragRect;
		
	be_app->SetCursor(B_HAND_CURSOR);
	DragMessage(drag, dragRect);
	
	// we're a proud owner of a drag
	mDragOwner = TRUE;
	Window()->SetPulseRate(100000.0);
}


// ------------------------------------------------------------
// 	 MessageDropped
// ------------------------------------------------------------
// Respond to dropped messages

bool
STEngine::MessageDropped(
	BMessage	*inMessage,
	BPoint 		where,
	BPoint		offset)
{
#pragma unused(offset)
		
	if (mDragOwner) {
		// our drag has come to an end
		mDragOwner = FALSE;
		Window()->SetPulseRate(500000.0);
	}
	
	// make sure the drag caret is erased
	DragCaret(-1);

	if (mActive)
		be_app->SetCursor(B_I_BEAM_CURSOR);
		
	// are we sure we like this message?
	if (!CanDrop(inMessage))
		return (FALSE);
		
	long dropOffset = PointToOffset(where);
	
	// if this was our drag, move instead of copy
	if ((mActive) && (mSelStart != mSelEnd)) {
		// dropping onto itself?
		if ((dropOffset >= mSelStart) && (dropOffset <= mSelEnd))
			return (TRUE);
			
		// adjust the offset if the drop is after the selection
		if (dropOffset > mSelEnd)
			dropOffset -= (mSelEnd - mSelStart);
			
		// delete the selection
		DrawSelection(mSelStart, mSelEnd);
		RemoveRange(mSelStart, mSelEnd);
		mSelEnd = mSelStart;
		Refresh(mSelStart, mSelEnd, TRUE, FALSE);
	}
		
	Select(dropOffset, dropOffset);
	
	long dataLen = 0;
	char *text = (char *)inMessage->FindData("text", B_ASCII_TYPE, &dataLen);
	if (text != NULL) {
		long 				styleLen = 0;
		STEStyleRangePtr 	styles = NULL;
		styles = (STEStyleRangePtr)inMessage->FindData("style", STE_STYLE_TYPE, &styleLen);
	
		Insert(text, dataLen, styles);	
	}
	else {
		char theChar = inMessage->FindLong("char");
		if (inMessage->Error() == B_NO_ERROR)
			Insert(&theChar, 1);
	}
		
	return (TRUE);
}


// ------------------------------------------------------------
// 	 UpdateScrollbars
// ------------------------------------------------------------
// Adjust the scroll bars so that they reflect the current 
// visible size and position of the text area

void
STEngine::UpdateScrollbars()
{
	BRect bounds = Bounds();
	
	// do we have a horizontal scroll bar?
	BScrollBar *hScroll = ScrollBar(B_HORIZONTAL);
	if (hScroll != NULL) {
		long viewWidth = bounds.IntegerWidth();
		long dataWidth = mTextRect.IntegerWidth();
		dataWidth += (ceil(mTextRect.left) + 1);
		
		long maxRange = dataWidth - viewWidth;
		maxRange = (maxRange < 0) ? 0 : maxRange;
		
		hScroll->SetRange(0, maxRange);
		hScroll->SetSteps(10, dataWidth / 10);
	}
	
	// how about a vertical scroll bar?
	BScrollBar *vScroll = ScrollBar(B_VERTICAL);
	if (vScroll != NULL) {
		long viewHeight = bounds.IntegerHeight();
		long dataHeight = mTextRect.IntegerHeight();
		dataHeight += (ceil(mTextRect.top) + 1);
		
		long maxRange = dataHeight - viewHeight;
		maxRange = (maxRange < 0) ? 0 : maxRange;
		
		vScroll->SetRange(0, maxRange);
		vScroll->SetSteps(12, viewHeight);
	}
}


// ------------------------------------------------------------
// 	 Activate
// ------------------------------------------------------------
// Activate the text area
//
// Draw the caret/hilite the selection, set the cursor

void
STEngine::Activate()
{
	mActive = TRUE;
	
	if (mSelStart != mSelEnd) {
		if (mSelectable)
			DrawSelection(mSelStart, mSelEnd);
	}
	else {
		if (mEditable)
			InvertCaret();
	}
	
	BPoint 	where;
	ulong	buttons;
	GetMouse(&where, &buttons);
	if (Bounds().Contains(where))
		be_app->SetCursor(B_I_BEAM_CURSOR);
}


// ------------------------------------------------------------
// 	 Deactivate
// ------------------------------------------------------------
// Deactivate the text area
//
// Hide the caret/unhilite the selection, set the cursor

void
STEngine::Deactivate()
{
	mActive = FALSE;
	
	if (mSelStart != mSelEnd) {
		if (mSelectable)
			DrawSelection(mSelStart, mSelEnd);
	}
	else {
		if (mCaretVisible)
			InvertCaret();
	}
	
	BPoint 	where;
	ulong	buttons;
	GetMouse(&where, &buttons);
	if (Bounds().Contains(where))
		be_app->SetCursor(B_HAND_CURSOR);
}


// ------------------------------------------------------------
// 	 HandleModification
// ------------------------------------------------------------
// This function gets called when the text or a style run
// has been modified
//
// The default implementation of this function is empty, override
// if you wish to do something useful

void
STEngine::HandleModification()
{
}


