// ============================================================
//  STESupport.cpp	1996 Hiroshi Lockheimer
// ============================================================
// 	STE Version 1.0a4

#include "STESupport.h"


STETextBuffer::STETextBuffer()
{
	mExtraCount = 2048;
	mItemCount = 0;
	mBuffer = (char *)malloc(mExtraCount + mItemCount);
	mBufferCount = mExtraCount + mItemCount;
	mGapIndex = mItemCount;
	mGapCount = mBufferCount - mGapIndex;
	mScratchBuffer = (char *)malloc(0);
	mScratchSize = 0;
}

STETextBuffer::~STETextBuffer()
{
	free(mBuffer);
	free(mScratchBuffer);
}

void
STETextBuffer::InsertText(
	const char	*inText,
	long 		inNumItems,
	long 		inAtIndex)
{
	if (inNumItems < 1)
		return;
	
	inAtIndex = (inAtIndex > mItemCount) ? mItemCount : inAtIndex;
	inAtIndex = (inAtIndex < 0) ? 0 : inAtIndex;
		
	if (mGapIndex != inAtIndex) 
		MoveGapTo(inAtIndex);
	
	if (mGapCount < inNumItems)
		SizeGapTo(inNumItems + mExtraCount);
		
	memcpy(mBuffer + mGapIndex, inText, inNumItems);
	
	mGapCount -= inNumItems;
	mGapIndex += inNumItems;
	mItemCount += inNumItems;
}

void
STETextBuffer::RemoveRange(
	long	start,
	long 	end)
{ 
	long atIndex = start;
	atIndex = (atIndex > mItemCount) ? mItemCount : atIndex;
	atIndex = (atIndex < 0) ? 0 : atIndex;
	
	long numItems = end - atIndex;
	numItems = (numItems > mItemCount - atIndex) ? (mItemCount - atIndex) : numItems;
	
	if (numItems < 1)
		return;
	
	if (mGapIndex != atIndex)
		MoveGapTo(atIndex);
	
	mGapCount += numItems;
	mItemCount -= numItems;
	
	if (mGapCount > mExtraCount)
		SizeGapTo(mExtraCount / 2);
}

void
STETextBuffer::MoveGapTo(
	long	toIndex)
{	
	long srcIndex = 0;
	long dstIndex = 0;
	long count = 0;
	
	if (toIndex > mGapIndex) {
		long gapEndIndex = mGapIndex + mGapCount;
		long trailGapCount = mBufferCount - gapEndIndex;
		srcIndex = gapEndIndex;
		dstIndex =  mGapIndex;
		count = toIndex - mGapIndex;
		count = (count > trailGapCount) ? trailGapCount : count;
	}
	else {
		srcIndex = toIndex;
		dstIndex = toIndex + mGapCount;
		count = mGapIndex - toIndex;
	}
	
	if (count > 0)
		memmove(mBuffer + dstIndex, mBuffer + srcIndex, count);	

	mGapIndex = toIndex;
}

void
STETextBuffer::SizeGapTo(
	long	inCount)
{
	bool smaller = inCount < mGapCount;
	
	if (smaller)
		memmove(mBuffer + mGapIndex + inCount,
				mBuffer + mGapIndex + mGapCount,
				mBufferCount - (mGapIndex + mGapCount));
			
	mBuffer = (char *)realloc(mBuffer, mItemCount + inCount);
	
	if (!smaller)
		memmove(mBuffer + mGapIndex + inCount, 
				mBuffer + mGapIndex + mGapCount, 
				mBufferCount - (mGapIndex + mGapCount));

	mGapCount = inCount;
	mBufferCount = mItemCount + mGapCount;
}

const char*
STETextBuffer::GetString(
	long	fromOffset,
	long 	numChars)
{
	char *result = "";
	
	if (numChars < 1)
		return (result);
	
	bool isStartBeforeGap = (fromOffset < mGapIndex);
	bool isEndBeforeGap = ((fromOffset + numChars - 1) < mGapIndex);

	if (isStartBeforeGap == isEndBeforeGap) {
		result = mBuffer + fromOffset;
		if (!isStartBeforeGap)
			result += mGapCount;
	}
	else {
		if (mScratchSize < numChars) {
			mScratchBuffer = (char *)realloc(mScratchBuffer, numChars);
			mScratchSize = numChars;
		}
		
		for (long i = 0; i < numChars; i++)
			mScratchBuffer[i] = (*this)[fromOffset + i];

		result = mScratchBuffer;
	}
	
	return (result);
}

bool
STETextBuffer::FindChar(
	char	inChar,
	long	fromIndex,
	long	*ioDelta)
{
	long numChars = *ioDelta;
	for (long i = 0; i < numChars; i++) {
		if ((*this)[fromIndex + i] == inChar) {
			*ioDelta = i;
			return (TRUE);
		}
	}
	
	return (FALSE);
}

const char*
STETextBuffer::Text()
{
	if (mGapIndex != mItemCount)
		MoveGapTo(mItemCount);
	
	mBuffer[mItemCount] = '\0';
	
	return (mBuffer);
}

template<class T>
STEBuffer<T>::STEBuffer(
	long	inExtraCount,
	long	inCount)
{
	mExtraCount = inExtraCount;
	mItemCount = inCount;
	mBufferCount = mItemCount + mExtraCount;
	mBuffer = (T *)calloc(mBufferCount, sizeof(T));
}

template<class T>
STEBuffer<T>::~STEBuffer()
{
	free(mBuffer);
}

template<class T>
void
STEBuffer<T>::InsertItemsAt(
	long	inNumItems,
	long 	inAtIndex,
	const T	*inItem)
{
	if (inNumItems < 1)
		return;
	
	inAtIndex = (inAtIndex > mItemCount) ? mItemCount : inAtIndex;
	inAtIndex = (inAtIndex < 0) ? 0 : inAtIndex;

	if ((inNumItems + mItemCount) >= mBufferCount) {
		mBufferCount = mItemCount + inNumItems + mExtraCount;
		mBuffer = (T *)realloc(mBuffer, mBufferCount * sizeof(T));
	}
	
	T *loc = mBuffer + inAtIndex;
	memmove(loc + inNumItems, loc, (mItemCount - inAtIndex) * sizeof(T));
	memcpy(loc, inItem, inNumItems * sizeof(T));
	
	mItemCount += inNumItems;
}

template<class T>
void
STEBuffer<T>::RemoveItemsAt(
	long	inNumItems,
	long 	inAtIndex)
{
	if (inNumItems < 1)
		return;
	
	inAtIndex = (inAtIndex >= mItemCount) ? (mItemCount - 1) : inAtIndex;
	inAtIndex = (inAtIndex < 0) ? 0 : inAtIndex;
	
	T *loc = mBuffer + inAtIndex;
	memmove(loc, loc + inNumItems, 
			(mItemCount - (inNumItems + inAtIndex)) * sizeof(T));
	
	mItemCount -= inNumItems;
	
	if ((mBufferCount - mItemCount) > mExtraCount) {
		mBufferCount = mItemCount + mExtraCount;
		mBuffer = (T *)realloc(mBuffer, mBufferCount * sizeof(T));
	}
}

STEWidthBuffer::STEWidthBuffer()
	: STEBuffer<STEWidthTable>()
{
}

float
STEWidthBuffer::StringWidth(
	STETextBuffer		&inText,
	long				fromOffset,
	long 				length,
	ConstSTEStylePtr	inStyle,
	BView				*inView)
{
	if (length < 1)
		return (0.0);
			
	long index = 0;
	if (!FindTable(inStyle->font, &index)) {
		index = mItemCount;
		
		STEWidthTable newTable;
		strcpy(newTable.font, inStyle->font);
		newTable.factor = 0.0;
		
		InsertTable(&newTable, index);
		AddToTable(' ', index, inStyle, inView);
	}
	
	float 				result = 0.0;
	uchar				theChar;
	STEWidthEntryPtr	table = (STEWidthEntryPtr)&mBuffer[index].table;
	float 				escapement = 0.0;
	float 				conversion = mBuffer[index].factor * inStyle->size * 72.0;	
	conversion = (conversion == 0.0) ? 1.0 : conversion;	
	
	for (long i = 0; i < length; i++) {
		theChar = inText[fromOffset + i];
		if (table[theChar].valid)
			escapement = table[theChar].width;
		else
			escapement = AddToTable(theChar, index, inStyle, inView);
		result += (escapement * conversion); 
	}

	return (result);
}

void
STEWidthBuffer::InsertTable(
	STEWidthTablePtr	inTable,
	long 				index)
{
	for (long i = 0; i < 256; i++)
		inTable->table[i].valid = FALSE;
		
	InsertItemsAt(1, index, inTable);
}

bool
STEWidthBuffer::FindTable(
	const font_name	inFont,
	long 			*outIndex)
{
	for (long i = 0; i < mItemCount; i++) {
		if (strcmp(inFont, mBuffer[i].font) == 0) {
			*outIndex = i;
			return (TRUE);
		}
	}
	
	return (FALSE);
}

float
STEWidthBuffer::AddToTable(
	uchar				inChar,
	long				index,
	ConstSTEStylePtr	inStyle,
	BView				*inView)
{
	float result = 0.0;
	float factor = 0.0;

	inView->SetFontName(inStyle->font);
	inView->GetCharEscapements((char *)&inChar, 1, &result, &factor);
	
	// Kludge!! GetCharEscapements() can't handle bitmap fonts!
	if (result == 0.0) {
		inView->GetCharEscapements(" ", 1, &result, &factor);
		if (result == 0.0) {
			factor = 0.0;
			result = inView->StringWidth((char *)&inChar, 1);
		}
	}
	
	mBuffer[index].factor = factor;
	mBuffer[index].table[inChar].width = result;	
	mBuffer[index].table[inChar].valid = TRUE;
	
	// should this char even have a width?
	if (inChar < ' ') {
		mBuffer[index].table[inChar].width = 0.0;
		result = 0.0;
	}
		
	return (result);
}

STELineBuffer::STELineBuffer()
	: STEBuffer<STELine>(20, 2)
{
}
	
void
STELineBuffer::InsertLine(
	STELinePtr	inLine,
	long		index)
{
	InsertItemsAt(1, index, inLine);
}

void
STELineBuffer::RemoveLines(
	long	index,
	long	count)
{
	RemoveItemsAt(count, index);
}

void
STELineBuffer::RemoveLineRange(
	long	fromOffset,
	long 	toOffset)
{
	long fromLine = OffsetToLine(fromOffset);
	long toLine = OffsetToLine(toOffset);

	long count = toLine - fromLine;
	if (count > 0)
		RemoveLines(fromLine + 1, count);

	BumpOffset(fromOffset - toOffset, fromLine + 1);
}

long
STELineBuffer::OffsetToLine(
	long	offset) const
{
	long minIndex = 0;
	long maxIndex = mItemCount - 1;
	long index = 0;
	
	while (minIndex < maxIndex) {
		index = (minIndex + maxIndex) >> 1;
		if (offset >= mBuffer[index].offset) {
			if (offset < mBuffer[index + 1].offset)
				break;
			else
				minIndex = index + 1;
		}
		else
			maxIndex = index;
	}
	
	return (index);
}

long
STELineBuffer::PixelToLine(
	float	pixel) const
{
	long minIndex = 0;
	long maxIndex = mItemCount - 1;
	long index = 0;
	
	while (minIndex < maxIndex) {
		index = (minIndex + maxIndex) >> 1;
		if (pixel >= mBuffer[index].origin) {
			if (pixel < mBuffer[index + 1].origin)
				break;
			else
				minIndex = index + 1;
		}
		else
			maxIndex = index;
	}
	
	return (index);
} 

void
STELineBuffer::BumpOrigin(
	float	delta,
	long	index)
{	
	for (long i = index; i < mItemCount; i++)
		mBuffer[i].origin += delta;
}

void
STELineBuffer::BumpOffset(
	long	delta,
	long 	index)
{
	for (long i = index; i < mItemCount; i++)
		mBuffer[i].offset += delta;
}

STEStyleRunDescBuffer::STEStyleRunDescBuffer()
	: STEBuffer<STEStyleRunDesc>(20)
{
}

void
STEStyleRunDescBuffer::InsertDesc(
	STEStyleRunDescPtr	inDesc,
	long				index)
{
	InsertItemsAt(1, index, inDesc);
}

void
STEStyleRunDescBuffer::RemoveDescs(
	long	index,
	long 	count)
{
	RemoveItemsAt(count, index);
}

long
STEStyleRunDescBuffer::OffsetToRun(
	long	offset) const
{
	if (mItemCount <= 1)
		return (0);
		
	long minIndex = 0;
	long maxIndex = mItemCount;
	long index = 0;
	
	while (minIndex < maxIndex) {
		index = (minIndex + maxIndex) >> 1;
		if (offset >= mBuffer[index].offset) {
			if (index >= (mItemCount - 1)) {
				break;
			}
			else {
				if (offset < mBuffer[index + 1].offset)
					break;
				else
					minIndex = index + 1;
			}
		}
		else
			maxIndex = index;
	}
	
	return (index);
}

void
STEStyleRunDescBuffer::BumpOffset(
	long	delta,
	long 	index)
{
	for (long i = index; i < mItemCount; i++)
		mBuffer[i].offset += delta;
}


STEStyleRecordBuffer::STEStyleRecordBuffer()
	: STEBuffer<STEStyleRecord>()
{
}

long
STEStyleRecordBuffer::InsertRecord(
	ConstSTEStylePtr	inStyle,
	BView				*inView)
{
	long index = 0;

	// look for style in buffer
	if (MatchRecord(inStyle, &index))
		return (index);

	// style not found, add it
	inView->SetFontName(inStyle->font);
	inView->SetFontSize(inStyle->size);
	inView->SetFontShear(inStyle->shear);
	
	font_info fInfo;
	inView->GetFontInfo(&fInfo);
	
	// check if there's any unused space
	for (index = 0; index < mItemCount; index++) {
		if (mBuffer[index].refs < 1) {
			mBuffer[index].refs = 0;
			mBuffer[index].ascent = fInfo.ascent;
			mBuffer[index].descent = fInfo.descent + fInfo.leading;
			mBuffer[index].style = *inStyle;
			return (index);
		}	
	}
	
	// no unused space, expand the buffer
	index = mItemCount;
	STEStyleRecord 	newRecord;
	newRecord.refs = 0;
	newRecord.ascent = fInfo.ascent;
	newRecord.descent = fInfo.descent + fInfo.leading;
	newRecord.style = *inStyle;
	InsertItemsAt(1, index, &newRecord);

	return (index);
}

void
STEStyleRecordBuffer::CommitRecord(
	long	index)
{
	mBuffer[index].refs++;
}

void
STEStyleRecordBuffer::RemoveRecord(
	long	index)
{
	mBuffer[index].refs--;
}

bool
STEStyleRecordBuffer::MatchRecord(
	ConstSTEStylePtr	inRecord,
	long				*outIndex)
{
	for (long i = 0; i < mItemCount; i++) {
		if ( (inRecord->size == mBuffer[i].style.size) &&
			 (inRecord->shear == mBuffer[i].style.shear) &&
			 (inRecord->underline == mBuffer[i].style.underline) &&
			 (inRecord->color.red == mBuffer[i].style.color.red) &&
			 (inRecord->color.green == mBuffer[i].style.color.green) &&
			 (inRecord->color.blue == mBuffer[i].style.color.blue) &&
			 (inRecord->color.alpha == mBuffer[i].style.color.alpha) &&
			 (inRecord->extra == mBuffer[i].style.extra) ) {
			if (strcmp(inRecord->font, mBuffer[i].style.font) == 0) {
				*outIndex = i;
				return (TRUE);
			}
		}
	}

	return (FALSE);
}

STEStyleBuffer::STEStyleBuffer(
	ConstSTEStylePtr	inStyle)
{
	mValidNullStyle = TRUE;
	mNullStyle = *inStyle;
}

void
STEStyleBuffer::InvalidateNullStyle()
{
	if (mStyleRunDesc.ItemCount() < 1)
		return;
		
	mValidNullStyle = FALSE;
}

bool
STEStyleBuffer::IsValidNullStyle() const
{
	return (mValidNullStyle);
}

void
STEStyleBuffer::SyncNullStyle(
	long	offset)
{
	if ((mValidNullStyle) || (mStyleRunDesc.ItemCount() < 1))
		return;
	
	long index = OffsetToRun(offset);
	mNullStyle = mStyleRecord[mStyleRunDesc[index]->index]->style;

	mValidNullStyle = TRUE;
}	

void
STEStyleBuffer::SetNullStyle(
	ulong				inMode,
	ConstSTEStylePtr	inStyle,
	long				offset)
{
	if ((mValidNullStyle) || (mStyleRunDesc.ItemCount() < 1))
		SetStyle(inMode, inStyle, &mNullStyle);
	else {
		long index = OffsetToRun(offset - 1);
		mNullStyle = mStyleRecord[mStyleRunDesc[index]->index]->style;
		SetStyle(inMode, inStyle, &mNullStyle);	
	}
	
	mValidNullStyle = TRUE;
}

STEStyle
STEStyleBuffer::GetNullStyle() const
{
	return (mNullStyle);
}	

bool
STEStyleBuffer::IsContinuousStyle(
	ulong		*ioMode,
	STEStylePtr	outStyle,
	long		fromOffset,
	long 		toOffset) const
{	
	if (mStyleRunDesc.ItemCount() < 1) {
		SetStyle(*ioMode, &mNullStyle, outStyle);
		return (TRUE);
	}
		
	bool result = TRUE;
	long fromIndex = OffsetToRun(fromOffset);
	long toIndex = OffsetToRun(toOffset - 1);
	
	if (fromIndex == toIndex) {		
		long				styleIndex = mStyleRunDesc[fromIndex]->index;
		ConstSTEStylePtr	style = &mStyleRecord[styleIndex]->style;
		
		SetStyle(*ioMode, style, outStyle);
		result = TRUE;
	}
	else {
		long				styleIndex = mStyleRunDesc[toIndex]->index;
		STEStyle			theStyle = mStyleRecord[styleIndex]->style;
		ConstSTEStylePtr	style = NULL;
		
		for (long i = fromIndex; i < toIndex; i++) {
			styleIndex = mStyleRunDesc[i]->index;
			style = &mStyleRecord[styleIndex]->style;
			
			if (*ioMode & doFont) {
				if (strcmp(theStyle.font, style->font) != 0) {
					*ioMode &= ~doFont;
					result = FALSE;
				}	
			}
				
			if (*ioMode & doSize) {
				if (theStyle.size != style->size) {
					*ioMode &= ~doSize;
					result = FALSE;
				}
			}
				
			if (*ioMode & doShear) {
				if (theStyle.shear != style->shear) {
					*ioMode &= ~doShear;
					result = FALSE;
				}
			}
				
			if (*ioMode & doUnderline) {
				if (theStyle.underline != style->underline) {
					*ioMode &= ~doUnderline;
					result = FALSE;
				}
			}
				
			if (*ioMode & doColor) {
				if ( (theStyle.color.red != style->color.red) ||
					 (theStyle.color.green != style->color.green) ||
					 (theStyle.color.blue != style->color.blue) ||
					 (theStyle.color.alpha != style->color.alpha) ) {
					*ioMode &= ~doColor;
					result = FALSE;
				}
			}
			
			if (*ioMode & doExtra) {
				if (theStyle.extra != style->extra) {
					*ioMode &= ~doExtra;
					result = FALSE;
				}
			}
		}
		
		SetStyle(*ioMode, &theStyle, outStyle);
	}
	
	return (result);
}

void
STEStyleBuffer::SetStyleRange(
	long				fromOffset,
	long				toOffset,
	long				textLen,
	ulong				inMode,
	ConstSTEStylePtr	inStyle,
	BView				*inView)
{
	if (inStyle == NULL)
		inStyle = &mNullStyle;

	if (fromOffset == toOffset) {
		SetNullStyle(inMode, inStyle, fromOffset);
		return;
	}
	
	if (mStyleRunDesc.ItemCount() < 1) {
		STEStyleRunDesc newDesc;
		newDesc.offset = fromOffset;
		newDesc.index = mStyleRecord.InsertRecord(inStyle, inView);
		mStyleRunDesc.InsertDesc(&newDesc, 0);
		mStyleRecord.CommitRecord(newDesc.index);
		return;
	}

	long styleIndex = 0;
	long offset = fromOffset;
	long runIndex = OffsetToRun(offset);
	do {
		STEStyleRunDesc	runDesc = *mStyleRunDesc[runIndex];
		long			runEnd = textLen;
		if (runIndex < (mStyleRunDesc.ItemCount() - 1))
			runEnd = mStyleRunDesc[runIndex + 1]->offset;
		
		STEStyle style = mStyleRecord[runDesc.index]->style;
		SetStyle(inMode, inStyle, &style);

		styleIndex = mStyleRecord.InsertRecord(&style, inView);
		
		if ( (runDesc.offset == offset) && (runIndex > 0) && 
			 (mStyleRunDesc[runIndex - 1]->index == styleIndex) ) {
			RemoveStyles(runIndex);
			runIndex--;	
		}

		if (styleIndex != runDesc.index) {
			if (offset > runDesc.offset) {
				STEStyleRunDesc newDesc;
				newDesc.offset = offset;
				newDesc.index = styleIndex;
				mStyleRunDesc.InsertDesc(&newDesc, runIndex + 1);
				mStyleRecord.CommitRecord(newDesc.index);
				runIndex++;
			}
			else {
				((STEStyleRunDescPtr)mStyleRunDesc[runIndex])->index = styleIndex;
				mStyleRecord.CommitRecord(styleIndex);
			}
				
			if (toOffset < runEnd) {
				STEStyleRunDesc newDesc;
				newDesc.offset = toOffset;
				newDesc.index = runDesc.index;
				mStyleRunDesc.InsertDesc(&newDesc, runIndex + 1);
				mStyleRecord.CommitRecord(newDesc.index);
			}
		}
		
		runIndex++;
		offset = runEnd;	
	} while (offset < toOffset);
	
	if ( (offset == toOffset) && (runIndex < mStyleRunDesc.ItemCount()) &&
		 (mStyleRunDesc[runIndex]->index == styleIndex) )
		RemoveStyles(runIndex);
}		 

void
STEStyleBuffer::GetStyle(
	long		inOffset,
	STEStylePtr	outStyle) const
{
	if (mStyleRunDesc.ItemCount() < 1) {
		*outStyle = mNullStyle;
		return;
	}
	
	long runIndex = OffsetToRun(inOffset);
	long styleIndex = mStyleRunDesc[runIndex]->index;
	*outStyle = mStyleRecord[styleIndex]->style;
}

STEStyleRangePtr
STEStyleBuffer::GetStyleRange(
	long	startOffset,
	long	endOffset) const
{
	STEStyleRangePtr result = NULL;
	
	long startIndex = OffsetToRun(startOffset);
	long endIndex = OffsetToRun(endOffset);
	
	long numStyles = endIndex - startIndex + 1;
	numStyles = (numStyles < 1) ? 1 : numStyles;
	result = (STEStyleRangePtr)malloc(sizeof(long) +
									  (sizeof(STEStyleRun) * numStyles));

	result->count = numStyles;
	STEStyleRunPtr run = &result->runs[0];
	for (long index = 0; index < numStyles; index++) {
		*run = (*this)[startIndex + index];
		run->offset -= startOffset;
		run->offset = (run->offset < 0) ? 0 : run->offset;
		run++;
	}
	
	return (result);
}

void
STEStyleBuffer::RemoveStyleRange(
	long	fromOffset,
	long	toOffset)
{
	long fromIndex = mStyleRunDesc.OffsetToRun(fromOffset);
	long toIndex = mStyleRunDesc.OffsetToRun(toOffset) - 1;

	long count = toIndex - fromIndex;
	if (count > 0) {
		RemoveStyles(fromIndex + 1, count);
		toIndex = fromIndex;
	}
	
	mStyleRunDesc.BumpOffset(fromOffset - toOffset, fromIndex + 1);

	if ((toIndex == fromIndex) && (toIndex < (mStyleRunDesc.ItemCount() - 1)))
		((STEStyleRunDescPtr)mStyleRunDesc[toIndex + 1])->offset = fromOffset;
	
	if (fromIndex < (mStyleRunDesc.ItemCount() - 1)) {
		ConstSTEStyleRunDescPtr runDesc = mStyleRunDesc[fromIndex];
		if (runDesc->offset == (runDesc + 1)->offset) {
			RemoveStyles(fromIndex);
			fromIndex--;
		}
	}
	
	if ((fromIndex >= 0) && (fromIndex < (mStyleRunDesc.ItemCount() - 1))) {
		ConstSTEStyleRunDescPtr runDesc = mStyleRunDesc[fromIndex];
		if (runDesc->index == (runDesc + 1)->index)
			RemoveStyles(fromIndex + 1);
	}
}

void
STEStyleBuffer::RemoveStyles(
	long	index,
	long 	count)
{
	for (long i = index; i < (index + count); i++)
		mStyleRecord.RemoveRecord(mStyleRunDesc[i]->index);
		
	mStyleRunDesc.RemoveDescs(index, count);
}

long
STEStyleBuffer::Iterate(
	long				fromOffset,
	long				length,
	ConstSTEStylePtr	*outStyle,
	float				*outAscent,
	float				*outDescent) const
{
	long numRuns = mStyleRunDesc.ItemCount();
	if ((length < 1) || (numRuns < 1))
		return (0);

	long 					result = length;
	long 					runIndex = mStyleRunDesc.OffsetToRun(fromOffset);
	ConstSTEStyleRunDescPtr	run = mStyleRunDesc[runIndex];

	if (outStyle != NULL)
		*outStyle = &mStyleRecord[run->index]->style;
	if (outAscent != NULL)
		*outAscent = mStyleRecord[run->index]->ascent;
	if (outDescent != NULL)
		*outDescent = mStyleRecord[run->index]->descent;
	
	if (runIndex < (numRuns - 1)) {
		long nextOffset = (run + 1)->offset - fromOffset;
		result = (result > nextOffset) ? nextOffset : result;
	}
	
	return (result);
}

long
STEStyleBuffer::OffsetToRun(
	long	offset) const
{
	return (mStyleRunDesc.OffsetToRun(offset));
}

void
STEStyleBuffer::BumpOffset(
	long	delta,
	long	index)
{
	mStyleRunDesc.BumpOffset(delta, index);
}

void
STEStyleBuffer::SetStyle(
	ulong				mode,
	ConstSTEStylePtr	fromStyle,
	STEStylePtr			toStyle) const
{	
	if (mode & doFont) 
		strcpy(toStyle->font, fromStyle->font);
		
	if (mode & doSize) {
		if (mode & addSize)
			toStyle->size += fromStyle->size;
		else
			toStyle->size = fromStyle->size;
	}
		
	if (mode & doShear)
		toStyle->shear = fromStyle->shear;
		
	if (mode & doUnderline)
		toStyle->underline = fromStyle->underline;
		
	if (mode & doColor)
		toStyle->color = fromStyle->color;
		
	if (mode & doExtra)
		toStyle->extra = fromStyle->extra;
}

STEStyleRun
STEStyleBuffer::operator[](
	long	index) const
{
	STEStyleRun run;
	
	if (mStyleRunDesc.ItemCount() < 1) {
		run.offset = 0;
		run.style = mNullStyle;
	} else {
		ConstSTEStyleRunDescPtr	runDesc = mStyleRunDesc[index];
		ConstSTEStyleRecordPtr	record = mStyleRecord[runDesc->index];
		run.offset = runDesc->offset;
		run.style = record->style;
	}
	
	return (run);
}
