// ============================================================
//  CStyledEditWindow.cpp	1996 Hiroshi Lockheimer
// ============================================================

#include "CStyledEditWindow.h"
#include "CStyledEditApp.h"
#include "CFontMenuItem.h"
#include "CColorMenuItem.h"


const ulong	msg_Save				= 'Save';
const ulong	msg_SaveAs				= 'SvAs';
const ulong msg_Revert				= 'Rvrt';
const ulong	msg_Undo				= 'Undo';
const ulong	msg_Clear				= 'Cler';
const ulong	msg_SelectAll			= 'SAll';
const ulong	msg_FontSelected 		= 'FSel';
const ulong	msg_SizeSelected 		= 'SSel';
const ulong	msg_Plain				= 'Plai';
const ulong	msg_Underline			= 'Undr';
const ulong	msg_ColorSelected		= 'CSel';
const ulong	msg_BackColorSelected	= 'BSel';
const ulong	msg_SetSelectable		= 'Slbl';
const ulong	msg_SetEditable			= 'Edbl';
const ulong	msg_SetWrap				= 'Wrap';
const ulong	msg_SetOffscreen		= 'OffS';

const rgb_color	kRed 		= {255, 0, 0, 0};
const rgb_color	kGreen 		= {0, 255, 0, 0};
const rgb_color	kBlue 		= {0, 0, 255, 0};
const rgb_color	kCyan 		= {0, 255, 255, 0};
const rgb_color	kMagenta 	= {255, 0, 255, 0};
const rgb_color	kYellow 	= {255, 255, 0, 0};
const rgb_color	kBlack 		= {0, 0, 0, 0};
const rgb_color	kWhite		= {255, 255, 255, 0};
const rgb_color	kGray		= {238, 238, 238, 0};
const rgb_color	kBackYellow	= {255, 255, 153, 0};
const rgb_color	kBackGreen	= {204, 255, 204, 0};
const rgb_color	kBackBlue	= {153, 255, 255, 0};
const rgb_color	kPink		= {255, 204, 204, 0};
const rgb_color	kPurple		= {204, 204, 255, 0};

const STEStyle kNullStyle = { "Times New Roman", 	// font
							  12.0, 				// size
							  90.0, 				// shear
							  FALSE, 				// underline?
							  {0, 0, 0, 0}, 		// color
							  0 };					// extra
							
							
BList 	CStyledEditWindow::sWindowList;
long	CStyledEditWindow::sUntitledCount = 0;


CStyledEditWindow::CStyledEditWindow()
	: BWindow(GetFrameRect(), "Untitled", B_DOCUMENT_WINDOW, 0)
{
	sUntitledCount++;
	
	InitWindow();
	
	Lock();
	
	char title[20];
	sprintf(title, "Untitled %ld", sUntitledCount);
	SetTitle(title);
	
	Unlock();
}


CStyledEditWindow::CStyledEditWindow(
	record_ref	inRef)
		: BWindow(GetFrameRect(), "Untitled", B_DOCUMENT_WINDOW, 0)
{
	InitWindow();
	
	Lock();
	
	mFile = new BFile(inRef);
	
	char name[B_FILE_NAME_LENGTH + 1];
	mFile->GetName(name);
	SetTitle(name);
	
	ReadData();
	
	Unlock();
}


CStyledEditWindow::~CStyledEditWindow()
{
	if (mFile != NULL) {
		mFile->Close();
		delete (mFile);
	}
	
	sWindowList.RemoveItem(this);
	be_app->PostMessage(msg_WindowRemoved);
}


void
CStyledEditWindow::MessageReceived(
	BMessage	*inMessage)
{
	switch (inMessage->what) {
		case msg_Save:
			Save();
			break;
			
		case msg_SaveAs:
			SaveAs();
			break;
			
		case msg_Revert:
			Revert();
			break;
			
		case msg_Undo:
			// not supported
			break;
			
		case msg_Clear:
			mTextView->Clear();
			break;
			
		case msg_SelectAll:
			mTextView->SelectAll();
			break;
			
		case msg_FontSelected:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			FontSelected(menu->Label());
			break;
		}
			
		case msg_SizeSelected:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			SizeSelected(atof(menu->Label()));
			break;
		}
			
		case msg_Plain:
		case msg_Underline:
			UnderlineChanged(inMessage->what == msg_Underline);
			break;
			
		case msg_ColorSelected:
		case msg_BackColorSelected:
		{
			long 		dataSize = 0;
			rgb_color	*color = (rgb_color *)inMessage->FindData("color", B_RGB_COLOR_TYPE, &dataSize);
			ColorSelected(*color, inMessage->what == msg_ColorSelected);
			break;
		}
		
		case msg_SetSelectable:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			menu->SetMarked(!menu->IsMarked());
			mTextView->MakeSelectable(menu->IsMarked());
			break;
		}
		
		case msg_SetEditable:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			menu->SetMarked(!menu->IsMarked());
			mTextView->MakeEditable(menu->IsMarked());
			break;
		}
			
		case msg_SetWrap:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			menu->SetMarked(!menu->IsMarked());
			mTextView->SetWordWrap(menu->IsMarked());
			BRect textRect = mTextView->TextRect();
			if (!menu->IsMarked())
				textRect.right = textRect.left + 1500.0;
			else {
				textRect = mTextView->Frame();
				textRect.OffsetTo(B_ORIGIN);
				textRect.InsetBy(3.0, 3.0);
			}
			mTextView->SetTextRect(textRect);
			break;
		}
			
		case msg_SetOffscreen:
		{
			BMenuItem *menu = (BMenuItem *)inMessage->FindObject("source");
			menu->SetMarked(!menu->IsMarked());
			if (menu->IsMarked())
				mTextView->UseOffscreen(B_COLOR_8_BIT);
			else
				mTextView->DontUseOffscreen();
			break;
		}
			
		default:
			BWindow::MessageReceived(inMessage);
			break;
	}
}


void
CStyledEditWindow::SaveRequested(
	record_ref	directory,
	const char	*name)
{
	SetTitle(name);

	if (mFile != NULL)
		delete (mFile);
	mFile = new BFile();

	BDirectory dir(directory);
	dir.Create(name, mFile);
	
	Save();
	
	if (mWaitForSave)
		PostMessage(B_CLOSE_REQUESTED);
}


bool
CStyledEditWindow::QuitRequested()
{
	bool result = TRUE;
	mWaitForSave = FALSE;
	
	if (mDirty) {
		const char 	*text1 = "Save changes to the document ";
		const char 	*text2 = "?";
		long		len = strlen(text1) + B_FILE_NAME_LENGTH + strlen(text2); 
		char		*title = (char *)malloc(len + 1);
		sprintf(title, "%s%s%s", text1, Title(), text2);
		
		BAlert *alert = new BAlert(B_EMPTY_STRING, title, "Don't Save", 
	 							   "Cancel", "Save"); 
		switch (alert->Go()) {
			case 0:
				break;
				
			case 1:
				result = FALSE;
				break;
				
			default:
				if (mFile == NULL) {
					result = FALSE;
					mWaitForSave = TRUE;
				}
				Save();
				break;
		}
		
		free(title);
	}
	 
	return (result);
}


void
CStyledEditWindow::MenusWillShow()
{
	BMenuItem *menuItem = NULL;
	
	menuItem = mFileMenu->FindItem("Save");
	menuItem->SetEnabled(mDirty);
	
	menuItem = mFileMenu->FindItem("Revert");
	menuItem->SetEnabled((mDirty) && (mFile != NULL)); 
	
	long selStart = 0;
	long selEnd = 0;
	mTextView->GetSelection(&selStart, &selEnd);
	
	menuItem = mEditMenu->FindItem("Cut");
	menuItem->SetEnabled(selStart != selEnd);

	menuItem = mEditMenu->FindItem("Copy");
	menuItem->SetEnabled(selStart != selEnd);

	menuItem = mEditMenu->FindItem("Paste");
	menuItem->SetEnabled(mTextView->CanPaste());
	
	menuItem = mEditMenu->FindItem("Clear");
	menuItem->SetEnabled(selStart != selEnd);

	ulong	 	mode = doFont | doSize | doUnderline | doColor;
	STEStyle	style;
	mTextView->IsContinuousStyle(&mode, &style);
	
	menuItem = mFontMenu->FindMarked();
	if (menuItem != NULL)
		menuItem->SetMarked(FALSE);
	if (mode & doFont) {
		menuItem = mFontMenu->FindItem(style.font);
		if (menuItem != NULL)
			menuItem->SetMarked(TRUE);
	}
	
	menuItem = mSizeMenu->FindMarked();
	if (menuItem != NULL)
		menuItem->SetMarked(FALSE);
	if (mode & doSize) {
		char numStr[3];
		sprintf(numStr, "%ld", (long)style.size);
		menuItem = mSizeMenu->FindItem(numStr);
		if (menuItem != NULL)
			menuItem->SetMarked(TRUE);
	}
	
	menuItem = mStyleMenu->FindMarked();
	if (menuItem != NULL)
		menuItem->SetMarked(FALSE);
	if (mode & doUnderline) {
		if (style.underline)
			menuItem = mStyleMenu->FindItem("Underline");
		else
			menuItem = mStyleMenu->FindItem("Plain");
		if (menuItem != NULL)
			menuItem->SetMarked(TRUE);
	}
	
	menuItem = mColorMenu->FindMarked();
	if (menuItem != NULL)
		menuItem->SetMarked(FALSE);
	if (mode & doColor) {
		menuItem = NULL;
		CColorMenuItem *colorItem = NULL;
		for ( long i = 0; 
			  (colorItem = (CColorMenuItem *)mColorMenu->ItemAt(i)) != NULL;
			  i++) {
			if (colorItem->IsColorEqual(style.color)) {
				menuItem = colorItem;
				break;
			}
		}
		if (menuItem != NULL)
			menuItem->SetMarked(TRUE);
	}
}


void
CStyledEditWindow::SetDirty(
	bool	inDirty)
{
	mDirty = inDirty;
}


CStyledEditWindow*
CStyledEditWindow::FindWindow(
	record_ref	inRef)
{
	BFile theFile(inRef);
	
	CStyledEditWindow *window = NULL;
	for ( long i = 0; 
		  (window = (CStyledEditWindow *)sWindowList.ItemAt(i)) != NULL; 
		  i++) {
		if (window->mFile != NULL) {
			if (*(window->mFile) == theFile)
				return (window);
		}
	}
	
	return (NULL);
}


void
CStyledEditWindow::InitWindow()
{
	sWindowList.AddItem(this);
	be_app->PostMessage(msg_WindowAdded);
	
	Lock();
	
	mFile = NULL;
	mDirty = FALSE;
	mStyled = TRUE;
	
	BRect mbarRect = Bounds();
	mbarRect.bottom = mbarRect.top + 15.0;
	BMenuBar *mbar = new BMenuBar(mbarRect, "mbar");

	mFileMenu = new BMenu("File");	
	mbar->AddItem(mFileMenu);
	
	mEditMenu = new BMenu("Edit");
	mbar->AddItem(mEditMenu);	

	mFontMenu = new BMenu("Font");
	mFontMenu->SetRadioMode(TRUE);
	mbar->AddItem(mFontMenu);

	mSizeMenu = new BMenu("Size");
	mSizeMenu->SetRadioMode(TRUE);
	mbar->AddItem(mSizeMenu);

	mStyleMenu = new BMenu("Style");
	mStyleMenu->SetRadioMode(TRUE);
	mbar->AddItem(mStyleMenu);

	mColorMenu = new BMenu("Color");
	mColorMenu->SetRadioMode(TRUE);
	mbar->AddItem(mColorMenu);

	mBackColorMenu = new BMenu("Background");
	mBackColorMenu->SetRadioMode(TRUE);
	mbar->AddItem(mBackColorMenu);

	BMenu* optionsMenu = new BMenu("Options");
	mbar->AddItem(optionsMenu);

	AddChild(mbar);

	BRect textFrame = Bounds();
	textFrame.top = mbar->Bounds().bottom + 1.0;
	textFrame.right -= B_V_SCROLL_BAR_WIDTH;
	textFrame.bottom -= B_H_SCROLL_BAR_HEIGHT;

	BRect textRect = textFrame;
	textRect.OffsetTo(B_ORIGIN);
	textRect.InsetBy(3.0, 3.0);
	
	mTextView = new CTextView(textFrame, "text", textRect, &kNullStyle);
	BScrollView *scroller = new BScrollView("scroller", mTextView,
											B_FOLLOW_ALL, 0,
											TRUE, TRUE, FALSE);
	AddChild(scroller);
	
	mFileMenu->AddItem(new BMenuItem("Save", new BMessage(msg_Save), 'S'));
	mFileMenu->AddItem(new BMenuItem("Save As", new BMessage(msg_SaveAs)));
	mFileMenu->AddItem(new BMenuItem("Revert", new BMessage(msg_Revert)));
	mFileMenu->AddItem(new BMenuItem("Close", new BMessage(B_CLOSE_REQUESTED), 'W'));
	
	BMenuItem *editItem = NULL;
	editItem = new BMenuItem("Undo", new BMessage(msg_Undo), 'Z');
	editItem->SetEnabled(FALSE);
	mEditMenu->AddItem(editItem);
	mEditMenu->AddSeparatorItem();
	editItem = new BMenuItem("Cut", new BMessage(B_CUT), 'X');
	editItem->SetTarget(mTextView);
	mEditMenu->AddItem(editItem);
	editItem = new BMenuItem("Copy", new BMessage(B_COPY), 'C');
	editItem->SetTarget(mTextView);
	mEditMenu->AddItem(editItem);
	editItem = new BMenuItem("Paste", new BMessage(B_PASTE), 'V');
	editItem->SetTarget(mTextView);
	mEditMenu->AddItem(editItem);
	mEditMenu->AddItem(new BMenuItem("Clear", new BMessage(msg_Clear)));
	mEditMenu->AddItem(new BMenuItem("Select All", new BMessage(msg_SelectAll), 'A'));	

	for (long i = 0; i < count_fonts(); i++) {
		font_name fontName;
		get_font_name(i, &fontName);
		mFontMenu->AddItem(new CFontMenuItem(fontName, new BMessage(msg_FontSelected)));
	}
	
	mSizeMenu->AddItem(new BMenuItem("9", new BMessage(msg_SizeSelected)));
	mSizeMenu->AddItem(new BMenuItem("10", new BMessage(msg_SizeSelected)));	
	mSizeMenu->AddItem(new BMenuItem("12", new BMessage(msg_SizeSelected)));
	mSizeMenu->AddItem(new BMenuItem("14", new BMessage(msg_SizeSelected)));
	mSizeMenu->AddItem(new BMenuItem("18", new BMessage(msg_SizeSelected)));
	mSizeMenu->AddItem(new BMenuItem("24", new BMessage(msg_SizeSelected)));
	mSizeMenu->AddItem(new BMenuItem("36", new BMessage(msg_SizeSelected)));	
	mSizeMenu->AddItem(new BMenuItem("48", new BMessage(msg_SizeSelected)));	
	mSizeMenu->AddItem(new BMenuItem("72", new BMessage(msg_SizeSelected)));	
	
	mStyleMenu->AddItem(new BMenuItem("Plain", new BMessage(msg_Plain)));
	mStyleMenu->AddItem(new BMenuItem("Underline", new BMessage(msg_Underline)));
	
	mColorMenu->AddItem(new CColorMenuItem("Black", kBlack, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Red", kRed, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Green", kGreen, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Blue", kBlue, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Cyan", kCyan, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Magenta", kMagenta, new BMessage(msg_ColorSelected)));
	mColorMenu->AddItem(new CColorMenuItem("Yellow", kYellow, new BMessage(msg_ColorSelected)));	
	
	BMenuItem *whiteBack = new CColorMenuItem("White", kWhite, new BMessage(msg_BackColorSelected), FALSE);
	whiteBack->SetMarked(TRUE);
	mBackColorMenu->AddItem(whiteBack);
	mBackColorMenu->AddItem(new CColorMenuItem("Gray", kGray, new BMessage(msg_BackColorSelected), FALSE));
	mBackColorMenu->AddItem(new CColorMenuItem("Yellow", kBackYellow, new BMessage(msg_BackColorSelected), FALSE));
	mBackColorMenu->AddItem(new CColorMenuItem("Green", kBackGreen, new BMessage(msg_BackColorSelected), FALSE));
	mBackColorMenu->AddItem(new CColorMenuItem("Blue", kBackBlue, new BMessage(msg_BackColorSelected), FALSE));
	mBackColorMenu->AddItem(new CColorMenuItem("Pink", kPink, new BMessage(msg_BackColorSelected), FALSE));
	mBackColorMenu->AddItem(new CColorMenuItem("Purple", kPurple, new BMessage(msg_BackColorSelected), FALSE));	

	BMenuItem* optionsItem = NULL;
	optionsItem = new BMenuItem("Selectable", new BMessage(msg_SetSelectable));
	optionsItem->SetMarked(mTextView->IsSelectable());	
	optionsMenu->AddItem(optionsItem);
	optionsItem = new BMenuItem("Editable", new BMessage(msg_SetEditable));
	optionsItem->SetMarked(mTextView->IsEditable());	
	optionsMenu->AddItem(optionsItem);	
	optionsItem = new BMenuItem("Wrap", new BMessage(msg_SetWrap));
	optionsItem->SetMarked(mTextView->DoesWordWrap());	
	optionsMenu->AddItem(optionsItem);	
	optionsItem = new BMenuItem("Offscreen", new BMessage(msg_SetOffscreen));
	optionsItem->SetMarked(mTextView->DoesUseOffscreen());	
	optionsMenu->AddItem(optionsItem);	
			
	Unlock();
}


void
CStyledEditWindow::FontSelected(
	const char*	font)
{
	BView *theView = CurrentFocus();
	if ((theView != NULL) && (is_kind_of(theView, STEngine))) {
		STEStyle newStyle;
		strcpy(newStyle.font, font);
		((STEngine *)theView)->SetStyle(doFont, &newStyle);
	}
}


void
CStyledEditWindow::SizeSelected(
	float	size)
{
	BView *theView = CurrentFocus();
	if ((theView != NULL) && (is_kind_of(theView, STEngine))) {
		STEStyle newStyle;
		newStyle.size = size;
		((STEngine *)theView)->SetStyle(doSize, &newStyle);
	}
}


void
CStyledEditWindow::UnderlineChanged(
	bool	underline)
{
	BView *theView = CurrentFocus();
	if ((theView != NULL) && (is_kind_of(theView, STEngine))) {
		STEStyle newStyle;
		newStyle.underline = underline;
		((STEngine *)theView)->SetStyle(doUnderline, &newStyle);
	}
}


void
CStyledEditWindow::ColorSelected(
	rgb_color	color,
	bool		penColor)
{
	BView *theView = CurrentFocus();
	if ((theView != NULL) && (is_kind_of(theView, STEngine))) {
		if (penColor) {		
			STEStyle newStyle;
			newStyle.color = color;
			((STEngine *)theView)->SetStyle(doColor, &newStyle);
		}
		else {
			theView->SetViewColor(color);
			theView->Invalidate();
		}
	}
}


void
CStyledEditWindow::Save()
{
	if (mFile == NULL)
		SaveAs();
	else
		WriteData();
}


void
CStyledEditWindow::SaveAs()
{
	RunSavePanel(Title());
}


void
CStyledEditWindow::Revert()
{ 
	const char 	*text1 = "Revert to the latest version of ";
	const char 	*text2 = "?";
	long		len = strlen(text1) + B_FILE_NAME_LENGTH + strlen(text2); 
	char		*title = (char *)malloc(len + 1);
	sprintf(title, "%s%s%s", text1, Title(), text2);
	
	BAlert *alert = new BAlert(B_EMPTY_STRING, title, "Cancel", "OK");
	if (alert->Go() != 0)
		ReadData();
}


long
CStyledEditWindow::WriteData()
{
	long err = B_NO_ERROR;

	if (!mFile->IsOpen())
		err = mFile->Open(B_READ_WRITE);
		
	if (err == B_NO_ERROR) {
		mFile->Seek(0, B_SEEK_TOP);
		
		long textLen = mTextView->TextLength();
		long styleLen = 0;
		
		if (mStyled) {
			mFile->SetTypeAndApp('TEXT', 'STEE');
			
			long nameLen = strlen("StyledEdit");
			void *styles = mTextView->GetStyleRange(0, textLen, &styleLen);

			mFile->Write("\0StyledEdit", 1 + nameLen);
			mFile->Write(&styleLen, sizeof(long));
			mFile->Write(styles, styleLen);

			free(styles);
			
			styleLen += (1 + nameLen + sizeof(long));
		}
		mFile->Write(mTextView->Text(), textLen);
		mFile->SetSize(styleLen + textLen);
		
		// does this force a flush?
		mFile->Close();
		mFile->Open(B_READ_WRITE);
		
		SetDirty(FALSE);
	}
	
	return (err);
}


long
CStyledEditWindow::ReadData()
{		
	long err = B_NO_ERROR;
	
	err = mFile->Open(B_READ_WRITE);
	if (err == B_NO_ERROR) {
		mFile->Seek(0, B_SEEK_TOP);

		long	len = mFile->Size();
		char	*buffer = (char *)malloc(len);
		err = mFile->Read(buffer, len);
		if (err != B_ERROR) {
			mStyled = FALSE;
			char				*text = buffer;
			STEStyleRangePtr	styles = NULL;
			
			if (buffer[0] == '\0') {
				long nameLen = strlen("StyledEdit");
				if (strncmp(&buffer[1], "StyledEdit", nameLen) == 0) {
					mStyled = TRUE;
					
					long styleLen = 0;
					memcpy(&styleLen, buffer + 1 + nameLen, sizeof(long));

					text = buffer + 1 + nameLen + sizeof(long) + styleLen;
					len -= (1 + nameLen + sizeof(long) + styleLen);
					
					styles = (STEStyleRangePtr)malloc(styleLen);
					memcpy(styles, buffer + 1 + nameLen + sizeof(long), styleLen);
				}
			}

			mTextView->SetText(text, len, styles);
			SetDirty(FALSE);
			
			free(styles);		
		}
		free(buffer);
	}
	
	return (err);
}


BRect
CStyledEditWindow::GetFrameRect()
{
	screen_info screenInfo;
	get_screen_info(&screenInfo);
	
	float	dockWidth = 0.0;	
	get_dock_width(&dockWidth);
	
	BRect	frame;
	frame.left = screenInfo.frame.left + dockWidth + 2.0;
	frame.top = screenInfo.frame.top + 28.0;
	frame.right = frame.left + 500.0;
	frame.bottom = frame.top + 400.0;
	
	BRect 	offsetFrame = frame;
	long	numWindows = sWindowList.CountItems();
	offsetFrame.OffsetBy(15.0 * numWindows, 15.0 * numWindows);
	if (screenInfo.frame.Contains(offsetFrame))
		frame = offsetFrame;
		
	return (frame);
}
