/* BeOS daliclock by Abraham Flaxman (abie@kenlaw.com)
   Copyright (C) 1996

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */ 

#include "DaliClockView.h"
#include <Debug.h>
#define DEBUG 1

DaliClockView::DaliClockView(BRect area)
		: BView(area, "DaliClockView", B_FOLLOW_NONE, B_WILL_DRAW) {
	BBitmap *temp_map;
	BView *temp_view;
	font_info f_info;

	//make the temp_map, lock it, and attach a temp_view to it
	temp_map = new BBitmap(BRect(0,0,64,64), B_COLOR_8_BIT, TRUE);
	temp_map->Lock();

	temp_view = new BView(BRect(0,0,64,64), "tmp_view", B_FOLLOW_NONE, 0);
	temp_map->AddChild(temp_view);
	
	//set the font
	temp_view->SetFontName("Baskerville MT Bold Italic");
	temp_view->SetFontSize(48);
	temp_view->GetFontInfo(&f_info);
	
	//make the digit images
	digits = (BBitmap **)malloc(sizeof(BBitmap *)*12);
	char *chars = "0123456789:-";
	for(int i=0; i<12; i++) {
		temp_view->FillRect(BRect(0,0,64,64), B_SOLID_LOW);
		temp_view->DrawChar(chars[i], BPoint(0.0, 64.0 - f_info.descent));
		temp_view->Sync();

		digits[i] = new BBitmap(BRect(0,0,64,64), B_COLOR_8_BIT);
		digits[i]->SetBits(temp_map->Bits(), temp_map->BitsLength(), 0, B_COLOR_8_BIT);
	}
	//unlock and delete the temp_map
	temp_map->Unlock();
	delete temp_map;

	//initialize all the other variables
	offscreen = new BBitmap(area, B_COLOR_8_BIT);
	mode = CLOCK_MODE;
	timer=0;
	pct = 0.0;
	old_msg = new char[10];
	new_msg = new char[10];

	army_time = TRUE;

	//start the painter thread
	painter = spawn_thread(_run_clock, "Clocker", B_NORMAL_PRIORITY, this);
	resume_thread(painter);
}

DaliClockView::~DaliClockView() {
	//kill the painter
	kill_thread(painter);

	//delete messages
	delete old_msg;
	delete new_msg;

	//delete the digit bitmaps
	for(int i=0; i<12; i++)
		delete digits[i];
	delete digits;

	//delete the offscreen bitmap
	delete offscreen;
}

void DaliClockView::Draw(BRect area) {
	// blank offscreen to white
	uchar white = index_for_color(255, 255, 255);
	PRINT(("got white color\n"));

	uchar black = index_for_color(0, 0, 0);
	PRINT(("got black color\n"));

	for(long i=0; i<offscreen->BitsLength(); i++)
		((uchar *)offscreen->Bits())[i] = white;
	PRINT(("blanked offscreen\n"));

	//increment the percent of morph completed, but never go past 1.0
	pct += 0.1;
	if(pct > 1.0)
		pct = 1.0;
	PRINT(("pct = %f\n", pct));

	//check to see if we still know the correct time, if we don't update things
	time_t cur_time;
	time(&cur_time);
	if(timer != cur_time) {
		PRINT(("timer out of sync, updating\n"));
		timer = cur_time;
		pct = 0.0;

		cur_time+=1;
		struct tm *tm;

		switch(mode) {
			case ALMOST_TO_DATE_MODE:
				tm = localtime(&timer);
				mode = TO_DATE_MODE;
				sprintf(old_msg, "%02d:%02d:%02d", army_time ? tm->tm_hour : (tm->tm_hour+11)%12+1, tm->tm_min, tm->tm_sec);
				tm = localtime(&cur_time);
				sprintf(new_msg, "%02d-%02d-%02d", tm->tm_mon+1, tm->tm_mday, tm->tm_year%100);
				break;
			case TO_DATE_MODE:
				tm = localtime(&timer);
				mode = DATE_MODE;
				sprintf(old_msg, "%02d-%02d-%02d", tm->tm_mon+1, tm->tm_mday, tm->tm_year%100);
				tm = localtime(&cur_time);
				sprintf(new_msg, "%02d-%02d-%02d", tm->tm_mon+1, tm->tm_mday, tm->tm_year%100);
				break;
			case DATE_MODE:
				tm = localtime(&timer);
				mode = TO_CLOCK_MODE;
				sprintf(old_msg, "%02d-%02d-%02d", tm->tm_mon+1, tm->tm_mday, tm->tm_year%100);
				tm = localtime(&cur_time);
				sprintf(new_msg, "%02d:%02d:%02d", army_time ? tm->tm_hour : (tm->tm_hour+11)%12+1, tm->tm_min, tm->tm_sec);
				break;
			case TO_CLOCK_MODE:
			case CLOCK_MODE:
				tm = localtime(&timer);
				mode = CLOCK_MODE;
				sprintf(old_msg, "%02d:%02d:%02d", army_time ? tm->tm_hour : (tm->tm_hour+11)%12+1, tm->tm_min, tm->tm_sec);
				tm = localtime(&cur_time);
				sprintf(new_msg, "%02d:%02d:%02d", army_time ? tm->tm_hour : (tm->tm_hour+11)%12+1, tm->tm_min, tm->tm_sec);
				break;
		}
	}
	PRINT(("finished update. strings:\n\t%s\n\t%s\n", old_msg, new_msg));

	//step through the characters in the message,
	//drawing each one to the offscreen bitmap
	char *dest = (char *)offscreen->Bits();
	long dest_width = offscreen->BytesPerRow();
	int offset = 0;

	for(int i=0; i<strlen(new_msg); i++) {
		PRINT(("drawing char %d\n", i));
		int old_index, new_index;

		new_index=indexFromChar(new_msg[i]);
		old_index=indexFromChar(old_msg[i]);

		PRINT(("index new, index old: %d,%d\n", new_index, old_index));

		for(int j=0; j<64; j++) {
			long old_x=0, new_x=0;
			long t1=-1, t2=-1;
			while(!(old_x == t1 && new_x == t2)) {
				t1=old_x;
				t2=new_x;
				old_x = nextPixOnLine(old_x, j, digits[old_index]);
				new_x = nextPixOnLine(new_x, j, digits[new_index]);
				if(new_x == 0 && old_x == 0)
					break;
				else if(new_x == 0)
					setPix(i*CHAR_WIDTH - offset + old_x, j,
							index_for_color(255*pct, 255*pct, 255*pct), offscreen);
				else if(old_x == 0)
					setPix(i*CHAR_WIDTH - offset + new_x, j,
							index_for_color(255*(1-pct), 255*(1-pct), 255*(1-pct)),
							offscreen);
				else
					setPix(i*CHAR_WIDTH - offset + new_x*pct + old_x*(1-pct), j,
							black, offscreen);
			}
		}
		if(new_index >= 10)
			offset += CHAR_WIDTH/2;
	}

	PRINT(("about to draw something\n"));

	if(Window() != NULL) {
		Window()->Lock();
		DrawBitmap(offscreen, B_ORIGIN);
		Window()->Unlock();
	}

	PRINT(("finished draw\n"));
}

//determine if there are any more colored pixels on a scanline
bool DaliClockView::morePixOnLine(long& t, long x, long y, BBitmap *bits) {
	t = nextPixOnLine(x, y, bits);
	return x != t;
}

//find the next colored pixel on a scanline, returning x if none is
//found
long DaliClockView::nextPixOnLine(long x, long y, BBitmap *bits) {
	for(long t = x+1; t<CHAR_WIDTH; t++)
		if(((char *)bits->Bits())[y*bits->BytesPerRow()+t] == 0)
			return t;

	return x;
}

//sets a pixel at (x,y) to value color
void DaliClockView::setPix(long x, long y, uchar color, BBitmap *bits) {
	((char *)bits->Bits())[y*bits->BytesPerRow()+x] = color;
}

void DaliClockView::MouseDown(BPoint point) {
	if(mode == CLOCK_MODE)
		mode = ALMOST_TO_DATE_MODE;
	else
		mode = TO_CLOCK_MODE;
}

long DaliClockView::_run_clock(void *data) {
	DaliClockView *clock_view = (DaliClockView *)data;
	while(TRUE) {
		clock_view->Draw(BRect());
		if(clock_view->mode == CLOCK_MODE)
			snooze(10.0 * 1000.0);
	}
	return 0;
}

int DaliClockView::indexFromChar(char ch) {
	switch(ch) {
		case ':':
			return 10;
			break;
		case '-':
			return 11;
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			return ch-'0';
			break;
		default:
			return 0;
	}
}

void DaliClockView::SetArmyTimeMode(bool mode) {
	army_time = mode;
}
