/*
 *  SineGenerator.cpp - Simple sine wave generator
 *
 *  Written by Christian Bauer, Public Domain
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <MediaKit.h>
#include <math.h>

#include "SliderBits.h"


// Sample frequency in Hz
const int SAMPLE_FREQ = 48000;

// Highest output frequency = SAMPLE_FREQ / DIVISOR
int DIVISOR = 4;


const char APP_SIGNATURE[] = "application/x-be-executable";

const ulong MSG_DIVISOR = 'divi';

const rgb_color light_color = {255, 255, 255, 0};
const rgb_color fill_color = {216, 216, 216, 0};
const rgb_color dark_color = {184, 184, 184, 0};

#define SLIDER_BACK_WIDTH	560
#define SLIDER_BACK_HEIGHT	15
#define BACK_COLOR			216


// Application object
class SineGenerator : public BApplication {
public:
	SineGenerator();
	virtual void ReadyToRun(void);
	virtual void AboutRequested(void);
};


// Slider view
class TSliderView : public BView {
public:
	TSliderView(BRect, char*);
	~TSliderView();

	virtual	void Draw(BRect);
	virtual void MouseDown(BPoint);

	void DrawSlider();
	void SetValue(float);
	float Value();

private:
	float fValue;
	BBitmap* fSlider;
	BBitmap* fKnob;
	BView* fOffView;
};


// Main window object
class MainWindow : public BWindow {
public:
	MainWindow();
	virtual bool QuitRequested(void);
	virtual void MessageReceived(BMessage *msg);

private:
	static bool stream_func(void *arg, char *buf, size_t count, void *header);
	void calc_buffer(int16 *buf, size_t count);

	TSliderView *the_slider;
	BSubscriber *the_sub;
	BDACStream *the_stream;
	bool in_stream;
};


/*
 *  Create application object and start it
 */

int main(int argc, char **argv)
{
	SineGenerator *the_app;

	the_app = new SineGenerator();
	if (the_app != NULL) {
		the_app->Run();
		delete the_app;
	}

	return 0;
}


/*
 *  Constructor
 */

SineGenerator::SineGenerator() : BApplication(APP_SIGNATURE) {}


/*
 *  Arguments processed
 */

void SineGenerator::ReadyToRun(void)
{
	new MainWindow();
}


/*
 *  About requested
 */

void SineGenerator::AboutRequested(void)
{
	char str[256];

	sprintf(str, "Sine generator by Christian Bauer\n<cbauer@iphcip1.physik.uni-mainz.de>\nPublic domain.");
	BAlert *the_alert = new BAlert("", str, "Neat");
	the_alert->Go();
}


/*
 *  Window creator
 */

MainWindow::MainWindow() : BWindow(BRect(0, 0, 599, 59), "SineGenerator", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	Lock();
	BRect b = Bounds();

	// Move window to right position
	MoveTo(80, 80);

	// Light gray background
	BView *top = new BView(BRect(0, 0, b.right, b.bottom), "", B_FOLLOW_NONE, B_WILL_DRAW);
	AddChild(top);
	top->SetViewColor(fill_color);

	// Frequency slider
	the_slider = new TSliderView(BRect(10, 10, 579, 39), "SliderView");
	top->AddChild(the_slider);

	// Divisor control
	BCheckBox *box = new BCheckBox(BRect(10, 40, 200, 55), "", "/10", new BMessage(MSG_DIVISOR));
	top->AddChild(box);
	box->SetViewColor(fill_color);

	// Create and set up audio subscriber
	in_stream = FALSE;
	the_sub = new BSubscriber("Sine Generator");
	the_stream = new BDACStream();
	if (the_sub->Subscribe(the_stream) == B_NO_ERROR) {
		the_sub->EnterStream(NULL, TRUE, this, stream_func, NULL, TRUE);
		the_stream->SetSamplingRate(SAMPLE_FREQ);
		in_stream = TRUE;
	} else
		be_app->PostMessage(B_QUIT_REQUESTED);

	// Show window
	Show();
	Unlock();
}


/*
 *  Quit requested, close all and quit program
 */

bool MainWindow::QuitRequested(void)
{
	// Delete subscriber
	if (in_stream) {
		the_sub->ExitStream(TRUE);
		in_stream = FALSE;
	}
	the_sub->Unsubscribe();
	delete the_sub;
	delete the_stream;

	// Quit program
	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}


/*
 *  Message received
 */

void MainWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case MSG_DIVISOR:
			if (DIVISOR != 4)
				DIVISOR = 4;
			else
				DIVISOR = 40;
			the_slider->SetValue(the_slider->Value());
			break;
		default:
			inherited::MessageReceived(msg);
			break;
	}
}


/*
 *  Stream function 
 */

bool MainWindow::stream_func(void *arg, char *buf, size_t count, void *header)
{
	((MainWindow *)arg)->calc_buffer((int16 *)buf, count);
	return TRUE;
}


/*
 *  Fill one audio buffer
 */

void MainWindow::calc_buffer(int16 *buf, size_t count)
{
	float freq = the_slider->Value() * SAMPLE_FREQ / DIVISOR;
	static float phi = 0.0;		// Phase angle

	count >>= 2;	// 16 bit stereo output, count is in bytes
	while (count--) {
		int16 data = (int16)(sin(phi) * 30000);
		*buf++ += data;
		*buf++ += data;
		phi += 2 * M_PI * freq / SAMPLE_FREQ;
		phi = fmod(phi, 2 * M_PI);
	}
}


//====================================================================

TSliderView::TSliderView(BRect rect, char *title)
			:BView(rect, title, B_FOLLOW_ALL, B_WILL_DRAW)
{
	BRect	r;

	fValue = 1.00;

	// load in our canned slider knob bitmap
	r.Set(0, 0, ((KNOB_WIDTH + 7) & 0xfff8) - 1, KNOB_HEIGHT);
	fKnob = new BBitmap(r, B_COLOR_8_BIT);
	fKnob->SetBits((char*)knob, fKnob->BitsLength(), 0, B_COLOR_8_BIT);

	// create our offscreen drawing environment
	r = Bounds();
	fSlider = new BBitmap(r, B_COLOR_8_BIT, TRUE);
	fSlider->AddChild(fOffView = new BView(r, "", B_FOLLOW_ALL, B_WILL_DRAW));
}

//--------------------------------------------------------------------

TSliderView::~TSliderView()
{
	delete fSlider;
	delete fKnob;
}

//--------------------------------------------------------------------

void TSliderView::Draw(BRect where)
{
	DrawSlider();
}

//--------------------------------------------------------------------

void TSliderView::MouseDown(BPoint thePoint)
{
	ulong	buttons;
	float	old_value;
	float	temp;
	BPoint	where;
	BRect	r;

	r.left = ((SLIDER_BACK_WIDTH - KNOB_WIDTH - 4.0) * fValue) + 2.0;
	r.top = 0.0;
	r.right = r.left + KNOB_WIDTH;
	r.bottom = KNOB_HEIGHT;

	if (!r.Contains(thePoint)) {
		temp = (thePoint.x / (SLIDER_BACK_WIDTH - KNOB_WIDTH - 4.0)) -
							 ((KNOB_WIDTH / 2.0) / SLIDER_BACK_WIDTH);
		if (temp < 0.00)
			temp = 0.00;
		if (temp > 1.00)
			temp = 1.00;
		if (temp != fValue)
			SetValue(temp);
	}
	old_value = fValue;

	do {
		GetMouse(&where, &buttons);
		temp = old_value + ((where.x - thePoint.x) /
							(SLIDER_BACK_WIDTH - KNOB_WIDTH - 2.0));
		if (temp < 0.00)
			temp = 0.00;
		if (temp > 1.00)
			temp = 1.00;
		if (temp != fValue) {
			SetValue(temp);
		}
		snooze(500.0);	// be a good citizen
	} while (buttons);
}

//--------------------------------------------------------------------

void TSliderView::DrawSlider()
{
	BRect	sr;
	BRect	dr;

	fSlider->Lock();
	sr = fOffView->Bounds();

	// slider background
	fOffView->SetHighColor(BACK_COLOR, BACK_COLOR, BACK_COLOR);
	fOffView->FillRect(sr);
	fOffView->SetHighColor(176, 176, 176);
	fOffView->StrokeLine(BPoint(0, 5), BPoint(0, 5));
	fOffView->StrokeLine(BPoint(0, 10), BPoint(0, 10));
	fOffView->StrokeLine(BPoint(SLIDER_BACK_WIDTH, 5), BPoint(SLIDER_BACK_WIDTH, 5));
	fOffView->SetHighColor(255, 255, 255);
	fOffView->StrokeLine(BPoint(1, 10), BPoint(SLIDER_BACK_WIDTH, 10));
	fOffView->StrokeLine(BPoint(SLIDER_BACK_WIDTH, 9), BPoint(SLIDER_BACK_WIDTH, 6));
	fOffView->SetHighColor(144, 144, 144);
	fOffView->StrokeLine(BPoint(0, 6), BPoint (0, 9));
	fOffView->StrokeLine(BPoint(1, 5), BPoint(SLIDER_BACK_WIDTH - 1, 5));
	fOffView->SetHighColor(0, 0, 0);
	fOffView->StrokeLine(BPoint(1, 6), BPoint(SLIDER_BACK_WIDTH - 1, 6));
	fOffView->StrokeLine(BPoint(1, 7), BPoint(SLIDER_BACK_WIDTH - 1, 7));
	fOffView->StrokeLine(BPoint(1, 8), BPoint(1, 9));
	fOffView->StrokeLine(BPoint(SLIDER_BACK_WIDTH - 1, 8), BPoint(SLIDER_BACK_WIDTH - 1, 9));
	fOffView->SetHighColor(64, 64, 64);
	sr.Set(2, 8, SLIDER_BACK_WIDTH - 2, 9);
	fOffView->FillRect(sr);

	// knob
	sr.Set(0, 0, KNOB_WIDTH, KNOB_HEIGHT);
	dr.left = ((SLIDER_BACK_WIDTH - KNOB_WIDTH - 4.0) * fValue) + 2.0;
	dr.top = 0;
	dr.right = dr.left + KNOB_WIDTH;
	dr.bottom = dr.top + KNOB_HEIGHT;
	fOffView->DrawBitmap(fKnob, sr, dr);
	fOffView->SetHighColor(0, 0, 0);

	fOffView->Sync();	// make sure offscreen drawing completes

	// copy offscreen to screen
	DrawBitmap(fSlider, BPoint(0, 0));

	// Show frequency
	char str[80];
	sprintf(str, "%ld Hz         ", (long)(fValue * SAMPLE_FREQ / DIVISOR));
	DrawString(str, BPoint(0, 25));

	fSlider->Unlock();
}

//--------------------------------------------------------------------

void TSliderView::SetValue(float value)
{
	fValue = value;
	DrawSlider();
}

//--------------------------------------------------------------------

float TSliderView::Value()
{
	return (fValue);
}
