/*
 *  main.cpp - Main program
 *
 *  SIDPlayer (C) 1996 Christian Bauer
 */

#include <AppKit.h>
#include <InterfaceKit.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "CPU.h"
#include "SID.h"

#include "panprop.h"
#include "volprop.h"
#include "SliderBits.h"


const ulong APP_SIGNATURE = 'SidP';
const rgb_color light_color = {255, 255, 255, 0};
const rgb_color fill_color = {216, 216, 216, 0};
const rgb_color dark_color = {184, 184, 184, 0};

const ulong MSG_PLAY_PAUSE = 'plpa';
const ulong MSG_STOP = 'stop';
const ulong MSG_NEXT = 'next';
const ulong MSG_PREV = 'prev';
const ulong MSG_LOAD_MODULE = 'load';
const ulong MSG_FILTERS = 'filt';
const ulong MSG_FASTER = 'fast';
const ulong MSG_SLOWER = 'slow';


// Application object
class SIDPlayer : public BApplication {
public:
	SIDPlayer();
	virtual void ArgvReceived(int argc, char **argv);
	virtual void RefsReceived(BMessage *msg);
	virtual void ReadyToRun(void);
	virtual void AboutRequested(void);
};


// Main window object
class MainWindow : public BWindow {
public:
	MainWindow();
	virtual bool QuitRequested(void);
	virtual void MessageReceived(BMessage *msg);

private:
	BStringView *make_name_display(BRect frame, char *label_text, BView *parent);
	bool load_psid(char *filename);
	void select_song(int num);

	UBYTE *the_ram;		// Pointer to 64K RAM
	MOS6510 *the_cpu;	// Pointer to 6510
	MOS6581 *the_sid;	// Pointer to 6581

	int current_song;			// Song currently playing
	int number_of_songs;		// Number of songs in module
	UWORD init_adr, play_adr;	// C64 init/play routine addresses
	ULONG song_speeds;			// Speed flags
	int current_freq;			// Current replay frequency in Hz
	char module_name[32];		// Module name
	char author[32];			// Author
	char copyright[32];			// Copyright info

	BStringView *name_view;
	BStringView *author_view;
	BStringView *copyright_view;
	BStringView *position_view;
	BCheckBox *filter_view;
};


// Top view object (handles drag&drop)
class TopView : public BView {
public:
	TopView(BRect frame, const char *name, ulong resizingMode, ulong flags);
	virtual void MessageReceived(BMessage *msg);
	virtual void KeyDown(ulong aChar);
};


// Buttons
class PlayPauseButton : public BButton {
public:
	PlayPauseButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class StopButton : public BButton {
public:
	StopButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class NextButton : public BButton {
public:
	NextButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};

class PrevButton : public BButton {
public:
	PrevButton(BRect frame, BMessage *msg);
	virtual void Draw(BRect update);
};


// PSID file header (big-endian)
struct PSIDHeader {
	ULONG	id;				// 'PSID'
	UWORD	version;		// Version 1 or 2
	UWORD	length;			// Length of header
	UWORD	start;			// C64 load address
	UWORD	init;			// C64 address of init routine
	UWORD	main;			// C64 address of replay routine
	UWORD	number;			// Number of subsongs
	UWORD	defsong;		// Main subsong number (1..255)
	UWORD	speedhi;		// Speed flags (1 bit/song)
	UWORD	speedlo;
	UBYTE	name[32];		// Module name
	UBYTE	author[32];		// Author
	UBYTE	copyright[32];	// Copyright info
	UWORD	flags;			// Flags (only in version 2 header)
	ULONG	reserved;
};


// Global: path name of currently loaded PSID module
char ModuleName[256] = "";


/*
 *  Create application object and start it
 */

int main(int argc, char **argv)
{	
	SIDPlayer *the_app;

	srand(system_time());

	the_app = new SIDPlayer();
	the_app->Run();
	delete the_app;

	return 0;
}


/*
 *  Constructor
 */

SIDPlayer::SIDPlayer() : BApplication(APP_SIGNATURE) {}


/*
 *  Shell arguments received
 */

void SIDPlayer::ArgvReceived(int argc, char **argv)
{
	if (argc >= 2)
		strncpy(ModuleName, argv[1], 255);
	else
		printf("Usage: %s [name of PSID file]\n", argv[0]);
}


/*
 *  Browser arguments received
 */

void SIDPlayer::RefsReceived(BMessage *msg)
{
	record_ref ref = msg->FindRef("refs");
	if (ref.database >= 0 && ref.record >= 0)
		if (does_ref_conform(ref, "File")) {
			BFile *file = new BFile(ref);
			file->GetPath(ModuleName, 255);
			delete file;
		}
}


/*
 *  Arguments processed
 */

void SIDPlayer::ReadyToRun(void)
{
	// Create player window
	new MainWindow();
}


/*
 *  About requested
 */

void SIDPlayer::AboutRequested(void) {
	char str[256];

	sprintf(str, "SIDPlayer V2.2 by Christian Bauer\n<cbauer@iphcip1.physik.uni-mainz.de>\n"
				"\nStereo support by Marco Nelissen <marcone@xs4all.nl>\n"
				"\nFreely distributable.");
	BAlert *the_alert = new BAlert("", str, "OK");
	the_alert->Go();
}


/*
 *  Window creator
 */

MainWindow::MainWindow() : BWindow(BRect(0, 0, 480, 94), "SIDPlayer", B_TITLED_WINDOW, B_NOT_ZOOMABLE)
{
	Lock();
	BRect b = Bounds();

	// Create objects for emulation
	the_ram = new UBYTE[0x10000];
	the_cpu = new MOS6510();
	the_cpu->RAM = the_ram;
	the_sid = new MOS6581();
	the_cpu->TheSID = the_sid;
	the_sid->TheCPU = the_cpu;
	the_sid->TheRAM = the_ram;
	current_freq = 50;

	// Move window to right position
	MoveTo(80, 80);

	// Light gray background
	TopView *top = new TopView(BRect(0, 0, b.right, b.bottom), "", B_FOLLOW_NONE, B_WILL_DRAW);
	AddChild(top);
	top->SetViewColor(fill_color);

	// Create slider knobs
	BBitmap *myknob;
	BBitmap *myknob2;
	BRect kr;

	kr.Set(0, 0, KNOB_WIDTH-1, KNOB_HEIGHT-1);
	myknob = new BBitmap(kr, B_COLOR_8_BIT);
	myknob->SetBits((char*)knob, myknob->BitsLength(), 0, B_COLOR_8_BIT);

	kr.Set(0, 0, KNOB_WIDTH2-1, KNOB_HEIGHT2-1);
	myknob2 = new BBitmap(kr, B_COLOR_8_BIT);
	myknob2->SetBits((char*)knob2, myknob2->BitsLength(), 0, B_COLOR_8_BIT);
	
	SetSizeLimits(284, 480, 94, 94);

	// Name/author/copyright display
	name_view = make_name_display(BRect(0, 5, 279, 20), "Name", top);
	author_view = make_name_display(BRect(0, 25, 279, 40), "Author", top);
	copyright_view = make_name_display(BRect(0, 45, 279, 60), "Copyright", top);

	// Buttons
	top->AddChild(new PlayPauseButton(BRect(6, 66, 46, 90), new BMessage(MSG_PLAY_PAUSE)));
	top->AddChild(new StopButton(BRect(47, 66, 87, 90), new BMessage(MSG_STOP)));
	top->AddChild(new PrevButton(BRect(88, 66, 128, 90), new BMessage(MSG_PREV)));
	top->AddChild(new NextButton(BRect(129, 66, 169, 90), new BMessage(MSG_NEXT)));
	top->AddChild(new BButton(BRect(170, 66, 184, 90), NULL, "+", new BMessage(MSG_FASTER)));
	top->AddChild(new BButton(BRect(185, 66, 199, 90), NULL, "-", new BMessage(MSG_SLOWER)));

	// Position indicator
	top->AddChild(position_view = new BStringView(BRect(210, 64, 279, 76), "", ""));
	position_view->SetViewColor(fill_color);
	position_view->SetLowColor(fill_color);
	position_view->SetFontName("Emily");
	position_view->SetFontSize(12);

	// Filter enable/disable checkbox
	top->AddChild(filter_view = new BCheckBox(BRect(210, 75, 279, 90), "", "Filters", new BMessage(MSG_FILTERS)));
	filter_view->SetViewColor(fill_color);
	filter_view->SetLowColor(fill_color);
	filter_view->SetFontName("Erich");
	filter_view->SetFontSize(9);
	filter_view->SetValue(B_CONTROL_ON);

	// Volume/panning sliders
	BStringView *label;
	top->AddChild(label = new BStringView(BRect(284, 1, 384, 14), "", "Panning"));
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_CENTER);
	label->SetFontName("Emily");
	label->SetFontSize(12);

	top->AddChild(the_sid->panprop[0] = new panprop(0.01, 0.374, -1, -1, myknob));
	the_sid->panprop[0]->layoutprefs();
	the_sid->panprop[0]->layout(BRect(284, 15, 384, 15+KNOB_HEIGHT));
	top->AddChild(the_sid->panprop[1] = new panprop(0.01, 0.5, -1, -1, myknob));
	the_sid->panprop[1]->layoutprefs();
	the_sid->panprop[1]->layout(BRect(284, 35, 384, 35+KNOB_HEIGHT));
	top->AddChild(the_sid->panprop[2] = new panprop(0.01, 0.627, -1, -1, myknob));
	the_sid->panprop[2]->layoutprefs();
	the_sid->panprop[2]->layout(BRect(284, 55, 384, 55+KNOB_HEIGHT));
	top->AddChild(the_sid->panprop[3] = new panprop(0.01, 0.5, -1, -1, myknob));
	the_sid->panprop[3]->layoutprefs();
	the_sid->panprop[3]->layout(BRect(284, 75, 384, 75+KNOB_HEIGHT));

	top->AddChild(label = new BStringView(BRect(394, 1, 454+KNOB_WIDTH, 14), "", "Volume"));
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_CENTER);
	label->SetFontName("Emily");
	label->SetFontSize(12);

	top->AddChild(the_sid->volprop[0] = new volprop(-1, -1, 0.01, 1, myknob2));
	the_sid->volprop[0]->layoutprefs();
	the_sid->volprop[0]->layout(BRect(394, 15, 394+KNOB_WIDTH, 75+KNOB_HEIGHT));
	top->AddChild(the_sid->volprop[1] = new volprop(-1, -1, 0.01, 1, myknob2));
	the_sid->volprop[1]->layoutprefs();
	the_sid->volprop[1]->layout(BRect(414, 15, 414+KNOB_WIDTH, 75+KNOB_HEIGHT));
	top->AddChild(the_sid->volprop[2] = new volprop(-1, -1, 0.01, 1, myknob2));
	the_sid->volprop[2]->layoutprefs();
	the_sid->volprop[2]->layout(BRect(434, 15, 444+KNOB_WIDTH, 75+KNOB_HEIGHT));
	top->AddChild(the_sid->volprop[3] = new volprop(-1, -1, 0.01, 1, myknob2));
	the_sid->volprop[3]->layoutprefs();
	the_sid->volprop[3]->layout(BRect(454, 15, 454+KNOB_WIDTH, 75+KNOB_HEIGHT));

	// Show window
	top->MakeFocus();
	Show();
	Unlock();

	// Load PSID module
	PostMessage(MSG_LOAD_MODULE);
}


/*
 *  Create name display field
 */

BStringView *MainWindow::make_name_display(BRect frame, char *label_text, BView *parent)
{
	// Label to the left of the display field
	BRect label_rect = frame;
	label_rect.right = label_rect.left + 65;

	BStringView *label = new BStringView(label_rect, "", label_text);
	parent->AddChild(label);
	label->SetViewColor(fill_color);
	label->SetLowColor(fill_color);
	label->SetAlignment(B_ALIGN_RIGHT);
	label->SetFontName("Emily");
	label->SetFontSize(12);

	// Box around display field
	BRect frame_rect = frame;
	frame_rect.left += 70;

	BBox *box = new BBox(frame_rect);
	parent->AddChild(box);
	box->SetViewColor(fill_color);
	box->SetLowColor(fill_color);

	// The display field
	BRect textview_rect = frame_rect;
	textview_rect.OffsetTo(0, 1);
	textview_rect.InsetBy(4, 2);

	BStringView *text = new BStringView(textview_rect, "", "");
	box->AddChild(text);
	text->SetViewColor(fill_color);
	text->SetLowColor(fill_color);
	text->SetSymbolSet("ISO 8859/1 Latin 1");
	text->SetFontName("Erich");
	text->SetFontSize(9);

	return text;
}


/*
 *  Quit requested, close all and quit program
 */

bool MainWindow::QuitRequested(void)
{
	// Delete objects
	if (the_sid != NULL)
		delete the_sid;
	if (the_cpu != NULL)
		delete the_cpu;
	if (the_ram != NULL)
		delete[] the_ram;

	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}


/*
 *  Message received
 */

void MainWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {

		case MSG_PLAY_PAUSE:
			if (the_sid->IsPaused())
				the_sid->ResumeSound();
			else
				the_sid->PauseSound();
			break;

		case MSG_STOP:
			the_sid->PauseSound();
			the_cpu->Emulate(init_adr, current_song, 0, 0);
			break;

		case MSG_NEXT:
			if (current_song < number_of_songs-1)
				select_song(current_song + 1);
			break;

		case MSG_PREV:
			if (current_song > 0)
				select_song(current_song - 1);
			break;

		case MSG_LOAD_MODULE:
			the_sid->PauseSound();

			if (load_psid(ModuleName)) {
				Lock();
				name_view->SetText(module_name);
				author_view->SetText(author);
				copyright_view->SetText(copyright);
				Unlock();

				select_song(current_song);

				if (play_adr)
					the_sid->PlayAdr = play_adr;
				else
					if (the_ram[1] & 2)
						the_sid->PlayAdr = (the_ram[0x0315] << 8) | the_ram[0x0314];
					else
						the_sid->PlayAdr = (the_ram[0xffff] << 8) | the_ram[0xfffe];

				the_sid->ResumeSound();
			}
			break;

		case MSG_FILTERS:
			the_sid->EnableFilters(filter_view->Value());
			break;

		case MSG_FASTER:
			current_freq += 5;
			the_sid->SetReplayFreq(current_freq);
			break;

		case MSG_SLOWER:
			if (current_freq > 5) {
				current_freq -= 5;
				the_sid->SetReplayFreq(current_freq);
			}
			break;

		default:
			BWindow::MessageReceived(msg);
			break;
	}
}


/*
 *  Load PSID module
 */

bool MainWindow::load_psid(char *filename)
{
	// Open PSID file
	FILE *file = fopen(filename, "rb");
	if (file) {

		// Read header
		PSIDHeader header;
		if (fread(&header, 1, sizeof(PSIDHeader), file) == sizeof(PSIDHeader)) {

			// Check header ID and version
			if (header.id == 'PSID' && (header.version == 1 || header.version == 2)) {

				// OK, seek to start of module data
				fseek(file, header.length, SEEK_SET);

				// Extract data
				current_song = header.defsong;
				if (current_song)
					current_song--;

				number_of_songs = header.number;
				if (!number_of_songs)
					number_of_songs = 1;
				if (current_song >= number_of_songs)
					current_song = 0;

				init_adr = header.init;
				play_adr = header.main;

				song_speeds = (header.speedhi << 16) | header.speedlo;

				memcpy(module_name, header.name, 32);
				memcpy(author, header.author, 32);
				memcpy(copyright, header.copyright, 32);
				module_name[31] = 0;
				author[31] = 0;
				copyright[31] = 0;

				// Initialize memory
				memset(the_ram, 0, 0xe000);
				memset(the_ram + 0xe000, 0x40, 0x2000);
				the_ram[1] = 7;

				// Load module into memory
				UWORD load_adr = header.start;
				if (!load_adr) {
					UBYTE hi, lo;
					fread(&lo, 1, 1, file);
					fread(&hi, 1, 1, file);
					load_adr = (hi << 8) | lo;
				}
				fread(the_ram + load_adr, 1, 0x10000 - load_adr, file);
				fclose(file);

				if (!init_adr)
					init_adr = load_adr;

				return TRUE;
			}
		}
		fclose(file);
	}
	return FALSE;
}


/*
 *  Select a subsong for playing
 */

void MainWindow::select_song(int num)
{
	bool need_resume = FALSE;
	
	// Pause sound if running
	if (!the_sid->IsPaused()) {
		need_resume = TRUE;
		the_sid->PauseSound();
	}

	current_song = num;
	the_sid->Reset();

	// Set replay frequency
	if (num < 32)
		current_freq = song_speeds & (1 << num) ? 60 : 50;
	else
		current_freq = 50;
	the_sid->SetReplayFreq(current_freq);

	the_cpu->Emulate(init_adr, current_song, 0, 0);

	// Resume sound if it was running
	if (need_resume)
		the_sid->ResumeSound();

	// Update position indicator
	char str[16];
	sprintf(str, "Song %d/%d", current_song + 1, number_of_songs);
	position_view->SetText(str);
}


/*
 *  TopView handles dropped messages (load new PSID module) and keypresses
 */

TopView::TopView(BRect frame, const char *name, ulong resizingMode, ulong flags)
 : BView(frame, name, resizingMode, flags) {};

void TopView::MessageReceived(BMessage *msg)
{
	if (msg->what == B_SIMPLE_DATA) {
		record_ref ref = msg->FindRef("refs");
		if (ref.database >= 0 && ref.record >= 0)
			if (does_ref_conform(ref, "File")) {
				BFile *file = new BFile(ref);
				file->GetPath(ModuleName, 255);
				delete file;
				Window()->PostMessage(MSG_LOAD_MODULE);
			}
	} else
		BView::MessageReceived(msg);
}

void TopView::KeyDown(ulong aChar)
{
	switch (aChar) {
		case 'p':
		case 'P':
			Window()->PostMessage(MSG_PLAY_PAUSE);
			break;

		case B_ESCAPE:
		case B_SPACE:
		case 's':
		case 'S':
			Window()->PostMessage(MSG_STOP);
			break;

		case B_LEFT_ARROW:
			Window()->PostMessage(MSG_PREV);
			break;

		case B_RIGHT_ARROW:
		case 'n':
		case 'N':
			Window()->PostMessage(MSG_NEXT);
			break;

		case B_UP_ARROW:
			Window()->PostMessage(MSG_FASTER);
			break;

		case B_DOWN_ARROW:
			Window()->PostMessage(MSG_SLOWER);
			break;
	}
}


/*
 *  Play/Pause button
 */

PlayPauseButton::PlayPauseButton(BRect frame, BMessage *msg) : BButton(frame, "", "", msg) {};

void PlayPauseButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw play/pause image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 18, 16));
	FillTriangle(BPoint(21, 8), BPoint(21, 16), BPoint(25, 12));
}


/*
 *  Stop button
 */

StopButton::StopButton(BRect frame, BMessage *msg) : BButton(frame, "", "", msg) {};

void StopButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw stop image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(16, 8, 25, 16));
}


/*
 *  "Next" button
 */

NextButton::NextButton(BRect frame, BMessage *msg) : BButton(frame, "", "", msg) {};

void NextButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw "next" image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillTriangle(BPoint(17, 8), BPoint(17, 16), BPoint(21, 12));
	FillRect(BRect(22, 8, 24, 16));
}


/*
 *  "Prev" button
 */

PrevButton::PrevButton(BRect frame, BMessage *msg) : BButton(frame, "", "", msg) {};

void PrevButton::Draw(BRect update)
{
	// First draw normal button
	BButton::Draw(update);

	// Then draw "prev" image on top of it
	if (Value())
		SetHighColor(255, 255, 255);
	else
		SetHighColor(0, 0, 0);

	FillRect(BRect(17, 8, 19, 16));
	FillTriangle(BPoint(24, 8), BPoint(24, 16), BPoint(20, 12));
}
