/* CGEN generic disassembler support code.

   Copyright (C) 1996, 1997, 1998 Free Software Foundation, Inc.

   This file is part of the GNU Binutils and GDB, the GNU debugger.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "sysdep.h"
#include <stdio.h>
#include "ansidecl.h"
#include "libiberty.h"
#include "bfd.h"
#include "symcat.h"
#include "opcode/cgen.h"

/* This is not published as part of the public interface so we don't
   declare this in cgen.h.  */
extern CGEN_OPCODE_TABLE * cgen_current_opcode_table;

/* Disassembler instruction hash table.  */
static CGEN_INSN_LIST ** dis_hash_table;
static CGEN_INSN_LIST * dis_hash_table_entries;

void
cgen_dis_init ()
{
  if (dis_hash_table)
    {
      free (dis_hash_table);
      dis_hash_table = NULL;
    }
}

/* Subroutine of build_dis_hash_table to add INSNS to the hash table.

   COUNT is the number of elements in INSNS.
   ENTSIZE is sizeof (CGEN_INSN) for the target.
   This is a bit tricky as the size of the attribute member of CGEN_INSN
   is variable among architectures.  This code could be moved to
   cgen-asm.in, but I prefer to keep it here for now.
   OTABLE is the opcode table.
   HTABLE points to the hash table.
   HENTBUF is a pointer to sufficiently large buffer of hash entries.
   The result is a pointer to the next entry to use.

   The table is scanned backwards as additions are made to the front of the
   list and we want earlier ones to be prefered.  */

static CGEN_INSN_LIST *
hash_insn_array (insns, count, entsize, otable, htable, hentbuf)
     const CGEN_INSN * insns;
     int count;
     int entsize;
     const CGEN_OPCODE_TABLE * otable;
     CGEN_INSN_LIST ** htable;
     CGEN_INSN_LIST * hentbuf;
{
  int bigend = cgen_current_endian == CGEN_ENDIAN_BIG;
  CGEN_INSN * insn;

  for (insn = (CGEN_INSN *) ((char *) insns + entsize * (count - 1));
       insn >= insns;
       insn = (CGEN_INSN *) ((char *) insn - entsize), ++ hentbuf)
    {
      unsigned int hash;
      char buf [4];
      unsigned long value;

      if (! (*otable->dis_hash_p) (insn))
	continue;

      /* We don't know whether the target uses the buffer or the base insn
	 to hash on, so set both up.  */

      value = CGEN_INSN_VALUE (insn);
      switch (CGEN_INSN_MASK_BITSIZE (insn))
	{
	case 8:
	  buf[0] = value;
	  break;
	case 16:
	  if (bigend)
	    bfd_putb16 ((bfd_vma) value, buf);
	  else
	    bfd_putl16 ((bfd_vma) value, buf);
	  break;
	case 32:
	  if (bigend)
	    bfd_putb32 ((bfd_vma) value, buf);
	  else
	    bfd_putl32 ((bfd_vma) value, buf);
	  break;
	default:
	  abort ();
	}

      hash = otable->dis_hash (buf, value);
      hentbuf->next = htable [hash];
      hentbuf->insn = insn;
      htable [hash] = hentbuf;
    }

  return hentbuf;
}

/* Subroutine of build_dis_hash_table to add INSNS to the hash table.
   This function is identical to hash_insn_array except the insns are
   in a list.  */

static CGEN_INSN_LIST *
hash_insn_list (insns, otable, htable, hentbuf)
     const CGEN_INSN_LIST * insns;
     const CGEN_OPCODE_TABLE * otable;
     CGEN_INSN_LIST ** htable;
     CGEN_INSN_LIST * hentbuf;
{
  int bigend = cgen_current_endian == CGEN_ENDIAN_BIG;
  const CGEN_INSN_LIST * ilist;

  for (ilist = insns; ilist != NULL; ilist = ilist->next, ++ hentbuf)
    {
      unsigned int hash;
      char buf [4];
      unsigned long value;

      if (! (*otable->dis_hash_p) (ilist->insn))
	continue;

      /* We don't know whether the target uses the buffer or the base insn
	 to hash on, so set both up.  */

      value = CGEN_INSN_VALUE (ilist->insn);
      switch (CGEN_INSN_MASK_BITSIZE (ilist->insn))
	{
	case 8:
	  buf[0] = value;
	  break;
	case 16:
	  if (bigend)
	    bfd_putb16 ((bfd_vma) value, buf);
	  else
	    bfd_putl16 ((bfd_vma) value, buf);
	  break;
	case 32:
	  if (bigend)
	    bfd_putb32 ((bfd_vma) value, buf);
	  else
	    bfd_putl32 ((bfd_vma) value, buf);
	  break;
	default:
	  abort ();
	}

      hash = otable->dis_hash (buf, value);
      hentbuf->next = htable [hash];
      hentbuf->insn = ilist->insn;
      htable [hash] = hentbuf;
    }

  return hentbuf;
}

/* Build the disassembler instruction hash table.  */

static void
build_dis_hash_table ()
{
  int count = cgen_insn_count () + cgen_macro_insn_count ();
  CGEN_OPCODE_TABLE * opcode_table = cgen_current_opcode_table;
  CGEN_INSN_TABLE * insn_table = opcode_table->insn_table;
  CGEN_INSN_TABLE * macro_insn_table = opcode_table->macro_insn_table;
  unsigned int hash_size = opcode_table->dis_hash_table_size;
  CGEN_INSN_LIST * hash_entry_buf;

  /* The space allocated for the hash table consists of two parts:
     the hash table and the hash lists.  */

  dis_hash_table = (CGEN_INSN_LIST **)
    xmalloc (hash_size * sizeof (CGEN_INSN_LIST *));
  memset (dis_hash_table, 0, hash_size * sizeof (CGEN_INSN_LIST *));
  dis_hash_table_entries = hash_entry_buf =
    (CGEN_INSN_LIST *) xmalloc ( + count * sizeof (CGEN_INSN_LIST));

  /* Add compiled in insns.
     Don't include the first one as it is a reserved entry.  */
  /* ??? It was the end of all hash chains, and also the special
     "illegal insn" marker.  May be able to do it differently now.  */

  hash_entry_buf = hash_insn_array ((CGEN_INSN *) ((char *) insn_table->init_entries
						   + insn_table->entry_size),
				    insn_table->num_init_entries - 1,
				    insn_table->entry_size,
				    opcode_table, dis_hash_table, hash_entry_buf);

  /* Add compiled in macro-insns.  */

  hash_entry_buf = hash_insn_array (macro_insn_table->init_entries,
				    macro_insn_table->num_init_entries,
				    macro_insn_table->entry_size,
				    opcode_table, dis_hash_table, hash_entry_buf);

  /* Add runtime added insns.
     Later added insns will be prefered over earlier ones.  */

  hash_entry_buf = hash_insn_list (insn_table->new_entries, opcode_table,
				   dis_hash_table, hash_entry_buf);

  /* Add runtime added macro-insns.  */

  hash_insn_list (macro_insn_table->new_entries,
		  opcode_table, dis_hash_table, hash_entry_buf);
}

/* Return the first entry in the hash list for INSN.  */

CGEN_INSN_LIST *
cgen_dis_lookup_insn (buf, value)
     const char * buf;
     unsigned long value;
{
  unsigned int hash;

  if (dis_hash_table == NULL)
    build_dis_hash_table ();

  hash = cgen_current_opcode_table->dis_hash (buf, value);

  return dis_hash_table [hash];
}
