/*
 *	Read PDP-11 file system.  Original version written
 *	by Rob Pike.  Modified Jan 81 by Mark Bartelt to
 *	support both V6 and V7 file systems (by combining
 *	code from two programs), and to permit PDP-11 to
 *	read VAX filesystems.
 *
 *	Modified [4-Nov-81, MLB] to eliminate the old file
 *	size limit (70656 bytes) associated with V7 files.
 *	Files of any size may now be read.  The 917504-byte
 *	limit for V6 files still exists for the time being.
 *
 *	hacked jan 82 by norman wilson:
 *	- implement ls -l (lots of work; involved
 *	  removing many global variables to be able
 *	  to look at two inodes at once!)
 *	- let output file for cp be a directory (as in normal cp)
 *	- clean up some cpu dependencies
 *
 *	Lots more hacking April/May 1982 by MLB:
 *	- Added code to permit use under VMS, using the
 *	  EUNICE C compiler and LIBC.OLB
 *	- Fixed a few minor bugs
 *	- Got rid of file size limit for V6 files (but
 *	  this code has NOT been extensively tested!!!)
 *
 *	A bit more hacking, February 1983 by MLB:
 *	- Added support for Berkeley UNIX 4.0/4.1 bsd filesystems
 *	- In process of aforementioned, parameterized all sorts of
 *	  things that should have been long ago .....
 *	- Moved definitions of the `struct fs' things into a .h file
 */

/*
 *  To compile and run under VMS using EUNICE, add the following line:
 *	#define	VMS	1
 */

/*
 *  Version of <sys/types.h>
 *  with longwords set up for swapping
 *  because VAX and 11 longs are different
 */
#if vax
typedef struct {unsigned short l_high; unsigned short l_low;} SLONG;
#else
typedef struct {unsigned unsigned short l_low; short l_high;} SLONG;
#endif vax
typedef		unsigned short	ushort;
typedef	struct { int r[1]; } *	physadr;
typedef		SLONG		daddr_t;
typedef		char *		caddr_t;
typedef		ushort		ino_t;
typedef		SLONG		time_t;
typedef		short		label_t[6];
typedef		short		dev_t;
typedef		SLONG		off_t;
typedef		SLONG		paddr_t;

#define	major(x)	(int)((unsigned)(x)>>8)
#define	minor(x)	(int)((x)&0377)
#define	makedev(x,y)	((dev_t)(((x)<<8)|(y)))

#include <sys/ino.h>
#include <sys/dir.h>
#include <stdio.h>
#include <pwd.h>
#include <sys/stat.h>
#include "getunix.h"

/*
 *  Warning: because of changed types.h stuff,
 *  struct stat is NOT correct for the machine
 *  we're on, if we're on a VAX reading a PDP-11,
 *  filesystem, or vice-versa.
 */

#define	V6	0
#define	V7	1
#define	V32	2
#define	S5P	3
#define	S5V	4

int	fstype = S5V;	/* Default file system type */

#define	PDP	0
#define	VAX	1

#define	S6IFMT	060000
#define	S6IFREG	000000
#define	S6IFDIR	040000
#define	S6IFBLK	060000
#define	S6IFCHR	020000
#define	S6IFIFO	(-1)	/* (To guarantee no match, ever) */
#define	S6IFMPC	(-1)
#define	S6IFMPB	(-1)

#define	S6LARGE	010000

#ifndef	S_IFIFO
#define	S_IFIFO	(-1)
#endif	S_IFIFO

struct s_modes {
	ushort	fs_ifmt;
	ushort	fs_ifreg;
	ushort	fs_ifdir;
	ushort	fs_ifblk;
	ushort	fs_ifchr;
	ushort	fs_ififo;
	ushort	fs_ifmpc;
	ushort	fs_ifmpb;
};

struct s_modes	v6modes = {
	S6IFMT, S6IFREG, S6IFDIR, S6IFBLK, S6IFCHR, S6IFIFO, S6IFMPC, S6IFMPB
};

struct s_modes	v7modes = {
	S_IFMT, S_IFREG, S_IFDIR, S_IFBLK, S_IFCHR, S_IFIFO, S_IFMPC, S_IFMPB
};

struct fs {
	struct fstab	*fs_fstbp;
	ino_t		fs_rootino;
	int		fs_bytpino;
	int		fs_bufsize;
	int		fs_dirlim;
	int		fs_cpu;
	struct s_modes	*fs_smp;
} fs[] = {
	 { V6fs,  1, 32,  512,  0, PDP, &v6modes },
	 { V7fs,  2, 64,  512, 10, PDP, &v7modes },
	 { V32fs, 2, 64,  512, 10, VAX, &v7modes },
	 { S5Pfs, 2, 64, 1024,  4, PDP, &v7modes },
	 { S5Vfs, 2, 64, 1024, 10, VAX, &v7modes },
};

#define	rootino	(fs[fstype].fs_rootino)
#define	bytpino	(fs[fstype].fs_bytpino)
#define	bufsize	(fs[fstype].fs_bufsize)
#define	fscpu	(fs[fstype].fs_cpu)
#define	dirlim	(fs[fstype].fs_dirlim)
#define	fstbp	(fs[fstype].fs_fstbp)
#define	s_ifmt	(fs[fstype].fs_smp->fs_ifmt)
#define	s_ifreg	(fs[fstype].fs_smp->fs_ifreg)
#define	s_ifdir	(fs[fstype].fs_smp->fs_ifdir)
#define	s_ifblk	(fs[fstype].fs_smp->fs_ifblk)
#define	s_ifchr	(fs[fstype].fs_smp->fs_ifchr)
#define	s_ififo	(fs[fstype].fs_smp->fs_ififo)
#define	s_ifmpc	(fs[fstype].fs_smp->fs_ifmpc)
#define	s_ifmpb	(fs[fstype].fs_smp->fs_ifmpb)

struct d6inode {
	ushort	d6i_mode;	/* Mode and type of file */
	char	d6i_nlink;	/* Number of links to file */
	char	d6i_uid;	/* Owner's user id */
	char	d6i_gid;	/* Owner's group id */
	char	d6i_size[3];	/* Number of bytes in file; [0] = high 8 bits */
	ushort	d6i_addr[8];	/* Disk block addresses */
	time_t	d6i_atime;	/* Time last accessed */
	time_t	d6i_mtime;	/* Time last modified */
};

#define	i6p	((struct d6inode *)ip)

#define	BUFSIZE	1024	/* Largest block size used by any type of filesystem */

#define	BUFSIZB	(bufsize)		/* Buffer size (bytes) */
#define	BUFSIZS	(bufsize/sizeof(short))	/* Buffer size (shorts) */
#define	BUFSIZL	(bufsize/sizeof(long))	/* Buffer size (longs) */

#define	NULLP	((long *)0)	/* Uninitialized pointer */

#define	APB	BUFSIZL		/* Number of addresses per block */

#define	DESIZE	sizeof(struct direct)
#define	DEPB	(BUFSIZB/DESIZE)

union {
	char		CHARbuf[BUFSIZE];
	struct direct	DIRbuf[BUFSIZE/DESIZE];
} bufu;

#define	charbuf	(bufu.CHARbuf)
#define	dirbuf	(bufu.DIRbuf)

long	  i1[BUFSIZE/sizeof(long)];	/* First indirect block */
ushort	v6i1[BUFSIZE/sizeof(short)];	/* Ditto for V6 */
long	  i2[BUFSIZE/sizeof(long)];	/* Second indirect block */
ushort	v6i2[BUFSIZE/sizeof(short)];	/* Ditto for V6 */
long	  i3[BUFSIZE/sizeof(long)];	/* Third indirect block */

long	*i1p = NULLP, *i2p = NULLP, *i3p = NULLP;	/* Buffer pointers */
ushort	*v6i1p, *v6i2p;					/* Ditto for V6 */

#define	INOPB	(BUFSIZB/bytpino)	/* Number of inodes per block */

#define	EXECUTEP	01
#define	WRITEP		02
#define	READP		04

#if vax
#define	CPU	VAX
#else
#define	CPU	PDP
#endif

int	cpu = CPU;	/* CPU that we're running on */
char	*progname;	/* Name of program */
int	lsflag;		/* Nonzero ==> list directory/file rather than copy file */
char	silent = 0;	/* Nonzero ==> suppress error messages */
int	longls = 0;	/* Nonzero ==> list directory/file verbosely */
char	*device;	/* Name of device containing UNIX filesystem */
char	*input;		/* Name of input file */
char	*output;	/* Name of output file */
long	offset = 0;	/* Offset value to be added value passed to lseek */
int	des;		/* File descriptor for reading input filesystem */
int	fsw;		/* Flag used to indicate end of pathname scan */
long	bnum;		/* Number of next block to be read */
long	iaddr;		/* Index to inode's pointer array */

#define	lastchr(x)	(x[strlen(x)-1])


main(argc,argv)
int	argc;
char	*argv[];
{
	char	*basename();
	char	*fullinp;
	register struct fstab	*fstp;
	ino_t	inum;
#if VMS
/* NOTE:  Some versions of UNIX use the name strchr() rather than index() */
	char	*index();
#endif VMS
	struct dinode	inode;

	lsflag = ( *(progname=basename(*argv)) == 'l' );

	switch ( progname[2] ) {
		case '6':  fstype = V6;  break;
		case '7':  fstype = V7;  break;
		case 'v':  fstype = V32; break;
		case '5':  fstype = S5P; break;
		case 'b':  fstype = S5V; break;
	}
	files(argc,argv);
	fullinp = input;
	if ( device == NULL ) {
		for ( fstp=fstbp; fstp->fs_pfx; ++fstp )
			if ( chkfs(input,fstp->fs_pfx) )
				break;
		/* fs is either a match or the default */
#if VMS
		if ( fstp->fs_mindev < 0 )
			error("%s -- inaccessible filesystem\n",fullinp);
		device = UNIXDEV;
		*index(device,'0') += fstp->fs_mindev >> 4;
		/*
		 *  The `-1' in the following line is to compensate for an
		 *  off-by-one error in the Eunice implementation of `lseek',
		 *  which is probably due to the fact that it adds 512 bytes
		 *  since virtual blocks are 1-based rather than 0-based.  So
		 *  it's probably the case that the error occcurs and must be
		 *  compensated for only when doing logical I/O rather than
		 *  virtual.  (Quite a long comment for one line of code!)
		 */
		offset = ( ( cyloff[fstp->fs_mindev%16] * BLKSPERCYL ) - 1 ) * BUFSIZB;
#else
		if ( (device=fstp->fs_spcl) == NULL )
			error("%s -- inaccessible filesystem\n",fullinp);
#endif VMS
	}
	if ( (des=open(device,0)) < 0 )
		error ("%s cannot be opened\n",device);
	setgid(getgid());
	inum = rootino;
	while ( *input && *input == '/' )
		input++;
	fsw = *input ? 1 : 0;
	while ( fsw ) {
		iread(&inode,inum);
		access(&inode,EXECUTEP);
		if ( (inum=trace(&inode)) == 0)
			error("%s -- input file not found\n",fullinp);
	}
	iread(&inode,inum);
	if ( lsflag )
		dols(fullinp,&inode);
	else
		docp(fullinp,&inode,output);
	exit(0);
}


files(ac,av)
int	ac;
char	**av;
{
	int	bump;
	long	atol();

	for ( --ac; **++av == '-'; ac -= bump+1, av += bump ) {
		bump = 0;
		while ( *++*av )
				switch ( **av ) {

				case 's':
					++silent;
					break;

				case 'l':
					++longls;
					break;

				case '6':
					fstype = V6;
					break;

				case '7':
					fstype = V7;
					break;

				case 'v':
					fstype = V32;
					break;

				case '5':
					fstype = S5P;
					break;

				case 'b':
					fstype = S5V;
					break;

				case 'o':
					if ( ac > 2 )
						offset = atol(av[++bump]) * BUFSIZB;
					else
						usage();
					break;

				case 'd':
#if VMS
					error("-d option not supported on vms\n");
#else
					if ( ac > 2 )
						device = av[++bump];
					else
						usage();
#endif VMS
					break;

				default:
					error("Invalid option (%c)\n",**av);
			}
	}
	input = *av;
	if ( !lsflag )
		output = (ac>2) ? *++av : NULL;
}


usage()
{
	error("Usage: %s [-6] [-7] [-v] [-b] [-5] [-d device] input-file %s\n",
		progname, lsflag? "" : "[output-file]");
}


chkfs(av1,fs)
char	*av1;
char	*fs;
{
	int	len;

	len = strlen(fs);
	if ( ( strncmp(av1,fs,len--) == 0 )
	|| ( ( av1[len] == '\0' ) && ( strncmp(av1,fs,len) == 0 ) ) ) {
		input += len;
		return (1);
	}
	return (0);
}


trace(ip)
struct dinode	*ip;
{
	char	*getname();
	char	*name;
	ino_t	inum;

	name = getname();
	for ( bnum=0; getblock(ip); )
		if ( (inum=dentry(name)) != 0 )
			return(inum);
	return(0);
}


char *
getname()
{
	register int	j;
	static char	name[DIRSIZ+1];

	for ( j=0; j<DIRSIZ && *input!='/'; j++ ) {
		if ( *input == '\0' )
			break;
		name[j] = *input++;
	}
	name[j] = '\0';
	while ( *input && *input != '/' )
		input++;
	while ( *input && *input == '/' )
		input++;
	if ( *input == '\0' )
		fsw = 0;
	return (name);
}


dentry(name)
char	*name;
{
	register int	m;
	ino_t		inum;

	for ( m=0; m<DEPB; m++ ) {
		if ( (inum=dirbuf[m].d_ino) == 0 )
			continue;
		if ( strncmp(dirbuf[m].d_name,name,DIRSIZ) == 0 )
			return (inum);
	}
	return(0);
}


dols(fullname,ip)
char	*fullname;
register struct dinode	*ip;
{
	long		isize();
	long		nbytes;
	register int	k;
	long		size = 0;
	char		fn[DIRSIZ+1];

	if ( (ip->di_mode & s_ifmt) != s_ifdir )
		lsone(fullname,ip,0);
	else {
		access(ip,READP);
		redirout();
		bnum = 0;
		nbytes = isize(ip);
		while ( getblock(ip) )
			for ( k=0; k<DEPB; k++ ) {
				if ( (size+=DESIZE) > nbytes )
					break;	/* and hope getblock gets EOF */
				if ( dirbuf[k].d_ino!=0 ) {
					strncpy(fn,dirbuf[k].d_name,DIRSIZ);
					if ( strcmp(fn,".") != 0
					&&   strcmp(fn,"..") != 0 )
						lsone(fn,NULL,dirbuf[k].d_ino);
				}
			}
		fclose(stdout);
		wait(0);
	}
}


lsone(name,ip,inum)
char	*name;
struct dinode	*ip;
ino_t	inum;
{
	static struct dinode	inode;
	int	i_uid;
	ushort	i_mode;
	char	i_type;
	ushort	i_nlink;
	dev_t	i_dev;
	long	i_mtime;
	long	getlong();
	char	*uidname();
	long	isize();
	char	*tstr();

	if ( longls == 0 ) {
		printf("%s\n",name);
		return;
	}
	if ( ip == NULL ) {
		ip = &inode;
		iread(ip,inum);
	}
	if ( fstype == V6 ) {
		i_uid   = i6p->d6i_uid & 0377;
		i_mode  = i6p->d6i_mode;
		i_nlink = i6p->d6i_nlink & 0377;
		i_dev   = i6p->d6i_addr[0];
		i_mtime = getlong(&i6p->d6i_mtime);
	} else {
		i_uid   = ip->di_uid;
		i_mode  = ip->di_mode;
		i_nlink = ip->di_nlink;
		if ( fstype == V7 )		/* IS THIS CORRECT? */
			i_dev = makedev(ip->di_addr[2],ip->di_addr[1]);
		else
			i_dev = makedev(ip->di_addr[1],ip->di_addr[0]);
		i_mtime = getlong(&ip->di_mtime);
	}
	putchar(i_type=ftype(i_mode));
	pmode(i_mode);
	printf("%2d %-6.6s",i_nlink,uidname(i_uid));
	if ( i_type == 'b' || i_type == 'c' )
		printf("%3d,%3d",major(i_dev),minor(i_dev));
	else
		printf("%7ld",isize(ip));
	printf(" %s %s\n",tstr(i_mtime),name);
}


iread(ip,ino)
struct dinode	*ip;
register ino_t	ino;
{
	long	j;

	j = (ino + (2*INOPB-1)) / INOPB;
	if ( lseek(des,BUFSIZB*j+offset,0) == -1 ) {
		perror("lseek");
		error("I/O error\n");
	}
	j = ((ino + (INOPB-1)) % INOPB) * bytpino;
	if ( lseek(des,j,1) == -1 ) {
		perror("lseek");
		error("I/O error\n");
	}
	if ( read(des,ip,bytpino) < 0 ) {
		perror("read");
		error("I/O error\n");
	}
}


ftype(mode)
register ushort	mode;
{
	register ushort	filtyp;

	filtyp = mode & s_ifmt;

	if ( filtyp == s_ifreg )
		return('-');
	if ( filtyp == s_ifdir )
		return('d');
	if ( filtyp == s_ifblk )
		return('b');
	if ( filtyp == s_ifchr )
		return('c');
	if ( filtyp == s_ififo )
		return('p');
	return('?');
}


/*
 *  Code to print the mode symbolically.
 *  Originally stolen straight from ls.
 *  Rewritten to put the program in the public domain,
 *  which is a pretty stupid waste of time
 *  (the rewriting), but less so than SMP.
 */

char mchars[] = "rwxrwxrwx";

pmode(mode)
register int	mode;
{
	char		buf[sizeof(mchars)];
	register int	bit;
	register char	*s;

	strcpy(buf,mchars);
	if ( mode & S_ISVTX ) {
		buf[8] = 't';
		mode |= S_IEXEC >> 6;
	}
	if ( mode & S_ISGID ) {
		buf[5] = 's';
		mode |= S_IEXEC >> 3;
	}
	if ( mode & S_ISUID ) {
		buf[2] = 's';
		mode |= S_IEXEC;
	}
	for ( bit = S_IREAD, s = buf; bit; bit >>= 1, s++ )
		if ((mode & bit) == 0)
			*s = '-';
	printf("%s",buf);
}


char *
uidname(uid)
int	uid;
{
	struct passwd	*pw;
	static char	buf[10];
	struct passwd	*getpwuid();

	if ( (pw=getpwuid(uid)) != NULL )
		return (pw->pw_name);
	else {
		sprintf(buf,"%d",uid);
		return (buf);
	}
}


char *
tstr(t)
long t;
{
	long		time();
	static long	year;
	char		*ctime();
	static char	buf[15];
	register char	*cp;

	if ( year == 0 )
		year = time(0) - 60L*60L*24L*30L*6L;	/* 6 months ago */
	cp = ctime(&t);

	if ( year < t && t < time(0) )
		sprintf(buf,"%-12.12s",cp+4);
	else
		sprintf(buf,"%-7.7s %-4.4s",cp+4,cp+20);
	return (buf);
}


redirout()
{
	int	pfd[2];
	int	fk;

#if VMS
	return;
#else
	if ( pipe(pfd) )
		error("Can't create pipe\n");
	while ( (fk=fork()) == -1 )
		sleep(1);
	if ( fk == 0 )
		dup2(pfd[0],0);
	else
		dup2(pfd[1],1);
	close(pfd[0]);
	close(pfd[1]);
	if ( fk == 0 ) {
		execl("/usr/bin/sort","sort",longls ? "+0.40" : 0, 0);
		execl("/bin/sort","sort",longls ? "+0.40" : 0, 0);
		error("Can't exec sort\n");
	}
#endif VMS
}


docp(fullname,ip,outfile)
char	*fullname;
register struct dinode	*ip;
char	*outfile;
{
	int	odes;
	long	isize();
	long	size = 0L;
	long	nbytes;
	int	k;

	if ( (ip->di_mode&s_ifmt) == s_ifdir )
		error("%s is a directory\n",fullname);
	if ( (ip->di_mode&s_ifmt) != s_ifreg )
		error("%s is a special file\n",fullname);
	else {
		access(ip,READP);
		odes = openout(outfile,fullname);
		bnum = 0;
		nbytes = isize(ip);
		while ( getblock(ip) ) {
			k = BUFSIZB;
			if ( size+BUFSIZB > nbytes )
				k = nbytes - size;
			write(odes,charbuf,k);
			size += k;
		}
	}
	close(odes);
}


#define OWNER	6
#define GROUP	3
#define GENRL	0

#if VMS
#define	SU_UID	4
#define	SU_GID	1
#else
#define SUSER	0
#endif VMS

access(ip,key)
register struct dinode	*ip;
register int	key;
{
	int	uid;
	int	i_uid;
	int	i_gid;
	int	i_mode;
#if VMS
	int	v_gid;
	int	u_gid = 0;
	int	mult = 1;
#endif VMS

#ifndef IGNPROT
	/* superuser has all permissions */
#if VMS
	if ( ( (v_gid=getgid()) == SU_GID ) && ( getuid() == SU_UID ) )
#else
	if ( (uid=getuid()) == SUSER )
#endif VMS
		return;

	if ( fstype == V6 ) {
		i_uid  = i6p->d6i_uid  & 0377;
		i_gid  = i6p->d6i_gid  & 0377;
		i_mode = i6p->d6i_mode & 0777;
	} else {
		i_uid  = ip->di_uid;
		i_gid  = ip->di_gid;
		i_mode = ip->di_mode;
	}

	if ( i_mode & (key<<GENRL) )
		return;
#if VMS
	while ( v_gid ) {
		u_gid += (v_gid%8) * mult;
		v_gid /= 8;
		mult *= 10;
	}
	if ( ( u_gid == i_gid ) && ( i_mode & (key<<GROUP) ) )
		return;
#else
	if ( ( uid == i_uid ) && ( i_mode & (key<<OWNER) ) )
		return;
	if ( ( getgid() == i_gid ) && ( i_mode & (key<<GROUP) ) )
		return;
#endif VMS
	error("No permission\n");
#endif IGNPROT
}


openout(outname,inname)
char *outname;
char *inname;
{
	char		c;
	char		*outnm;
	char		*basename();
	int		odes;
	static char	buf[80];

	if ( outname == NULL )
		return (fileno(stdout));
#if VMS
	outnm = outname;
	if ( (c=lastchr(outname)) == ':' || c == ']' ) {
		strcpy(buf,outname);
		strcat(buf,basename(inname));
		outnm = buf;
	}
	if ( (odes=creat(outnm,0644,"txt")) < 0 )
		error("cannot creat %s\n",outname);
#else
	if ( (odes=creat(outname,0666)) < 0 ) {
		sprintf(buf,"%s/%s",outname,basename(inname));
		if ( (odes=creat(buf,0666)) < 0 )
			error("cannot creat %s\n",outname);
	}
#endif VMS
	return (odes);
}


char *
basename(s)
register char	*s;
{
/* NOTE:  Some versions of UNIX use the name strrchr() rather than rindex() */
	char		*rindex();
	register char	*t;

	if ( (t=rindex(s,'/')) == NULL )
		return(s);
	else
		return(t+1);
}


/*
 * Getblock -- return next block in file
 *
 *	Does all indirection, etc.  BUT doesn't look
 *	at length of file, so it always returns a
 *	full block.  The other code has to watch out.
 */

getblock(ip)
register struct dinode	*ip;
{
	long		isize();
	static long	nbytes;
	long		block;
	register char	*p;

	if ( bnum == 0 ) {	/* new file */
		iaddr = 0;
		nbytes = isize(ip);
	}
	if ( BUFSIZB*bnum >= nbytes || nbytes == 0 )
		return(0);	/* end of file */

	if ( fstype == V6 ) {
		block = i6p->d6i_addr[iaddr] & 0177777L;
		if ( !(i6p->d6i_mode&S6LARGE) ) {
			get(block,charbuf);
			++iaddr;
			++bnum;
			return(1);
		}
		if ( bnum%256 == 0 ) {
			if ( bnum/256 < 7 ) {
				get(block,v6i1);
				++iaddr;
			} else {
				if ( bnum/256 == 7 ) {
					get(block,v6i2);
					v6i2p = v6i2;
				}
				get((long)(*v6i2p++),v6i1);
			}
			v6i1p = v6i1;
		}
		get((long)(*v6i1p++),charbuf);
		++bnum;
		return(1);
	} else {
		p = &ip->di_addr[3*iaddr];
		if ( fscpu == PDP )
			block = ((((long)p[0])&0377)<<16)
			      +  (((long)p[1])&0377)
			      + ((((long)p[2])&0377)<<8);
		else
			block =  (((long)p[0])&0377)
			      + ((((long)p[1])&0377)<<8)
			      + ((((long)p[2])&0377)<<16);

		if ( bnum < dirlim ) {
			get(block,charbuf);
			++iaddr;
			++bnum;
			return(1);
		}
		if ( i1p == NULLP ) {
			getfib(block,i1);
			i1p = i1 - 1;
		}
		if ( ++i1p == i1+APB ) {
			if ( i2p == NULLP ) {
				getfib(block,i2);
				i2p = i2 - 1;
			}
			if ( ++i2p == i2+APB ) {
				if ( i3p == NULLP ) {
					getfib(block,i3);
					i3p = i3 - 1;
				}
				if ( ++i3p == i3+APB ) {
					fprintf(stderr,"unexpected internal error\n");
					exit(-2);
				}
				getib(*i3p,i2);
				i2p = i2;
			}
			getib(*i2p,i1);
			i1p = i1;
		}
		get(*i1p,charbuf);
		++bnum;
		return(1);
	}
}


getfib(block,buf)
long	block;
char	*buf;
{
	getib(block,buf);
	++iaddr;
}


getib(block,buf)
long	block;
char	*buf;
{
	get(block,buf);
	if ( cpu != fscpu )
		swaw(buf);
}


swaw(buf)
short	*buf;
{
	register short	*p;
	register int	i;
	register short	tmp;

	for ( p=buf,i=0; i<BUFSIZL; ++i ) {
		tmp = p[0];
		p[0] = p[1];
		p[1] = tmp;
		p += 2;
	}
}


/*
 *  Get a block.  If block is 0, it is a hole in the file; act appropriately.
 */

get(block,buf)
long	block;
char	*buf;
{
	register char *p;

	if ( block == 0 ) {
		fprintf(stderr,"%s: file has hole(s)\n",progname);
		for ( p=buf; p<buf+BUFSIZB; )
			*p++ = 0;
	} else {
		if ( lseek(des,BUFSIZB*block+offset,0) == -1 ) {
			perror("lseek");
			error("I/O error\n");
		}
		if ( read(des,buf,BUFSIZB) != BUFSIZB ) {
			perror("read");
			error("IO error\n");
		}
	}
}


long
isize(ip)
register struct dinode	*ip;
{
	ushort	shortint();
	long	size;
	long	getlong();

	if ( fstype == V6 )		/* This is UGLY!!! */
		size = (((i6p->d6i_size[0])&0377L)<<16)
		     + (shortint(&i6p->d6i_size[1])&0177777L);
	else
		size = getlong(&ip->di_size);
	return (size);
}


ushort
shortint(p)
register char	*p;
{
	return((p[1]<<8)|(p[0]&0377));
}


/*
 *  Turn a 4-byte entity (which may be a long or
 *  an SLONG) into the local kind of long.
 */

long
getlong(lp)
register SLONG	*lp;
{
	if ( cpu == fscpu )
		return (*(long *)lp);
	else
		return (((long)lp->l_high << 16) | (lp->l_low));
}


/* VARARGS1 */

error(string,arg1,arg2)
char	*string;
char	*arg1;
char	*arg2;
{
	if ( !silent )
		fprintf(stderr,string,arg1,arg2);
	exit(-1);
}
