/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_phystrack[] = "$Id: phystrack.c,v 1.3 2003/02/16 15:32:33 bad Exp $";
#else
#define FILECODE "F(305)"
#endif 

/*
 * phystrack.c --
 *
 *    track down the utilization of the physical pages
 *    
 *    N.B: used only for debugging, does not provide any 
 *         functionality
 */


#include "hostif.h"     /* Must come before any linux header files */

#ifdef linux
# include "driver-config.h" /* Versioned symbols from linux/string.h */
# include <linux/string.h> /* memset() in the kernel */
#elif __FreeBSD__
# include <sys/param.h>
# include <sys/systm.h>
# include <string.h>
#elif defined(__NetBSD__)
# include <sys/param.h>
# include <sys/systm.h>
#elif defined(WINNT_DDK)
# include <string.h>
#else
# error "Unknown platform"
#endif

#include "vm_types.h"
#include "vmx86.h"
#include "vm_assert.h"

#define PHYSTRACK_PAGES_PER_ENTRY (32*1024)
#define PHYSTRACK_MAX_ENTRIES     (32)
#define BYTES_PER_ENTRY (PHYSTRACK_PAGES_PER_ENTRY/8)


typedef struct PhysTracker {
   int numVMs;
   void *dir[PHYSTRACK_MAX_ENTRIES];
} PhysTracker;

/* The global physTracker pointer */
PhysTracker *physTracker = NULL;

/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Init --
 *
 *      module initialization 
 *
 * Results:
 *      
 *      returns the PhysTracker. creates one if does not exist
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
PhysTracker *
PhysTrack_Init(void)
{
   /* allocate a new phystracker */
   if (physTracker == NULL) {
      physTracker = (PhysTracker *)HostIF_AllocKernelMem(sizeof(PhysTracker),FALSE);
      if (physTracker) { 
	 memset(physTracker,0,sizeof(PhysTracker));
      }
   }

   /* increment use count */
   if (physTracker) {
      physTracker->numVMs++;
   } else {
      Warning("PhysTrack_Init failed\n");
   }

   return physTracker;
} 



/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Cleanup --
 *
 *      module deallocation
 *
 * Results:
 *      
 *      reallocates all structures, including 'tracker'
 *      
 *      
 *
 * Side effects:
 *      tracker deallocated
 *
 *----------------------------------------------------------------------
 */
void
PhysTrack_Cleanup(PhysTracker *tracker) 
{ 
   int i, j;

   ASSERT(tracker && (tracker == physTracker));

   /* decrement use count */
   tracker->numVMs--;	

   /* deallocate phystracker if no more VMs */
   if (tracker->numVMs == 0) {
      for (i=0;i<PHYSTRACK_MAX_ENTRIES;i++) { 
	 if (tracker->dir[i]) { 
	    for (j = 0; j < BYTES_PER_ENTRY; j++) {
	       char *bitvector = (char *)tracker->dir[i];
	       if (bitvector[j] != 0) {
		  Warning("PhysTrack_Cleanup: pfns still locked\n");
	       }
	    }
	    HostIF_FreePage(tracker->dir[i]);
	    tracker->dir[i] = NULL;
	 }
      }
      HostIF_FreeKernelMem(tracker);
      physTracker = NULL;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Add --
 *
 *      add a page to the core map tracking.
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      Fatal if the page is already tracked.
 *
 *----------------------------------------------------------------------
 */
void
PhysTrack_Add(PhysTracker *tracker,
              MPN mpn)
{
   int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   int pos = off / 8;
   int bit = 1 <<(off % 8);
   char *bitvector;

   ASSERT(tracker && (tracker == physTracker));
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);

   if (tracker->dir[dir]==NULL) { 
      /* more efficient with page alloc */
      ASSERT(BYTES_PER_ENTRY == PAGE_SIZE);
      tracker->dir[dir] = HostIF_AllocPage(FALSE);
      if (tracker->dir[dir]==NULL) { 
         return;
      }
      memset(tracker->dir[dir],0,BYTES_PER_ENTRY);
   }
   
   bitvector = tracker->dir[dir];
   
   if (bitvector[pos] & bit) {
      PANIC();
   }
   
   bitvector[pos] |= bit;
}



/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Remove --
 *
 *      remove a page from the core map tracking
 *
 * Results:
 *      
 *      void

 *
 * Side effects:
 *      Fatal if the page is not tracked
 *
 *----------------------------------------------------------------------
 */
void
PhysTrack_Remove(PhysTracker *tracker,
              MPN mpn)
{
   int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   int pos = off / 8;
   int bit = 1 <<(off % 8);
   char *bitvector;

   ASSERT(tracker && (tracker == physTracker));
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);

   bitvector = tracker->dir[dir];
   if (bitvector == NULL) {
      PANIC();
   }
   if (!(bitvector[pos] & bit)) {
      PANIC();
   }

   bitvector[pos] &= ~bit;
}


/*
 *----------------------------------------------------------------------
 *
 * PhysTrack_Test --
 *
 *      tests whether a page is being tracked
 *
 * Results:
 *      
 *      TRUE if the page is tracked
 *      FALSE otherwise
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
Bool
PhysTrack_Test(PhysTracker *tracker,
               MPN mpn)
{
   int dir = mpn / PHYSTRACK_PAGES_PER_ENTRY;
   int off = mpn % PHYSTRACK_PAGES_PER_ENTRY;
   int pos = off / 8;
   int bit = 1 <<(off % 8);
   char *bitvector;

   ASSERT(tracker && (tracker == physTracker));
   ASSERT(dir < PHYSTRACK_MAX_ENTRIES);

   bitvector = tracker->dir[dir];
   if (bitvector == NULL) { 
      return FALSE;
   }
   if (bitvector[pos] & bit) {
      return TRUE;
   } else {
      return FALSE;
   }
}
