/* $NetBSD: fsu_mount.c,v 1.2 2009/03/24 09:04:14 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/mount.h>

#include <err.h>
#include <errno.h>
#include <stdbool.h>
#include <string.h>

#include <isofs/cd9660/cd9660_mount.h>
#include <fs/efs/efs_mount.h>
#include <ufs/ufs/ufsmount.h> /* ext2fs, ffs, lfs */
#include <fs/hfs/hfs.h>
#include <msdosfs/msdosfsmount.h>
#include <ntfs/ntfsmount.h>
#include <fs/tmpfs/tmpfs_args.h>
#include <fs/udf/udf_mount.h>

#include <rump/ukfs.h>

#include <fsu_utils.h>

#include "mount_cd9660.h"
#include "mount_efs.h"
#include "mount_ext2fs.h"
#include "mount_ffs.h"
#include "mount_hfs.h"
#include "mount_lfs.h"
#include "mount_msdos.h"
#include "mount_ntfs.h"
#include "mount_tmpfs.h"
#include "mount_udf.h"
#ifdef WITH_SYSPUFFS
#include "rump_syspuffs.h"
#endif

#include "fsu_mount.h"

#define MAX_MNT_ARGS (15)

struct fsu_fsalias_s {
	char *fsa_name;
	char *fsa_path;
	char *fsa_type;
	char *fsa_mntopt;
	char *fsa_puffsexec;
	struct fsu_fsalias_s *fsa_next;
};

union fsu_args {
	struct iso_args args_cd9660;
	struct efs_args args_efs;
	struct ufs_args args_ufs; /* ext2fs, ffs, lfs, ufs */
	struct hfs_args args_hfs;
	struct msdosfs_args args_msdos;
	struct ntfs_args args_ntfs;
	struct tmpfs_args args_tmpfs;
	struct udf_args args_udf;
#ifdef WITH_SYSPUFFS
	struct syspuffs_args args_syspuffs;
#endif
} args;

typedef int (*parseargs_fp)(int, char **, void *, int *, char *, char *);

typedef struct fsu_fs_s {
	const char *fs_name;
	void *fs_args;
	unsigned int fs_args_size;
	parseargs_fp fs_parseargs;
	unsigned int fs_flags;
#define FS_NO_AUTODETECT (1)
} fsu_fs_t;

fsu_fs_t fslist[] = {
	{ MOUNT_CD9660,	&args.args_cd9660,	sizeof(struct iso_args),
	  (parseargs_fp)mount_cd9660_parseargs,		0 },
	{ MOUNT_EXT2FS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ext2fs_parseargs,		0 },
	{ MOUNT_FFS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ffs_parseargs,		0 },
	{ MOUNT_HFS,	&args.args_hfs,		sizeof(struct hfs_args),
	  (parseargs_fp)mount_hfs_parseargs,		0 },
	{ MOUNT_LFS,	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_lfs_parseargs,		0 },
	{ MOUNT_MSDOS,	&args.args_msdos,	sizeof(struct msdosfs_args),
	  (parseargs_fp)mount_msdos_parseargs,		0 },
	{ MOUNT_NTFS,	&args.args_ntfs,	sizeof(struct ntfs_args),
	  (parseargs_fp)mount_ntfs_parseargs,		0 },
	{ MOUNT_UDF,	&args.args_udf,		sizeof(struct udf_args),
	  (parseargs_fp)mount_udf_parseargs,		0 },
	{ MOUNT_EFS,	&args.args_efs,		sizeof(struct efs_args),
	  (parseargs_fp)mount_efs_parseargs,		0 },
	{ MOUNT_TMPFS,	&args.args_tmpfs,	sizeof(struct tmpfs_args),
	  (parseargs_fp)mount_tmpfs_parseargs,		0 },
#ifdef WITH_SYSPUFFS
	{ MOUNT_PUFFS,	&args.args_syspuffs,	sizeof(struct syspuffs_args),
	  (parseargs_fp)mount_syspuffs_parseargs,	FS_NO_AUTODETECT },
#endif
	{ NULL,		NULL,			0,  NULL, 0 }
};

struct mount_data_s {
	fsu_fs_t *mntd_fs;
	char mntd_canon_dev[MAXPATHLEN];
	char mntd_canon_dir[MAXPATHLEN];
	int mntd_flags;
	int mntd_argc;
	char *mntd_argv[MAX_MNT_ARGS];
	char *mntd_cwd;
};

static struct fsu_fsalias_s *alias_head, *alias_tail;
static char *fsdevice, *fstype;

static int build_alias_list(void);
static void free_alias_list(void);
static struct ukfs *mount_alias(struct fsu_fsalias_s *, char *);
static struct ukfs *mount_fstype(fsu_fs_t *, char *, char *, char *);
_Bool fsu_readonly = false;

/*
 * Tries to mount an image.
 * if the fstype is not given try every supported types.
 */
struct ukfs *
fsu_mount(int *argc, char **argv[], char **fstp, char **fsdp)
{
	struct ukfs *ukfs;
	fsu_fs_t *fst;
	struct fsu_fsalias_s *cural;
	int ch, idx;
	char *mntopts, afsdev[MAXPATHLEN], *path;

#ifdef WITH_SYSPUFFS
	const char options[] = "f:o:p:t:", *puffsexec;
#else
	const char options[] = "f:o:t:";
#endif

	mntopts = fstype = fsdevice = NULL;
#ifdef WITH_SYSPUFFS
	puffsexec = NULL;
#endif
	fst = NULL;
	ukfs = NULL;

	ukfs_init();
	ukfs_modload_dir("/usr/lib");

	/*
	 * [-o mnt_args] [[-t] fstype] [ -p puffsexec] [-f] fsdevice
	 */
	while ((ch = getopt(*argc, *argv, options)) != -1) {
		switch (ch) {
		case 'f':
			if (fsdevice == NULL) {
				if (optarg[0] == '.') {
					realpath(optarg, afsdev);
					fsdevice = strdup(afsdev);
				} else
					fsdevice = optarg;
			}
			break;
		case 'o':
			if (mntopts == NULL) {
				mntopts = optarg;
				setenv("FSU_MNTOPTS", optarg, 1);
			}
			break;
#ifdef WITH_SYSPUFFS
		case 'p':
			if (puffsexec == NULL) {
				puffsexec = optarg;
				for (fst = fslist; fst->fs_name != NULL; ++fst)
					if (fst->fs_name == MOUNT_PUFFS)
						break;
				break;
				if (fstype == NULL)
					fstype = strdup(MOUNT_PUFFS);
				else if (strcmp(fstype, MOUNT_PUFFS) != 0) {
					warnx("-p is only for puffs");
					return NULL;
				}
			}
			break;
#endif
		case 't':
			if (fstype == NULL)
				fstype = optarg;
			break;
		case '?':
		default:
			errno = EINVAL;
			return NULL;
		}
	}
	idx = optind;
	optind = optreset = 1;

	if (mntopts == NULL)
		mntopts = getenv("FSU_MNTOPTS");

	if (fsu_readonly) {
		if (mntopts == NULL)
			mntopts = strdup("ro");
		else {
			char *tmp;

			tmp = malloc(strlen(mntopts) + 4);
			snprintf(tmp, strlen(mntopts) + 4, "%s,ro", mntopts);
			mntopts = tmp;
		}
	}

	if (fstype == NULL && fst == NULL)
		fstype = getenv("FSU_FSTYPE");

	if (fstype != NULL && fst == NULL) {
		for (fst = fslist; fst->fs_name != NULL; ++fst)
			if (strcmp(fstype, fst->fs_name) == 0)
				break;

		if (fst->fs_name == NULL) {
			fprintf(stderr, "%s: filesystem not supported\n",
				fstype);
			return NULL;
		}
	}

	if (fst == NULL && idx < *argc) {
		if (strcmp((*argv)[idx], "--") == 0) {
			errno = EINVAL;
			return NULL;
		}

		for (fst = fslist; fst->fs_name != NULL; ++fst) {
			if (fst->fs_flags & FS_NO_AUTODETECT)
				continue;
			if (strcmp((*argv)[idx], fst->fs_name) == 0)	
				break;
		}
		if (fst->fs_name != NULL)
			idx++;
		else
			fst = NULL;
	}

       	if (fsdevice == NULL) {
		fsdevice = getenv("FSU_DEVICE");
		if (fsdevice == NULL) {
			if (idx < *argc && strcmp((*argv)[idx], "--") != 0) {
				if ((*argv)[idx][0] == '.') {
					realpath((*argv)[idx], afsdev);
					fsdevice = strdup(afsdev);
					++idx;
				} else
					fsdevice = (*argv)[idx++];
			} else {
				errno = EINVAL;
				return NULL;
			}
		}

		build_alias_list();
		for (cural = alias_head; cural != NULL;
		     cural = cural->fsa_next) {
			if (cural->fsa_name != NULL &&
			    strcmp(cural->fsa_name, fsdevice) == 0) {
				ukfs = mount_alias(cural, mntopts);
				fstype = cural->fsa_type;
				fsdevice = cural->fsa_path;
				goto out;
			}
		}
		free_alias_list();
	}
#ifdef WITH_SYSPUFFS
	ukfs = mount_fstype(fst, fsdevice, mntopts, puffsexec);
#else
	ukfs = mount_fstype(fst, fsdevice, mntopts, NULL);
#endif
out:
	if ((*argv)[idx] != NULL && strcmp((*argv)[idx], "--") == 0)
		++idx;

	if (--idx > 0) {
		(*argv)[idx] = (*argv)[0];
		*argv += idx;
		*argc -= idx;
		optind = optreset = 1;
	}
	if (fstp != NULL)
		*fstp = fstype;
	if (fsdp !=  NULL)
		*fsdp = fsdevice;

	path = getenv("FSU_CWD");
	if (path != NULL) {
		if (ukfs_chdir(ukfs, path) != 0) {
			fprintf(stderr, "could not chdir to %s\n", path);
			ukfs_release(ukfs, 0);
			*fstp = NULL;
			*fsdp = NULL;
			return NULL;
		}
	}
	optind = optreset = 1;
	return ukfs;
}

static struct ukfs
*mount_fstype(fsu_fs_t *fs, char *fsdev, char *mntopts, char *puffsexec)
{
	struct mount_data_s mntd;
	struct ukfs *ukfs;
	int rv;

	mntd.mntd_fs = fs;
	mntd.mntd_argc = mntd.mntd_flags = 0;
	ukfs = NULL;

	/* setting up the argv array */
	mntd.mntd_argv[mntd.mntd_argc++] = strdup(getprogname());

#ifdef WITH_SYSPUFFS
	if (puffsexec != NULL && fs->fs_name == MOUNT_PUFFS)
		mntd.mntd_argv[mntd.mntd_argc++] = puffsexec;
#endif

	if (mntopts != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = mntopts;
	}

	mntd.mntd_argv[mntd.mntd_argc++] = fsdev;
	mntd.mntd_argv[mntd.mntd_argc++] = strdup("/");
	mntd.mntd_argv[mntd.mntd_argc] = NULL;

	/* filesystem given */
	if (fs != NULL) {
		optind = optreset = 1;
		rv = fs->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv, 
				      fs->fs_args, &mntd.mntd_flags,
				      mntd.mntd_canon_dev, mntd.mntd_canon_dir);
		optind = optreset = 1;
		if (rv != 0)
			return NULL;

		ukfs = ukfs_mount(fs->fs_name, mntd.mntd_canon_dev,
				  mntd.mntd_canon_dir, mntd.mntd_flags,
				  fs->fs_args, fs->fs_args_size);
		if (ukfs == NULL)
			fprintf(stderr, "%s is not a valid %s image\n",
				fsdev, fs->fs_name);
		goto out;
	}

	/* filesystem not given (auto detection) */
	for (fs = fslist; fs->fs_name != NULL; ++fs) {
		if (fs->fs_flags & FS_NO_AUTODETECT)
			continue;
		mntd.mntd_flags = 0;
		optind = optreset = 1;
		rv = fs->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv,
				      fs->fs_args, &mntd.mntd_flags,
				      mntd.mntd_canon_dev, mntd.mntd_canon_dir);
		optind = optreset = 1;
		if (rv != 0)
			continue;
		
		ukfs = ukfs_mount(fs->fs_name, mntd.mntd_canon_dev,
				  mntd.mntd_canon_dir, mntd.mntd_flags,
				  fs->fs_args, fs->fs_args_size);
		if (ukfs != NULL)
			break;
	}

out:
	if (fs->fs_name != NULL)
		fstype = strdup(fs->fs_name);

	return ukfs;
}

static struct ukfs
*mount_alias(struct fsu_fsalias_s *al, char *mntopts)
{
	struct mount_data_s mntd;
	struct ukfs *ukfs;
	fsu_fs_t *cur;
	int rv;

	ukfs = NULL;
	mntd.mntd_argc = mntd.mntd_flags = 0;

	mntd.mntd_argv[mntd.mntd_argc++] = strdup(getprogname());
	
#ifdef WITH_SYSPUFFS
	if (al->fsa_puffsexec != NULL)
		mntd.mntd_argv[mntd.mntd_argc++] = strdup(al->fsa_puffsexec);
#endif

	if (al->fsa_mntopt != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = al->fsa_mntopt;
		setenv("FSU_MNTOPTS", al->fsa_mntopt, 1);
	}

	if (mntopts != NULL) {
		mntd.mntd_argv[mntd.mntd_argc++] = strdup("-o");
		mntd.mntd_argv[mntd.mntd_argc++] = mntopts;
		setenv("FSU_MNTOPTS", mntopts, 1);
	}

	mntd.mntd_argv[mntd.mntd_argc++] = al->fsa_path;
	mntd.mntd_argv[mntd.mntd_argc++] = strdup("/");
	mntd.mntd_argv[mntd.mntd_argc] = NULL;
	
	for (cur = fslist; cur->fs_name != NULL; ++cur)
		if (strcmp(cur->fs_name, al->fsa_type) == 0)
			break;

	if (cur->fs_name != NULL) {
		mntd.mntd_fs = cur;
		rv = cur->fs_parseargs(mntd.mntd_argc, mntd.mntd_argv,
				       cur->fs_args, &mntd.mntd_flags,
				       mntd.mntd_canon_dev,
				       mntd.mntd_canon_dir);
		if (rv == 0)
			ukfs = ukfs_mount(cur->fs_name, mntd.mntd_canon_dev,
					  mntd.mntd_canon_dir,
					  mntd.mntd_flags, cur->fs_args,
					  cur->fs_args_size);
	}
	
	if (cur->fs_name != NULL)
		fstype = strdup(cur->fs_name);
	return ukfs;

}

void
fsu_unmount(struct ukfs *ukfs)
{

	if (ukfs == NULL)
		return;
	ukfs_release(ukfs, 0);
}

static int
build_alias_list(void)
{
	struct stat sb;
	struct fsu_fsalias_s *new;
	FILE *fd;
	int i, rv;
	size_t len, off;
	char buf[8192], file[PATH_MAX + 1], *fp, *home;

	fp = getenv("HOME");
	if (fp == NULL) {
		warn("getenv");
		return -1;
	}

	len = strlcpy(file, fp, PATH_MAX + 1);
	strlcat(file, "/.fsurc", PATH_MAX + 1);

	/* no file no alias loaded */
	if (stat(file, &sb) != 0 || (fd = fopen(file, "r")) == NULL)
		return -1;

	rv = 0;
	new = NULL;
	for (;;) {
		/* get a line */
		for (i = 0; i < (int)sizeof(buf); ++i) {
			buf[i] = fgetc(fd);
			if (buf[i] == '\n' || feof(fd)) {
				buf[i] = '\0';
				break;
			}
		}

		if (buf[0] == '#')
			continue;
		if (i == 0 && feof(fd))
			break;

		if (new == NULL) {
			new = malloc(sizeof(struct fsu_fsalias_s));
			if (new == NULL) {
				warn("malloc");
				rv = 1;
				goto out;
			}
		}

		fp = strtok(buf, ":");
		if (fp == NULL)
			continue;
		if (fp[0] == '-') {
			fprintf(stderr,
			  "alias starting with a dash ('-') are not suported");
			continue;
		}
		new->fsa_name = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;

		if (fp[0] == '~' && fp[1] == '/') {
			home = getenv("HOME");

			if (home == NULL) {
				warn("getenv");
				continue;
			}
			len = strlen(home) + strlen(fp);
			new->fsa_path = malloc(len);
			if (new->fsa_path == NULL) {
				warn("malloc");
				continue;
			}
			off = strlcpy(new->fsa_path, home, len);
			new->fsa_path[off] = '/';
			new->fsa_path[off + 1] = '\0';
			strlcat(new->fsa_path, fp + 2, len);
		} else
			new->fsa_path = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;
		new->fsa_type = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			new->fsa_mntopt = NULL;
		else
			new->fsa_mntopt = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			new->fsa_puffsexec = NULL;
		else
			new->fsa_puffsexec = strdup(fp);

		/* strtok skips zero length tokens */
		if (strcmp(new->fsa_type, "puffs") == 0 && 
		    new->fsa_puffsexec == NULL &&
		    new->fsa_mntopt != NULL) {
			new->fsa_puffsexec = new->fsa_mntopt;
			new->fsa_mntopt = NULL;
		}

		new->fsa_next = NULL;

		if (alias_head == NULL)
			alias_head = alias_tail = new;
		else
			alias_tail = alias_tail->fsa_next = new;
		new = NULL;
	}

out:
	fclose(fd);
	if (new != NULL)
		free(new);
	return rv;
}

static void
free_alias_list(void)
{
	struct fsu_fsalias_s *cur;

	while ((cur = alias_head) != NULL) {
		alias_head = alias_head->fsa_next;
		free(cur->fsa_name);
		free(cur->fsa_path);
		free(cur->fsa_type);
		if (cur->fsa_mntopt != NULL)
			free(cur->fsa_mntopt);
		if (cur->fsa_puffsexec != NULL)
			free(cur->fsa_puffsexec);
		free(cur);
	}
}

const char
*fsu_mount_usage(void)
{

#ifdef WITH_SYSPUFFS
	return "[-o mnt_args] [[-t] fstype] [-p puffs_exec] [-f] fsdevice";
#else
	return "[-o mnt_args] [[-t] fstype] [-f] fsdevice";
#endif
}
