/* $Id: fsu_rmdir.c,v 1.6 2008/10/08 18:44:40 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <errno.h>
#include <getopt.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <rump/ukfs.h>

#ifndef FSU_CONSOLE
#include <fsu_mount.h>

static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */
#define main(a, b) fsu_rmdir_main(struct ukfs *ukfs, a, b)

int fsu_rmdir_main(struct ukfs *, int, char *[]);

static char *progname;
#endif /* !FSU_CONSOLE */

static int fsu_rmdir(struct ukfs *, char *, bool);
static void usage(void);

int
main(int argc, char *argv[])
{
	int rv, cur_arg;
	bool pflag;

#ifndef FSU_CONSOLE
	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#else
	progname = argv[0];
#endif

	pflag = false;
	if ((rv = getopt(argc, argv, "p")) != -1 && rv == 'p')
		pflag = true;

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		usage();
		return -1;
	}
	
	for (rv = 0, cur_arg = 0; cur_arg < argc; ++cur_arg)
		rv |= fsu_rmdir(ukfs, argv[cur_arg], pflag);

	return rv;
}

static int
fsu_rmdir(struct ukfs *fs, char *dirname, bool pflag)
{
	int rv;
	char *p;

	do {
		rv = ukfs_rmdir(fs, dirname);
		if (errno == ENOTEMPTY) {
			errno = 0;
			return 0;
		} else if (rv == -1) {
			warn("%s", dirname);
			return -1;
		}
		p = strrchr(dirname, '/');
		if (p == NULL)
			break;
		*p = '\0';
	} while (pflag);
	return 0;
}


static void
usage(void)
{

#ifdef FSU_CONSOLE
	fprintf(stderr, "usage: %s [-p] dirname...\n", progname);
#else
	fprintf(stderr, "usage: %s %s [-p] dirname...\n",
		getprogname(), fsu_mount_usage());

	exit(EXIT_FAILURE);
#endif
}

#ifndef FSU_CONSOLE
void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
