/* $Id: fsu_dir.c,v 1.4 2008/09/08 12:24:37 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>

#include <assert.h>
#include <dirent.h>
#include <err.h>
#include <stdlib.h>
#include <string.h>

#include <rump/ukfs.h>

#include "fsu_dir.h"

static char *fsu_getdirname(struct ukfs *);

FSU_DIR
*fsu_opendir(struct ukfs *fs, const char *path)
{
	FSU_DIR *dir;
	char *cwd;
	int rv;

	assert(fs != NULL && path != NULL);

	dir = malloc(sizeof(FSU_DIR));
	if (dir == NULL)
		return NULL;

	/* get current absolute path */
	cwd = fsu_getcwd(fs);

	/* get target absolute path */
	rv = ukfs_chdir(fs, path);
	if (rv == -1) {
		free(dir);
		return NULL;
	}
	dir->dd_path = fsu_getcwd(fs);

	/* return to the original directory */
	rv = ukfs_chdir(fs, cwd);
	if (rv == -1) {
		free(dir);
		free(dir->dd_path);
		free(cwd);
		return NULL;
	}

	free(cwd);
	dir->dd_ukfs = fs;
	dir->dd_off = dir->dd_size = 0;
	return dir;
}

void
fsu_closedir(FSU_DIR *dir)
{

	assert(dir != NULL);

	free(dir->dd_path);
	free(dir);
}

struct dirent
*fsu_readdir(FSU_DIR *dir)
{
	struct dirent *dent;

	assert(dir != NULL);

	if (dir->dd_size <= 0) {
		dir->dd_size = ukfs_getdents(dir->dd_ukfs, dir->dd_path,
					     &dir->dd_off,
					     dir->dd_buf, sizeof(dir->dd_buf));
		if (dir->dd_size <= 0)
			return NULL;

		dir->dd_dent = (struct dirent *)dir->dd_buf;
	}

	dent = dir->dd_dent;
	
	dir->dd_size -= _DIRENT_SIZE(dir->dd_dent);
	dir->dd_dent = _DIRENT_NEXT(dir->dd_dent);

	return dent;
}

void
fsu_rewinddir(FSU_DIR *dir)
{
	
	assert(dir != NULL);

	dir->dd_off = 0;
	dir->dd_size = 0;
}

#define MAXLEVEL (25)

/*
 * Gets the absolute path of the current directory as long as it is not deeper
 * than MAXLEVEL
 * I know it is bad but it is in my TODO list
 */
char
*fsu_getcwd(struct ukfs *fs)
{
	int i, rv;
	char *names[MAXLEVEL];
	char *dirname;
	size_t clen, len, maxlen;

	for (rv = i = 0; i < MAXLEVEL; ++i) {
		names[i] = fsu_getdirname(fs);
		if (names[i][0] == '/' && names[i][1] == '\0')
			break;
		rv = ukfs_chdir(fs, "..");
		if (rv == -1 || names[i] == NULL)
			break;
	}
	if (rv == -1) {
		--i;
		for (rv = 0; i >= 0; --i) {
			if (rv == 0)
				rv |= ukfs_chdir(fs, names[i]);
			free(names[i]);
		}
		return NULL;
	}

	if (i == 0)
		return strdup("/");

	maxlen = 1024;
	dirname = malloc(maxlen);
	if (dirname == NULL)
		return NULL;

	dirname[0] = '\0';
	len = 0;
	for (--i; i >= 0; --i) {
		clen = strlen(names[i]);
		if (clen + len + 2 > maxlen) {
			maxlen = 2 * clen + len + 2;
			dirname = realloc(dirname, maxlen);
			if (dirname == NULL)
				return NULL;
		}
		dirname[len++] = '/';
		dirname[len] = '\0';
		len = strlcat(dirname, names[i], maxlen);
		free(names[i]);
	}
	ukfs_chdir(fs, dirname);
	return dirname;
}

/* Gets the name of the current directory */
static char
*fsu_getdirname(struct ukfs *fs)
{
	struct stat dot, dotdot;
	struct dirent *dent;
	int rv, size;
	uint8_t buf[8192];
	off_t off;

	off = 0;
	rv = ukfs_stat(fs, ".", &dot);
	if (rv == -1)
		return NULL;
	rv = ukfs_stat(fs, "..", &dotdot);
	if (rv == -1)
		return NULL;

	if (dot.st_ino == dotdot.st_ino)
		return strdup("/");

	for (;;) {
		size = ukfs_getdents(fs, "..", &off, buf, sizeof(buf));
		if (size == -1) {
			warn("..");
			return NULL;
		} else if (size == 0)
			break;
		dent = (struct dirent *)buf;

		while (size > 0) {
			if (dent->d_fileno == dot.st_ino)
				return strdup(dent->d_name);
			size -= _DIRENT_SIZE(dent);
			dent = _DIRENT_NEXT(dent);
		}
	}
	return NULL;
}
