/* $Id: fsu_mount.c,v 1.12 2008/09/29 16:14:03 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/mount.h>

#include <err.h>
#include <errno.h>
#include <stdbool.h>
#include <string.h>

#include <isofs/cd9660/cd9660_mount.h>
#include <fs/efs/efs_mount.h>
#include <ufs/ufs/ufsmount.h> /* ext2fs, ffs, lfs */
#include <fs/hfs/hfs.h>
#include <msdosfs/msdosfsmount.h>
#include <ntfs/ntfsmount.h>
#include <fs/tmpfs/tmpfs_args.h>
#include <fs/udf/udf_mount.h>

#include <rump/ukfs.h>

#include "mount_cd9660.h"
#include "mount_efs.h"
#include "mount_ext2fs.h"
#include "mount_ffs.h"
#include "mount_hfs.h"
#include "mount_lfs.h"
#include "mount_msdos.h"
#include "mount_ntfs.h"
#include "mount_tmpfs.h"
#include "mount_udf.h"
#include "rump_syspuffs.h"

#include "fsu_mount.h"

#define MAX_FS (20)

struct fsu_fsalias_s {
	char *fsa_name;
	char *fsa_path;
	char *fsa_type;
	char *fsa_mntopt;
	struct fsu_fsalias_s *fsa_next;
};

struct fsu_fsalias_s *alias_head, *alias_tail;

typedef int (*parseargs_fp)(int, char **, void *, int *, char *, char *);

union fsu_args {
	struct iso_args args_cd9660;
	struct efs_args args_efs;
	struct ufs_args args_ufs; /* ext2fs, ffs, lfs, ufs */
	struct hfs_args args_hfs;
	struct msdosfs_args args_msdos;
	struct ntfs_args args_ntfs;
	struct tmpfs_args args_tmpfs;
	struct udf_args args_udf;
	struct syspuffs_args args_syspuffs;
} args;

typedef struct fsu_fs_s {
	const char *fs_name;
	void *fs_args;
	unsigned int fs_args_size;
	parseargs_fp fs_parseargs;
} fsu_fs_t;

fsu_fs_t fslist[] = {
	{ "cd9660",	&args.args_cd9660,	sizeof(struct iso_args),
	  (parseargs_fp)mount_cd9660_parseargs },
	{ "ext2fs",	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ext2fs_parseargs },
	{ "ffs",	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_ffs_parseargs },
	{ "hfs",	&args.args_hfs,		sizeof(struct hfs_args),
	  (parseargs_fp)mount_hfs_parseargs },
	{ "lfs",	&args.args_ufs,		sizeof(struct ufs_args),
	  (parseargs_fp)mount_lfs_parseargs },
	{ "msdos",	&args.args_msdos,	sizeof(struct msdosfs_args),
	  (parseargs_fp)mount_msdos_parseargs },
	{ "ntfs",	&args.args_ntfs,	sizeof(struct ntfs_args),
	  (parseargs_fp)mount_ntfs_parseargs },
	{ "udf",	&args.args_udf,		sizeof(struct udf_args),
	  (parseargs_fp)mount_udf_parseargs },
	{ "tmpfs",	&args.args_tmpfs,	sizeof(struct tmpfs_args),
	  (parseargs_fp)mount_tmpfs_parseargs },
	{ "efs",	&args.args_efs,		sizeof(struct efs_args),
	  (parseargs_fp)mount_efs_parseargs },
	{ "puffs",	&args.args_syspuffs,	sizeof(struct syspuffs_args),
	  (parseargs_fp)mount_syspuffs_parseargs },
	{ NULL,		NULL,			0,  NULL }
};

static int build_alias_list(void);
static void free_alias_list(void);
static struct ukfs *mount_alias(struct fsu_fsalias_s *, int, int *, char ***);

/*
 * Tries to mount an image.
 * if the fstype given is not known or do not works for mounting the image,
 * try every supported types.
 */
struct ukfs *
fsu_mount(int *argc, char **argv[], char **fstp, char **fsdp)
{
	struct ukfs *fs;
	struct stat sb;
	fsu_fs_t *cur;
	struct fsu_fsalias_s *cural;
	int mntflags, mntargc, i, fsd_idx;
	char *fstype, *fsdevice, **mntargv;
	char canon_dev[MAXPATHLEN], canon_dir[MAXPATHLEN];
	bool fstype_dup;
	
	if (*argc < 2) {
		errno = EINVAL;
		return NULL;
	}

	fstype = fsdevice = NULL;
	mntargc = 1;
	cur = NULL;
	cural = NULL;
	fs = NULL;
	fsd_idx = mntflags = 0;
	fstype_dup = false;

	ukfs_init();
	ukfs_modload_dir("/usr/lib");

	i = 1;
	build_alias_list();
try_again:
	fsd_idx = 0;
	for (; i < *argc; ++i) {
		if ((*argv)[i][0] == '-')
			continue;
		if (stat((*argv)[i], &sb) == 0) {
			fsdevice = (*argv)[i];
			fstype = (*argv)[i - 1];
			fsd_idx = i;

			/* look for fstype in the list */
			for (cur = fslist; cur->fs_name != NULL; ++cur)
				if (strcmp(cur->fs_name, fstype) == 0)
					break;
				/* it is not in the list */
			if (cur->fs_name == NULL) {
				fstype = NULL;
				mntargc = fsd_idx + 2;
			} else
				mntargc = fsd_idx + 1;
			break;
		} else {
			/* alias support */
			for (cural = alias_head;
			     cural != NULL;
			     cural = cural->fsa_next) {
				if (strcmp(cural->fsa_name, (*argv)[i]) == 0) {
					/* got it */
					fs =  mount_alias(cural, i, argc, argv);
					if (fs == NULL)
						goto out;
					fstype = cural->fsa_type;
					fsdevice = cural->fsa_path;
					fsd_idx = i;
					goto out;
				}
			}
			cural = NULL;
		}
	}

	if (i == *argc && fsd_idx == 0) {
		fsd_idx = *argc - 1;
		fs = NULL;
		goto out;
	}

	mntargv = malloc((mntargc + 1) * sizeof(char *));
	memset(&args, 0, sizeof(args));

	for (i = 0; i < fsd_idx - 1; ++i)
		mntargv[i] = (*argv)[i];

	if (cur->fs_name == NULL) {
		mntargv[i] = (*argv)[i];
		mntargv[i + 1] = (*argv)[i + 1];
		mntargv[i + 2] = strdup("/");
		mntargv[i + 3] = NULL;
	} else {
		mntargv[i] = (*argv)[i + 1];
		mntargv[i + 1] = strdup("/");
		mntargv[i + 2] = NULL;

		cur->fs_parseargs(mntargc, mntargv, cur->fs_args,
				  &mntflags, canon_dev, canon_dir);
		
		fs = ukfs_mount(fstype, canon_dev, canon_dir, mntflags,
				cur->fs_args, cur->fs_args_size);

		if (fs != NULL)
			goto out;
	}

	i = 0;
	for (cur = fslist; cur->fs_name != NULL; ++cur) {
		mntflags = 0;
		optind = optreset = 1;
		memset(&args, 0, sizeof(union fsu_args));
		i = cur->fs_parseargs(mntargc, mntargv, cur->fs_args,
				  &mntflags, canon_dev, canon_dir);

		if (i != 0)
			continue;

		fs = ukfs_mount(cur->fs_name, canon_dev, canon_dir, mntflags,
				cur->fs_args, cur->fs_args_size);

		if (fs != NULL) {
			fstype = strdup(cur->fs_name);
			fstype_dup = true;
			break;
		}
	}

out:
	optind = optreset = 1;

	if (fs == NULL) {
		/*
		 * With commands like fsu_command -o ro image.iso
		 * if the file ro exists it takes the ro file as the image
		 * so try again to find an existing file
		 */
		if (cural == NULL && fsd_idx < *argc - 1 && i != 0) {
			i = ++fsd_idx;
			free(fstype);
			fstype_dup = false;
			goto try_again;
		} else
			errno = EINVAL;
	} else {
		if (fsdp != NULL)
			*fsdp = strdup(fsdevice);
		if (fstp != NULL) {
			if (fstype_dup)
				*fstp = fstype;
			else
				*fstp = strdup(fstype);
		}
	}

	free_alias_list();
	
	(*argv)[fsd_idx] = (*argv)[0];
	*argv += fsd_idx;
	*argc -= fsd_idx;

	return fs;
}

static int
build_alias_list(void)
{
	struct stat sb;
	struct fsu_fsalias_s *new;
	FILE *fd;
	int i;
	size_t len, off;
	char buf[8192], file[PATH_MAX + 1], *fp, *home;

	fp = getenv("HOME");
	if (fp == NULL) {
		warn("getenv");
		return -1;
	}

	len = strlcpy(file, fp, PATH_MAX + 1);
	file[len] = '/';
	file[len + 1] = '\0';
	strlcat(file, ".fsurc", PATH_MAX + 1);
	if (stat(file, &sb) != 0) {
		return -1;
	}

	fd = fopen(file, "r");
	if (fd == NULL) {
		warn(file);
		return -1;
	}
	for (;;) {
		/* get a line */
		for (i = 0; i < sizeof(buf) &&
			     (buf[i] = fgetc(fd)) != '\n' && buf[i] != EOF;
		     i++)
			continue;
		buf[i] = '\0';
		if (buf[0] == '#')
			continue;
		if (i == 0 && feof(fd))
			return 0;

		new = malloc(sizeof(struct fsu_fsalias_s));
		if (new == NULL) {
			warn("malloc");
			return -1;
		}

		fp = strtok(buf, ":");
		if (fp == NULL)
			continue;
		if (fp[0] == '-') {
			fprintf(stderr,
			  "alias starting with a dash ('-') are not suported");
			continue;
		}
		new->fsa_name = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;
		if (fp[0] == '~' && fp[1] == '/') {
			home = getenv("HOME");

			if (home == NULL) {
				warn("getenv");
				continue;
			}
			len = strlen(home) + strlen(fp);
			new->fsa_path = malloc(len);
			if (new->fsa_path == NULL) {
				warn("malloc");
				continue;
			}
			off = strlcpy(new->fsa_path, home, len);
			new->fsa_path[off] = '/';
			new->fsa_path[off + 1] = '\0';
			strlcat(new->fsa_path, fp + 2, len);
		} else
			new->fsa_path = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			continue;
		new->fsa_type = strdup(fp);

		fp = strtok(NULL, ":");
		if (fp == NULL)
			new->fsa_mntopt = NULL;
		else
			new->fsa_mntopt = strdup(fp);

		new->fsa_next = NULL;
		
		if (alias_head == NULL)
			alias_head = alias_tail = new;
		else
			alias_tail = alias_tail->fsa_next = new;
	}

	fclose(fd);
	return 0;
}

static void
free_alias_list(void)
{
	struct fsu_fsalias_s *cur;

	if (alias_tail == NULL)
		return;

	while ((cur = alias_head) != NULL) {
		alias_head = alias_head->fsa_next;
		free(cur->fsa_name);
		free(cur->fsa_path);
		free(cur->fsa_type);
		if (cur->fsa_mntopt != NULL)
			free(cur->fsa_mntopt);
		free(cur);
	}
}

static struct ukfs
*mount_alias(struct fsu_fsalias_s *al, int idx, int *argc, char ***argv)
{
	fsu_fs_t *cur;
	int mntargc, i, j, mntflags;
	char *mntargv[30], canon_dev[MAXPATHLEN], canon_dir[MAXPATHLEN];

	mntflags = 0;
	mntargv[0] = (*argv)[0];
	mntargv[1] = strtok(al->fsa_mntopt, " ");

	i = 2;
	if (mntargv[1] != NULL)
		for (; i < sizeof(mntargv) && mntargv[i - 1] != NULL; ++i)
			mntargv[i] = strtok(NULL, " ");
	i--;
	for (j = 1; j < idx; ++i, ++j)
		mntargv[i] = (*argv)[j];
	
	mntargv[i++] = al->fsa_path;
	mntargv[i++] = strdup("/");
	mntargv[i] = NULL;
	
	mntargc = i;

	for (cur = fslist; cur->fs_name != NULL; ++cur)
		if (strcmp(cur->fs_name, al->fsa_type) == 0) {
			i = cur->fs_parseargs(mntargc, mntargv, cur->fs_args,
					      &mntflags, canon_dev, canon_dir);
			if (i != 0)
				return NULL;
			break;
		}
	if (cur->fs_name == NULL)
		return NULL;

	return ukfs_mount(cur->fs_name, canon_dev, canon_dir, mntflags,
			  cur->fs_args, cur->fs_args_size);
}
