/* $Id: main.c,v 1.6 2008/09/23 16:20:42 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <err.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/param.h>
#include <sys/vnode.h>

#include <rump/ukfs.h>

#include <fsu_mount.h>

#include "fsu_mknod.h"
#include "pack_dev.h"

static void usage(void);

int
main(int argc, char *argv[])
{
	int ch, flags;
	gid_t gid = -1;
	uid_t uid = -1;
	int hasformat = 0, rv;
	pack_t *pack;
	char *p;
	void *modes = NULL;
	struct ukfs *fs = NULL;

	if (argc < 2)
		usage();

	fs = fsu_mount(&argc, &argv, NULL, NULL);
	if (fs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}

	pack = pack_native;
	flags = 0;
	while ((ch = getopt(argc, argv, "rRF:g:m:u:")) != -1) {
		switch (ch) {
		case 'R':
			flags |= FSU_MKNOD_RFLAG;
			break;
		case 'r':
			flags |= FSU_MKNOD_R2FLAG;
			break;
		case 'F':
			pack = pack_find(optarg);
			if (pack == NULL)
				errx(1, "invalid format: %s", optarg);
			hasformat++;
			break;

		case 'g':
			if (optarg[0] == '#') {
				gid = strtol(optarg + 1, &p, 10);
				if (*p == 0)
					break;
			}
			if (gid_from_group(optarg, &gid) == 0)
				break;
			gid = strtol(optarg, &p, 10);
			if (*p == 0)
				break;
			errx(1, "%s: invalid group name", optarg);

		case 'm':
			modes = setmode(optarg);
			if (modes == NULL)
				err(1, "Cannot set file mode `%s'", optarg);
			break;

		case 'u':
			if (optarg[0] == '#') {
				uid = strtol(optarg + 1, &p, 10);
				if (*p == 0)
					break;
			}
			if (uid_from_user(optarg, &uid) == 0)
				break;
			uid = strtol(optarg, &p, 10);
			if (*p == 0)
				break;
			errx(1, "%s: invalid user name", optarg);

		default:
		case '?':
			usage();
			return EXIT_FAILURE;
		}
	}
	argc -= optind;
	argv += optind;


	if (argc < 2 || argc > 12) {
		usage();
		return EXIT_FAILURE;
	}

	rv = fsu_mknod(fs, argc, argv, uid, gid, hasformat, pack, modes, flags);
	
	ukfs_release(fs, 0);

	return rv;
}

static void
usage(void)
{

	fprintf(stderr,
		"usage: %s [MNT_OPTIONS] [fstype] fsdevice [-rR] [-F format] [-m mode] [-u user] [-g group]\n",
		getprogname());
	fprintf(stderr,
		"                   [ name [b | c] major minor\n"
		"                   | name [b | c] major unit subunit\n"
		"                   | name [b | c] number\n"
		"                   | name p ]\n");
	exit(EXIT_FAILURE);
}
