/* $Id: fsu_mknod.c,v 1.4 2008/09/23 16:20:42 stacktic Exp $ */
/* from */
/*	$NetBSD: mknod.c,v 1.37 2008/04/28 20:23:09 martin Exp $	*/

/*-
 * Copyright (c) 1998, 2001 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Charles M. Hannum.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_NBTOOL_CONFIG_H
#include "nbtool_config.h"
#endif

#include <sys/cdefs.h>

#ifndef lint
/*__COPYRIGHT("@(#) Copyright (c) 1998 The NetBSD Foundation, Inc.  All rights reserved.\n");*/
__RCSID("$NetBSD: mknod.c,v 1.37 2008/04/28 20:23:09 martin Exp $");
#endif /* not lint */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#if !HAVE_NBTOOL_CONFIG_H
#include <sys/sysctl.h>
#endif

#include <err.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <ctype.h>

#include <rump/ukfs.h>

#include <fsu_mount.h>

#include "fsu_mknod.h"
#include "pack_dev.h"

static portdev_t callPack(pack_t *, int, u_long *);

static void	usage(void);

#define	MAXARGS	3		/* 3 for bsdos, 2 for rest */

static char *progname;

int
fsu_mknod_main(struct ukfs *fs, int argc, char *argv[])
{
	int ch, flags;
	gid_t gid = -1;
	uid_t uid = -1;
	int hasformat = 0, rv;
	pack_t *pack;
	char *p;
	void *modes = NULL;

	progname = argv[0];
	pack = pack_native;
	flags = 0;
	while ((ch = getopt(argc, argv, "rRF:g:m:u:")) != -1) {
		switch (ch) {
		case 'R':
			flags |= FSU_MKNOD_RFLAG;
			break;
		case 'r':
			flags |= FSU_MKNOD_R2FLAG;
			break;
		case 'F':
			pack = pack_find(optarg);
			if (pack == NULL) {
				warnx("invalid format: %s", optarg);
				return -1;
			}
			hasformat++;
			break;
		case 'g':
			if (optarg[0] == '#') {
				gid = strtol(optarg + 1, &p, 10);
				if (*p == 0)
					break;
			}
			if (gid_from_group(optarg, &gid) == 0)
				break;
			gid = strtol(optarg, &p, 10);
			if (*p == 0)
				break;
			warnx("%s: invalid group name", optarg);
			return -1;
		case 'm':
			modes = setmode(optarg);
			if (modes == NULL) {
				warn("Cannot set file mode `%s'", optarg);
				return -1;
			}
			break;
		case 'u':
			if (optarg[0] == '#') {
				uid = strtol(optarg + 1, &p, 10);
				if (*p == 0)
					break;
			}
			if (uid_from_user(optarg, &uid) == 0)
				break;
			uid = strtol(optarg, &p, 10);
			if (*p == 0)
				break;
			warnx("%s: invalid user name", optarg);
			return -1;
		default:
		case '?':
			usage();
			return -1;
		}
	}
	argc -= optind;
	argv += optind;
	
	rv = fsu_mknod(fs, argc, argv, uid, gid, hasformat, pack, modes, flags);
	
	return rv;
}

int
fsu_mknod(struct ukfs *fs, int argc, char **argv, uid_t uid, gid_t gid,
	  int hasformat, pack_t *pack, void *modes, int flags)
{
	char	*name, *p;
	mode_t	 mode;
	portdev_t	 dev;
	u_long	 numbers[MAXARGS];
	int	 n, fifo;
	int	 rval;

	dev = 0;
	fifo = 0;

	if (argc < 2 || argc > 10) {
		usage();
		return -1;
	}

	name = *argv;
	argc--;
	argv++;

	umask(mode = umask(0));
	mode = (S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH) & ~mode;

	if (argv[0][1] != '\0')
		goto badtype;
	switch (*argv[0]) {
	case 'c':
		mode |= S_IFCHR;
		break;

	case 'b':
		mode |= S_IFBLK;
		break;

	case 'p':
		if (hasformat) {
			warnx("format is meaningless for fifos");
			return -1;
		}
		mode |= S_IFIFO;
		fifo = 1;
		break;

	default:
 badtype:
		warnx("node type must be 'b', 'c' or 'p'.");
		return -1;
	}
	argc--;
	argv++;

	if (fifo) {
		if (argc != 0) {
			usage();
			return -1;
		}
	} else {
		if (argc < 1 || argc > MAXARGS) {
			usage();
			return -1;
		}
	}

	for (n = 0; n < argc; n++) {
		errno = 0;
		numbers[n] = strtoul(argv[n], &p, 0);
		if (*p == 0 && errno == 0)
			continue;
		warnx("invalid number: %s", argv[n]);
		return -1;
	}

	switch (argc) {
	case 0:
		dev = 0;
		break;
	case 1:
		dev = numbers[0];
		break;
	default:
		dev = callPack(pack, argc, numbers);
		if (dev == -1)
			return -1;
		break;
	}

	if (modes != NULL)
		mode = getmode(modes, mode);
	umask(0);
	rval = fifo ? ukfs_mkfifo(fs, name, mode) :
		ukfs_mknod(fs, name, mode, dev);
	if (rval < 0 && errno == EEXIST &&
	    (flags & (FSU_MKNOD_R2FLAG | FSU_MKNOD_RFLAG))) {
		struct stat sb;
		if (ukfs_lstat(fs, name, &sb) != 0 ||
		    (!fifo && sb.st_rdev != dev))
			sb.st_mode = 0;

		if ((sb.st_mode & S_IFMT) == (mode & S_IFMT)) {
			if (flags & FSU_MKNOD_R2FLAG) {
				/* Ignore permissions and user/group */
				rval = 0;
				goto out;
			}
			if (sb.st_mode != mode)
				rval = ukfs_chmod(fs, name, mode);
			else
				rval = 0;
		} else {
			ukfs_remove(fs, name);
			rval = fifo ? ukfs_mkfifo(fs, name, mode)
				: ukfs_mknod(fs, name, mode, dev);
		}
	}
	if (rval < 0) {
		warn("%s", name);
		rval = 1;
		goto out;
	}
	if ((uid != -1 || gid != -1) && ukfs_chown(fs, name, uid, gid) == -1)
		/* XXX Should we unlink the files here? */
		warn("%s: uid/gid not changed", name);

	
out:
	return rval;
}

static void
usage(void)
{

	fprintf(stderr,
	   "usage: %s [-rR] [-F format] [-m mode] [-u user] [-g group]\n"
		"                   [ name [b | c] major minor\n"
		"                   | name [b | c] major unit subunit\n"
		"                   | name [b | c] number\n"
		"                   | name p ]\n", progname);
}

static portdev_t
callPack(pack_t *f, int n, u_long *numbers)
{
	portdev_t d;
	const char *error = NULL;

	d = (*f)(n, numbers, &error);
	if (error != NULL) {
		warnx("%s", error);
		return -1;
	}
	return d;
}
