/*	$NetBSD: print.c,v 1.42 2006/12/14 14:15:26 christos Exp $	*/

/*
 * Copyright (c) 1989, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Michael Fischbein.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#ifndef lint
#if 0
static char sccsid[] = "@(#)print.c	8.5 (Berkeley) 7/28/94";
#else
__RCSID("$NetBSD: print.c,v 1.42 2006/12/14 14:15:26 christos Exp $");
#endif
#endif /* not lint */

#include <sys/param.h>
#include <sys/stat.h>

#include <err.h>
#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <tzfile.h>
#include <unistd.h>
#include <util.h>

#include <rump/ukfs.h>

#include "fsu_fts.h"

#include "fsu_ls.h"
#include "extern.h"

extern int ls_termwidth;

static int	printaname(FSU_FTSENT *, int, int);
static void	printlink(FSU_FTSENT *);
static void	printtime(time_t);
static int	printtype(u_int);

static time_t	now;

#define	IS_NOPRINT(p)	((p)->fts_number == NO_PRINT)

void
printscol(DISPLAY *dp)
{
	FSU_FTSENT *p;

	for (p = dp->list; p; p = p->fts_link) {
		if (IS_NOPRINT(p))
			continue;
		(void)printaname(p, dp->s_inode, dp->s_block);
		(void)putchar('\n');
		if (ls_exit)
			return;
	}
}

void
printlong(DISPLAY *dp)
{
	struct stat *sp;
	FSU_FTSENT *p;
	NAMES *np;
	char buf[20], szbuf[5];

	now = time(NULL);

	if (dp->list->fts_level != FTS_ROOTLEVEL && (ls_f_longform || ls_f_size)) {
		if (ls_f_humanize) {
			if ((humanize_number(szbuf, sizeof(szbuf), (int64_t)dp->stotal,
			    "", HN_AUTOSCALE,
			    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
				warn("humanize_number");
				ls_exit = true;
				return;
			}
			(void)printf("total %s\n", szbuf);
		} else {
			(void)printf("total %llu\n",
			    (long long)(howmany(dp->btotal, ls_blocksize)));
		}
	}

	for (p = dp->list; p; p = p->fts_link) {
		if (IS_NOPRINT(p))
			continue;
		sp = p->fts_statp;
		if (ls_f_inode)
			(void)printf("%*lu ", dp->s_inode,
			    (unsigned long)sp->st_ino);
		if (ls_f_size) {
			if (ls_f_humanize) {
				if ((humanize_number(szbuf, sizeof(szbuf),
					sp->st_blocks * S_BLKSIZE,
			    "", HN_AUTOSCALE,
			    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
					warn("humanize_number");
					ls_exit = true;
					return;
				}
			(void)printf("%*s ", dp->s_block, szbuf);
			} else {
			(void)printf("%*llu ", dp->s_block,
					(long long)howmany(sp->st_blocks,
							   	ls_blocksize));
			}
		}
		(void)strmode(sp->st_mode, buf);
		np = p->fts_pointer;
		(void)printf("%s %*lu ", buf, dp->s_nlink,
		    (unsigned long)sp->st_nlink);
		if (!ls_f_grouponly)
			(void)printf("%-*s  ", dp->s_user, np->user);
		(void)printf("%-*s  ", dp->s_group, np->group);
		if (ls_f_flags)
			(void)printf("%-*s ", dp->s_flags, np->flags);
		if (S_ISCHR(sp->st_mode) || S_ISBLK(sp->st_mode))
			(void)printf("%*u, %*u ",
			    dp->s_major, major(sp->st_rdev), dp->s_minor,
			    minor(sp->st_rdev));
		else
			if (ls_f_humanize) {
				if ((humanize_number(szbuf, sizeof(szbuf),
				    sp->st_size, "", HN_AUTOSCALE,
				    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
					warn("humanize_number");
					ls_exit = true;
					return;
				}
				(void)printf("%*s ", dp->s_size, szbuf);
			} else {
				(void)printf("%*llu ", dp->s_size,
				    (long long)sp->st_size);
			}
		if (ls_f_accesstime)
			printtime(sp->st_atime);
		else if (ls_f_statustime)
			printtime(sp->st_ctime);
		else
			printtime(sp->st_mtime);
		if (ls_f_octal || ls_f_octal_escape)
			(void)safe_print(p->fts_name);
		else if (ls_f_nonprint)
			(void)printescaped(p->fts_name);
		else
			(void)printf("%s", p->fts_name);
		if (ls_exit)
			return;
		if (ls_f_type || (ls_f_typedir && S_ISDIR(sp->st_mode)))
			(void)printtype(sp->st_mode);
		if (S_ISLNK(sp->st_mode))
			printlink(p);
		(void)putchar('\n');
		if (ls_exit)
			return;
	}
}

void
printcol(DISPLAY *dp)
{
	static FSU_FTSENT **array;
	static int lastentries = -1;
	FSU_FTSENT *p;
	int base, chcnt, col, colwidth, num;
	int numcols, numrows, row;
	char szbuf[5];

	colwidth = dp->maxlen;
	if (ls_f_inode)
		colwidth += dp->s_inode + 1;
	if (ls_f_size) {
		if (ls_f_humanize)
			colwidth += dp->s_size + 1;
		else
			colwidth += dp->s_block + 1;
	}
	if (ls_f_type || ls_f_typedir)
		colwidth += 1;

	colwidth += 1;

	if (ls_termwidth < 2 * colwidth) {
		printscol(dp);
		return;
	}

	/*
	 * Have to do random access in the linked list -- build a table
	 * of pointers.
	 */
	if (dp->entries > lastentries) {
		lastentries = dp->entries;
		if ((array =
		    realloc(array, dp->entries * sizeof(FSU_FTSENT *))) == NULL) {
			warn(NULL);
			printscol(dp);
		}
	}
	for (p = dp->list, num = 0; p; p = p->fts_link)
		if (p->fts_number != NO_PRINT)
			array[num++] = p;

	numcols = ls_termwidth / colwidth;
	colwidth = ls_termwidth / numcols;		/* spread out if possible */
	numrows = num / numcols;
	if (num % numcols)
		++numrows;

	if (dp->list->fts_level != FTS_ROOTLEVEL && (ls_f_longform || ls_f_size)) {
		if (ls_f_humanize) {
			if ((humanize_number(szbuf, sizeof(szbuf), (int64_t)dp->stotal,
			    "", HN_AUTOSCALE,
			    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
				warn("humanize_number");
				ls_exit = true;
				return;
			}
			(void)printf("total %s\n", szbuf);
		} else {
			(void)printf("total %llu\n",
			    (long long)(howmany(dp->btotal, ls_blocksize)));
		}
	}
	for (row = 0; row < numrows; ++row) {
		for (base = row, chcnt = col = 0; col < numcols; ++col) {
			chcnt = printaname(array[base], dp->s_inode,
			    ls_f_humanize ? dp->s_size : dp->s_block);
			if (ls_exit)
				return;
			if ((base += numrows) >= num)
				break;
			while (chcnt++ < colwidth)
				(void)putchar(' ');
		}
		(void)putchar('\n');
	}
}

void
printacol(DISPLAY *dp)
{
	FSU_FTSENT *p;
	int chcnt, col, colwidth;
	int numcols;
	char szbuf[5];

	colwidth = dp->maxlen;
	if (ls_f_inode)
		colwidth += dp->s_inode + 1;
	if (ls_f_size) {
		if (ls_f_humanize)
			colwidth += dp->s_size + 1;
		else
			colwidth += dp->s_block + 1;
	}
	if (ls_f_type || ls_f_typedir)
		colwidth += 1;

	colwidth += 1;

	if (ls_termwidth < 2 * colwidth) {
		printscol(dp);
		return;
	}

	numcols = ls_termwidth / colwidth;
	colwidth = ls_termwidth / numcols;		/* spread out if possible */

	if (dp->list->fts_level != FTS_ROOTLEVEL && (ls_f_longform || ls_f_size)) {
		if (ls_f_humanize) {
			if ((humanize_number(szbuf, sizeof(szbuf), (int64_t)dp->stotal,
			    "", HN_AUTOSCALE,
			    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
				warn("humanize_number");
				ls_exit = true;
				return;
			}
			(void)printf("total %s\n", szbuf);
		} else {
			(void)printf("total %llu\n",
			    (long long)(howmany(dp->btotal, ls_blocksize)));
		}
	}
	chcnt = col = 0;
	for (p = dp->list; p; p = p->fts_link) {
		if (IS_NOPRINT(p))
			continue;
		if (col >= numcols) {
			chcnt = col = 0;
			(void)putchar('\n');
		}
		chcnt = printaname(p, dp->s_inode,
		    ls_f_humanize ? dp->s_size : dp->s_block);
		if (ls_exit)
			return;
		while (chcnt++ < colwidth)
			(void)putchar(' ');
		col++;
	}
	(void)putchar('\n');
}

void
printstream(DISPLAY *dp)
{
	FSU_FTSENT *p;
	int col;
	int extwidth;

	extwidth = 0;
	if (ls_f_inode)
		extwidth += dp->s_inode + 1;
	if (ls_f_size) {
		if (ls_f_humanize)
			extwidth += dp->s_size + 1;
		else
			extwidth += dp->s_block + 1;
	}
	if (ls_f_type)
		extwidth += 1;

	for (col = 0, p = dp->list; p != NULL; p = p->fts_link) {
		if (IS_NOPRINT(p))
			continue;
		if (col > 0) {
			(void)putchar(','), col++;
			if (col + 1 + extwidth + p->fts_namelen >= ls_termwidth)
				(void)putchar('\n'), col = 0;
			else
				(void)putchar(' '), col++;
		}
		col += printaname(p, dp->s_inode,
		    ls_f_humanize ? dp->s_size : dp->s_block);
		if (ls_exit)
			return;
	}
	(void)putchar('\n');
}

/*
 * print [inode] [size] name
 * return # of characters printed, no trailing characters.
 */
static int
printaname(FSU_FTSENT *p, int inodefield, int sizefield)
{
	struct stat *sp;
	int chcnt;
	char szbuf[5];

	sp = p->fts_statp;
	chcnt = 0;
	if (ls_f_inode)
		chcnt += printf("%*lu ", inodefield, (unsigned long)sp->st_ino);
	if (ls_f_size) {
		if (ls_f_humanize) {
			if ((humanize_number(szbuf, sizeof(szbuf), sp->st_size,
			    "", HN_AUTOSCALE,
			    (HN_DECIMAL | HN_B | HN_NOSPACE))) == -1) {
				warn("humanize_number");
				ls_exit = true;
				return -1;
			}
			chcnt += printf("%*s ", sizefield, szbuf);
		} else {
			chcnt += printf("%*llu ", sizefield,
			    (long long)howmany(sp->st_blocks, ls_blocksize));
		}
	}
	if (ls_f_octal || ls_f_octal_escape) {
		chcnt += safe_print(p->fts_name);
		if (ls_exit)
			return -1;
	} else if (ls_f_nonprint)
		chcnt += printescaped(p->fts_name);
	else
		chcnt += printf("%s", p->fts_name);
	if (ls_f_type || (ls_f_typedir && S_ISDIR(sp->st_mode)))
		chcnt += printtype(sp->st_mode);
	return (chcnt);
}

static void
printtime(time_t ftime)
{
	int i;
	char *longstring;

	longstring = ctime(&ftime);
	for (i = 4; i < 11; ++i)
		(void)putchar(longstring[i]);

#define	SIXMONTHS	((DAYSPERNYEAR / 2) * SECSPERDAY)
	if (ls_f_sectime)
		for (i = 11; i < 24; i++)
			(void)putchar(longstring[i]);
	else if (ftime + SIXMONTHS > now && ftime - SIXMONTHS < now)
		for (i = 11; i < 16; ++i)
			(void)putchar(longstring[i]);
	else {
		(void)putchar(' ');
		for (i = 20; i < 24; ++i)
			(void)putchar(longstring[i]);
	}
	(void)putchar(' ');
}

static int
printtype(u_int mode)
{
	switch (mode & S_IFMT) {
	case S_IFDIR:
		(void)putchar('/');
		return (1);
	case S_IFIFO:
		(void)putchar('|');
		return (1);
	case S_IFLNK:
		(void)putchar('@');
		return (1);
	case S_IFSOCK:
		(void)putchar('=');
		return (1);
	case S_IFWHT:
		(void)putchar('%');
		return (1);
	}
	if (mode & (S_IXUSR | S_IXGRP | S_IXOTH)) {
		(void)putchar('*');
		return (1);
	}
	return (0);
}

static void
printlink(FSU_FTSENT *p)
{
	int lnklen;
	char name[MAXPATHLEN + 1], path[MAXPATHLEN + 1];
	extern struct ukfs *ls_ukfs;

	if (p->fts_level == FTS_ROOTLEVEL)
		(void)snprintf(name, sizeof(name), "%s", p->fts_name);
	else
		(void)snprintf(name, sizeof(name),
		    "%s/%s", p->fts_parent->fts_accpath, p->fts_name);
	if ((lnklen = ukfs_readlink(ls_ukfs, name, path,
				    sizeof(path) - 1)) == -1) {
		(void)fprintf(stderr, "\nls: %s: %s\n", name, strerror(errno));
		return;
	}
	path[lnklen] = '\0';
	(void)printf(" -> ");
	if (ls_f_octal || ls_f_octal_escape) {
		(void)safe_print(path);
		if (ls_exit)
			return;
	} else if (ls_f_nonprint)
		(void)printescaped(path);
	else
		(void)printf("%s", path);
}
