/* $NetBSD: utils.c,v 1.34 2007/10/26 16:21:25 hira Exp $ */

/*-
 * Copyright (c) 1991, 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#ifndef lint
#if 0
static char sccsid[] = "@(#)utils.c	8.3 (Berkeley) 4/1/94";
#else
__RCSID("$NetBSD: utils.c,v 1.34 2007/10/26 16:21:25 hira Exp $");
#endif
#endif /* not lint */

#include <sys/mman.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <rump/ukfs.h>

#include "fsu_fts.h"

#include "extern.h"

int
set_utimes(const char *file, struct stat *fs)
{
    static struct timeval tv[2];

    TIMESPEC_TO_TIMEVAL(&tv[0], &fs->st_atimespec);
    TIMESPEC_TO_TIMEVAL(&tv[1], &fs->st_mtimespec);

    if (ukfs_lutimes(cp_ukfs, file, tv)) {
	warn("lutimes: %s", file);
	return (1);
    }
    return (0);
}

int
copy_file(FSU_FTSENT *entp, int dne)
{
	static uint8_t buf[MAXBSIZE];
	struct stat to_stat, *fs;
	int ch, checkch, rv, rval, tolnk;
	off_t off;
	ssize_t rcount, wcount;

	fs = entp->fts_statp;
	tolnk = ((cp_Rflag && !(cp_Lflag || cp_Hflag)) || cp_Pflag);

	/*
	 * If the file exists and we're interactive, verify with the user.
	 * If the file DNE, set the mode to be the from file, minus setuid
	 * bits, modified by the umask; arguably wrong, but it makes copying
	 * executables work right and it's been that way forever.  (The
	 * other choice is 666 or'ed with the execute bits on the from file
	 * modified by the umask.)
	 */
	if (!dne) {
		if (cp_iflag) {
			(void)fprintf(stderr, "overwrite %s? ", cp_to.p_path);
			checkch = ch = getchar();
			while (ch != '\n' && ch != EOF)
				ch = getchar();
			if (checkch != 'y' && checkch != 'Y') {
				/*(void)close(from_fd);*/
				return (0);
			}
		}
		ukfs_remove(cp_ukfs, cp_to.p_path);
		
	}
	rv = ukfs_create(cp_ukfs, cp_to.p_path,
			 fs->st_mode & ~(S_ISUID | S_ISGID));
	
	if (rv == -1 && (cp_fflag || tolnk)) {
		/*
		 * attempt to remove existing destination file name and
		 * create a new file
		 */
		ukfs_remove(cp_ukfs, cp_to.p_path);
		rv = ukfs_create(cp_ukfs, cp_to.p_path,
				 fs->st_mode & ~(S_ISUID | S_ISGID));
	}

	if (rv == -1) {
		warn("%s", cp_to.p_path);
		return (1);
	}

	rval = 0;

	/*
	 * There's no reason to do anything other than close the file
	 * now if it's empty, so let's not bother.
	 */
	off = 0;
	if (fs->st_size > 0) {
		while ((rcount = ukfs_read(cp_ukfs, entp->fts_path,
					   off, buf, MAXBSIZE)) > 0) {
			wcount = ukfs_write(cp_ukfs, cp_to.p_path,
					    off, buf, (size_t)rcount);
			if (rcount != wcount || wcount == -1) {
				warn("%s", cp_to.p_path);
				rval = 1;
				break;
			}
			off += rcount;
		}
		if (rcount < 0) {
			warn("%s", entp->fts_path);
			rval = 1;
		}
	}

	if (rval == 1)
		return (1);

	if (cp_pflag && setfile(fs))
		rval = 1;
	/*
	 * If the source was setuid or setgid, lose the bits unless the
	 * copy is owned by the same user and group.
	 */
#define	RETAINBITS \
	(S_ISUID | S_ISGID | S_ISVTX | S_IRWXU | S_IRWXG | S_IRWXO)
	if (!cp_pflag && dne
	    && fs->st_mode & (S_ISUID | S_ISGID) && fs->st_uid == cp_myuid) {
		if (ukfs_stat(cp_ukfs, cp_to.p_path, &to_stat)) {
			warn("%s", cp_to.p_path);
			rval = 1;
		} else if (fs->st_gid == to_stat.st_gid &&
			   ukfs_chmod(cp_ukfs, cp_to.p_path,
				      fs->st_mode & RETAINBITS & ~cp_myumask)) {
			warn("%s", cp_to.p_path);
			rval = 1;
		}
	}

	/* set the mod/access times now after close of the fd */
	if (cp_pflag && set_utimes(cp_to.p_path, fs)) {
	    rval = 1;
	}
	return (rval);
}

int
copy_link(FSU_FTSENT *p, int exists)
{
	int len;
	char target[MAXPATHLEN];

	if ((len = ukfs_readlink(cp_ukfs, p->fts_path, target,
				 sizeof(target)-1)) == -1) {
		warn("readlink: %s", p->fts_path);
		return (1);
	}
	target[len] = '\0';
	if (exists && ukfs_remove(cp_ukfs, cp_to.p_path)) {
		warn("unlink: %s", cp_to.p_path);
		return (1);
	}
	if (ukfs_symlink(cp_ukfs, target, cp_to.p_path)) {
		warn("symlink: %s", target);
		return (1);
	}
	return (cp_pflag ? setfile(p->fts_statp) : 0);
}

int
copy_fifo(struct stat *from_stat, int exists)
{
	if (exists && ukfs_remove(cp_ukfs, cp_to.p_path)) {
		warn("unlink: %s", cp_to.p_path);
		return (1);
	}
	if (ukfs_mkfifo(cp_ukfs, cp_to.p_path, from_stat->st_mode)) {
		warn("mkfifo: %s", cp_to.p_path);
		return (1);
	}
	return (cp_pflag ? setfile(from_stat) : 0);
}

int
copy_special(struct stat *from_stat, int exists)
{
	if (exists && ukfs_remove(cp_ukfs, cp_to.p_path)) {
		warn("unlink: %s", cp_to.p_path);
		return (1);
	}
	if (ukfs_mknod(cp_ukfs, cp_to.p_path, from_stat->st_mode, from_stat->st_rdev)) {
		warn("mknod: %s", cp_to.p_path);
		return (1);
	}
	return (cp_pflag ? setfile(from_stat) : 0);
}


/*
 * Function: setfile
 *
 * Purpose:
 *   Set the owner/group/permissions for the "to" file to the information
 *   in the stat structure.  If fd is zero, also call set_utimes() to set
 *   the mod/access times.  If fd is non-zero, the caller must do a utimes
 *   itself after close(fd).
 */
int
setfile(struct stat *fs)
{
	int rval, islink;

	rval = 0;
	islink = S_ISLNK(fs->st_mode);
	fs->st_mode &= S_ISUID | S_ISGID | S_IRWXU | S_IRWXG | S_IRWXO;

	/*
	 * Changing the ownership probably won't succeed, unless we're root
	 * or POSIX_CHOWN_RESTRICTED is not set.  Set uid/gid before setting
	 * the mode; current BSD behavior is to remove all setuid bits on
	 * chown.  If chown fails, lose setuid/setgid bits.
	 */
	if (ukfs_lchown(cp_ukfs, cp_to.p_path, fs->st_uid, fs->st_gid)) {
		if (errno != EPERM) {
			warn("chown: %s", cp_to.p_path);
			rval = 1;
		}
		fs->st_mode &= ~(S_ISUID | S_ISGID);
	}
	if (ukfs_lchmod(cp_ukfs, cp_to.p_path, fs->st_mode)) {
		warn("chmod: %s", cp_to.p_path);
		rval = 1;
	}

	if (!islink && !cp_Nflag) {
		unsigned long fflags = fs->st_flags;
		/*
		 * XXX
		 * NFS doesn't support chflags; ignore errors unless
		 * there's reason to believe we're losing bits.
		 * (Note, this still won't be right if the server
		 * supports flags and we were trying to *remove* flags
		 * on a file that we copied, i.e., that we didn't create.)
		 */
		errno = 0;
		if (ukfs_chflags(cp_ukfs, cp_to.p_path, fflags) == -1)
			if (errno != EOPNOTSUPP || fs->st_flags != 0) {
				warn("chflags: %s", cp_to.p_path);
				rval = 1;
			}
	}
	/* if fd is non-zero, caller must call set_utimes() after close() */
	if (set_utimes(cp_to.p_path, fs))
	    rval = 1;
	return (rval);
}

