/*	$NetBSD: chown.c,v 1.32 2008/07/21 13:36:57 lukem Exp $	*/

/*
 * Copyright (c) 1988, 1993, 1994, 2003
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#ifndef lint
__COPYRIGHT("@(#) Copyright (c) 1988, 1993, 1994, 2003\
 The Regents of the University of California.  All rights reserved.");
#endif /* not lint */

#ifndef lint
#if 0
static char sccsid[] = "@(#)chown.c	8.8 (Berkeley) 4/4/94";
#else
__RCSID("$NetBSD: chown.c,v 1.32 2008/07/21 13:36:57 lukem Exp $");
#endif
#endif /* not lint */

#include <sys/types.h>
#include <sys/stat.h>

#include <ctype.h>
#include <dirent.h>
#include <err.h>
#include <errno.h>
#include <locale.h>
#include <grp.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#if defined(FSU_CONSOLE) && !defined(FSU)
#define FSU
#endif

#ifdef FSU
#include <rump/ukfs.h>

#include <fts2fsufts.h>
#include <fsu_mount.h>

#ifndef FSU_CONSOLE
static struct ukfs *ukfs;

static void	unmount_ukfs(void);
#else /* !FSU_CONSOLE */

#define main(a, b) fsu_chown_main(struct ukfs *ukfs, a, b)

int fsu_chown_main(struct ukfs *, int, char *[]);

static int chown_exit = 0;
#endif /* !FSU_CONSOLE */

#else /* FSU */
#include <fts.h>
#endif /* FSU */

static void	a_gid(const char *);
static void	a_uid(const char *);
static id_t	id(const char *, const char *);
static void	usage(void);

static uid_t uid;
static gid_t gid;
static int ischown;
static char *myname;

int
main(int argc, char **argv)
{
	FTS *ftsp;
	FTSENT *p;
	int Hflag, Lflag, Rflag, ch, fflag, fts_options, hflag, rval, vflag;
	int error;
	char *cp;
#ifdef FSU
	int (*change_owner)(struct ukfs *, const char *, uid_t, gid_t);
#else
	int (*change_owner)(const char *, uid_t, gid_t);
#endif

	(void)setlocale(LC_ALL, "");

	myname = (cp = strrchr(*argv, '/')) ? cp + 1 : *argv;
#if defined(FSU) && !defined(FSU_CONSOLE)
	ischown = (myname[6] == 'o');
#else
	ischown = (myname[2] == 'o');
#endif

#if defined(FSU) && !defined(FSU_CONSOLE)
	if (argc < 2)
		usage();

	ukfs = fsu_mount(&argc, &argv, NULL, NULL);
	if (ukfs == NULL) {
		warn("fsu_mount");
		usage();
		/* NOTREACHED */
	}
	atexit(unmount_ukfs);
#endif

	Hflag = Lflag = Rflag = fflag = hflag = vflag = 0;
	while ((ch = getopt(argc, argv, "HLPRfhv")) != -1)
		switch (ch) {
		case 'H':
			Hflag = 1;
			Lflag = 0;
			break;
		case 'L':
			Lflag = 1;
			Hflag = 0;
			break;
		case 'P':
			Hflag = Lflag = 0;
			break;
		case 'R':
			Rflag = 1;
			break;
		case 'f':
			fflag = 1;
			break;
		case 'h':
			/*
			 * In System V the -h option causes chown/chgrp to
			 * change the owner/group of the symbolic link.
			 * 4.4BSD's symbolic links didn't have owners/groups,
			 * so it was an undocumented noop.
			 * In NetBSD 1.3, lchown(2) is introduced.
			 */
			hflag = 1;
			break;
		case 'v':
			vflag = 1;
			break;
		case '?':
		default:
			usage();
			return EXIT_FAILURE;
		}
	argv += optind;
	argc -= optind;

	if (argc < 2) {
		usage();
		return EXIT_FAILURE;
	}

	fts_options = FTS_PHYSICAL;
	if (Rflag) {
		if (Hflag)
			fts_options |= FTS_COMFOLLOW;
		if (Lflag) {
			if (hflag) {
#ifdef FSU_CONSOLE
				warn("the -L and -h options "
				    "may not be specified together.");
				return EXIT_FAILURE;
#else
				errx(EXIT_FAILURE,
				    "the -L and -h options "
				    "may not be specified together.");
#endif
			}
			fts_options &= ~FTS_PHYSICAL;
			fts_options |= FTS_LOGICAL;
		}
	} else if (!hflag)
		fts_options |= FTS_COMFOLLOW;

	uid = (uid_t)-1;
	gid = (gid_t)-1;
	if (ischown) {
		if ((cp = strchr(*argv, ':')) != NULL) {
			*cp++ = '\0';
			a_gid(cp);
		}
#ifdef SUPPORT_DOT
		else if ((cp = strrchr(*argv, '.')) != NULL) {
			if (uid_from_user(*argv, &uid) == -1) {
				*cp++ = '\0';
				a_gid(cp);
			}
		}
#endif
		a_uid(*argv);
	} else
		a_gid(*argv);
#ifdef FSU_CONSOLE
	if (chown_exit != 0)
		return EXIT_FAILURE;
#endif
	if ((ftsp = fts_open(++argv, fts_options, NULL)) == NULL) {
#ifdef FSU_CONSOLE
		warn("fts_open");
		return EXIT_FAILURE;
#else
		err(EXIT_FAILURE, "fts_open");
#endif
	}

	for (rval = EXIT_SUCCESS; (p = fts_read(ftsp)) != NULL;) {
#ifdef FSU
		change_owner = ukfs_chown;
#else
		change_owner = chown;
#endif
		switch (p->fts_info) {
		case FTS_D:
			if (!Rflag)		/* Change it at FTS_DP. */
				fts_set(ftsp, p, FTS_SKIP);
			continue;
		case FTS_DNR:			/* Warn, chown, continue. */
			warnx("%s: %s", p->fts_path, strerror(p->fts_errno));
			rval = EXIT_FAILURE;
			break;
		case FTS_ERR:			/* Warn, continue. */
		case FTS_NS:
			warnx("%s: %s", p->fts_path, strerror(p->fts_errno));
			rval = EXIT_FAILURE;
			continue;
		case FTS_SL:			/* Ignore unless -h. */
			/*
			 * All symlinks we found while doing a physical
			 * walk end up here.
			 */
			if (!hflag)
				continue;
			/*
			 * Note that if we follow a symlink, fts_info is
			 * not FTS_SL but FTS_F or whatever.  And we should
			 * use lchown only for FTS_SL and should use chown
			 * for others.
			 */
#ifdef FSU
			change_owner = ukfs_lchown;
#else
			change_owner = lchown;
#endif
			break;
		case FTS_SLNONE:		/* Ignore. */
			/*
			 * The only symlinks that end up here are ones that
			 * don't point to anything.  Note that if we are
			 * doing a phisycal walk, we never reach here unless
			 * we asked to follow explicitly.
			 */
			continue;
		default:
			break;
		}

#ifdef FSU
		if ((*change_owner)(ukfs, p->fts_accpath, uid, gid) && !fflag) {
#else
		if ((*change_owner)(p->fts_accpath, uid, gid) && !fflag) {
#endif
			warn("%s", p->fts_path);
			rval = EXIT_FAILURE;
		} else {
			if (vflag)
				printf("%s\n", p->fts_path);
		}
	}
	error = errno;
	fts_close(ftsp);
	errno = error;
	if (errno) {
#ifdef FSU_CONSOLE
		warn("fts_read");
		return EXIT_FAILURE;
#else
		err(EXIT_FAILURE, "fts_read");
#endif
	}
#ifdef FSU_CONSOLE
	return rval;
#else
	exit(rval);
	/* NOTREACHED */
#endif
}

static void
a_gid(const char *s)
{
	struct group *gr;

	if (*s == '\0')			/* Argument was "uid[:.]". */
		return;
	gr = *s == '#' ? NULL : getgrnam(s);
	if (gr == NULL)
		gid = id(s, "group");
	else
		gid = gr->gr_gid;
	return;
}

static void
a_uid(const char *s)
{
	if (*s == '\0')			/* Argument was "[:.]gid". */
		return;
	if (*s == '#' || uid_from_user(s, &uid) == -1) {
		uid = id(s, "user");
	}
	return;
}

static id_t
id(const char *name, const char *type)
{
	id_t val;
	char *ep;

	errno = 0;
	if (*name == '#')
		name++;
	val = (id_t)strtoul(name, &ep, 10);
	if (errno) {
#ifdef FSU_CONSOLE
		warn("%s", name);
		chown_exit = 1;
#else
		err(EXIT_FAILURE, "%s", name);
#endif
	}
	if (*ep != '\0') {
#ifdef FSU_CONSOLE
		warn("%s: invalid %s name", name, type);
		chown_exit = 1;
#else
		errx(EXIT_FAILURE, "%s: invalid %s name", name, type);
#endif
	}
	return (val);
}

static void
usage(void)
{

	(void)fprintf(stderr,
#if defined(FSU) && !defined(FSU_CONSOLE)
     "usage: %s [MNT_OPTIONS] [fstype] fsdevice [-R [-H | -L | -P]] [-fhv] %s file ...\n",
#else
	    "usage: %s [-R [-H | -L | -P]] [-fhv] %s file ...\n",
#endif
	    myname, ischown ? "[owner][:group]" : "group");

#ifndef FSU_CONSOLE
	exit(EXIT_FAILURE);
#endif
}

#if defined(FSU) && !defined(FSU_CONSOLE)
void
unmount_ukfs(void)
{
	
	ukfs_release(ukfs, 0);
}
#endif
