/*	$OpenBSD: dma_sbus.c,v 1.12 2005/03/03 01:41:45 miod Exp $	*/
/*	$NetBSD: dma_sbus.c,v 1.5 2000/07/09 20:57:42 pk Exp $ */

/*-
 * Copyright (c) 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Paul Kranenburg.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the NetBSD
 *        Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1994 Peter Galbavy.  All rights reserved.
 * Copyright (c) 2005 Marius Strobl <marius@FreeBSD.org>. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD: stable/6/sys/sparc64/sbus/dma_sbus.c 182211 2008-08-26 18:18:30Z marius $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/bus.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/resource.h>
#include <sys/rman.h>

#include <dev/ofw/ofw_bus.h>
#include <dev/ofw/openfirm.h>

#include <machine/bus.h>
#include <machine/bus_common.h>
#include <machine/resource.h>

#include <sparc64/sbus/lsi64854reg.h>
#include <sparc64/sbus/lsi64854var.h>
#include <sparc64/sbus/ofw_sbus.h>
#include <sparc64/sbus/sbusreg.h>
#include <sparc64/sbus/sbusvar.h>

struct dma_devinfo {
	char			*ddi_compat;	/* PROM compatible */
	char			*ddi_model;	/* PROM model */
	char			*ddi_name;	/* PROM name */
	phandle_t		ddi_node;	/* PROM node */
	char			*ddi_type;	/* PROM device_type */

	struct resource_list	ddi_rl;
};

struct dma_softc {
	struct lsi64854_softc	sc_lsi64854;	/* base device */
	int			sc_ign;
	int			sc_slot;
};

static devclass_t dma_devclass;

static device_probe_t dma_probe;
static device_attach_t dma_attach;
static bus_print_child_t dma_print_child;
static bus_probe_nomatch_t dma_probe_nomatch;
static bus_get_resource_list_t dma_get_resource_list;
static ofw_bus_get_compat_t dma_get_compat;
static ofw_bus_get_model_t dma_get_model;
static ofw_bus_get_name_t dma_get_name;
static ofw_bus_get_node_t dma_get_node;
static ofw_bus_get_type_t dma_get_type;

static struct dma_devinfo *dma_setup_dinfo(device_t, phandle_t, char *);
static void dma_destroy_dinfo(struct dma_devinfo *);

static device_method_t dma_methods[] = {
        /* Device interface */
	DEVMETHOD(device_probe,		dma_probe),
	DEVMETHOD(device_attach,	dma_attach),
	DEVMETHOD(device_shutdown,	bus_generic_shutdown),
	DEVMETHOD(device_suspend,	bus_generic_suspend),
	DEVMETHOD(device_resume,	bus_generic_resume),

	/* Bus interface */
	DEVMETHOD(bus_print_child,	dma_print_child),
	DEVMETHOD(bus_probe_nomatch,	dma_probe_nomatch),
	DEVMETHOD(bus_setup_intr,	bus_generic_setup_intr),
	DEVMETHOD(bus_teardown_intr,	bus_generic_teardown_intr),
	DEVMETHOD(bus_alloc_resource,	bus_generic_rl_alloc_resource),
	DEVMETHOD(bus_release_resource, bus_generic_rl_release_resource),
	DEVMETHOD(bus_activate_resource, bus_generic_activate_resource),
	DEVMETHOD(bus_deactivate_resource, bus_generic_deactivate_resource),
	DEVMETHOD(bus_get_resource_list, dma_get_resource_list),
	DEVMETHOD(bus_get_resource,	bus_generic_rl_get_resource),

	/* ofw_bus interface */
	DEVMETHOD(ofw_bus_get_compat,	dma_get_compat),
	DEVMETHOD(ofw_bus_get_model,	dma_get_model),
	DEVMETHOD(ofw_bus_get_name,	dma_get_name),
	DEVMETHOD(ofw_bus_get_node,	dma_get_node),
	DEVMETHOD(ofw_bus_get_type,	dma_get_type),

	{ 0, 0 }
};

static driver_t dma_driver = {
	"dma",
	dma_methods,
	sizeof(struct dma_softc),
};

DRIVER_MODULE(dma, sbus, dma_driver, dma_devclass, 0, 0);
MODULE_DEPEND(dma, sbus, 1, 1, 1);
MODULE_VERSION(dma, 1);

static int
dma_probe(device_t dev)
{
	const char *name;

	name = ofw_bus_get_name(dev);
	if (strcmp(name, "espdma") == 0 || strcmp(name, "dma") == 0 ||
	    strcmp(name, "ledma") == 0) {
                device_set_desc_copy(dev, name);
                return (0);
	}
        return (ENXIO);
}

static int
dma_attach(device_t dev)
{
	struct dma_softc *dsc;
	struct lsi64854_softc *lsc;
	struct dma_devinfo *ddi;
	device_t cdev;
	const char *name;
	char *cabletype, *cname;
	uint32_t csr;
	phandle_t child, node;
	int error, burst, children;

	dsc = device_get_softc(dev);
	lsc = &dsc->sc_lsi64854;

	name = ofw_bus_get_name(dev);
	node = ofw_bus_get_node(dev);
	dsc->sc_ign = sbus_get_ign(dev);
	dsc->sc_slot = sbus_get_slot(dev);

	lsc->sc_rid = 0;
	lsc->sc_res = bus_alloc_resource_any(dev, SYS_RES_MEMORY, &lsc->sc_rid,
	    RF_ACTIVE);
	if (lsc->sc_res == NULL) {
		device_printf(dev, "cannot allocate resources\n");
		return (ENXIO);
	}
	lsc->sc_regt = rman_get_bustag(lsc->sc_res);
	lsc->sc_regh = rman_get_bushandle(lsc->sc_res);

	if (strcmp(name, "espdma") == 0 || strcmp(name, "dma") == 0)
		lsc->sc_channel = L64854_CHANNEL_SCSI;
	else if (strcmp(name, "ledma") == 0) {
		/*
		 * Check to see which cable type is currently active and
		 * set the appropriate bit in the ledma csr so that it
		 * gets used. If we didn't netboot, the PROM won't have
		 * the "cable-selection" property; default to TP and then
		 * the user can change it via a "media" option to ifconfig.
		 */
		csr = L64854_GCSR(lsc);
		if ((OF_getprop_alloc(node, "cable-selection", 1,
		    (void **)&cabletype)) == -1) {
			/* assume TP if nothing there */
			csr |= E_TP_AUI;
		} else {
			if (strcmp(cabletype, "aui") == 0)
				csr &= ~E_TP_AUI;
			else
				csr |= E_TP_AUI;
			free(cabletype, M_OFWPROP);
		}
		L64854_SCSR(lsc, csr);
		DELAY(20000);	/* manual says we need a 20ms delay */
		lsc->sc_channel = L64854_CHANNEL_ENET;
	} else {
		device_printf(dev, "unsupported DMA channel\n");
		error = ENXIO;
		goto fail_lres;
	}

	error = bus_dma_tag_create(
	    NULL,			/* parent */
	    PAGE_SIZE, 0,		/* alignment, boundary */
	    BUS_SPACE_MAXADDR,		/* lowaddr */
	    BUS_SPACE_MAXADDR,		/* highaddr */
	    NULL, NULL,			/* filter, filterarg */
	    BUS_SPACE_MAXSIZE_32BIT,	/* maxsize */
	    0,				/* nsegments */
	    BUS_SPACE_MAXSIZE_32BIT,	/* maxsegsize */
	    0,				/* flags */
	    NULL, NULL,			/* no locking */
	    &lsc->sc_parent_dmat);
	if (error != 0) {
		device_printf(dev, "cannot allocate parent DMA tag\n");
		goto fail_lres;
	}

	burst = sbus_get_burstsz(dev);
	lsc->sc_burst = (burst & SBUS_BURST_32) ? 32 :
	    (burst & SBUS_BURST_16) ? 16 : 0;
	lsc->sc_dev = dev;

	error = lsi64854_attach(lsc);
	if (error != 0) {
		device_printf(dev, "lsi64854_attach failed\n");
		goto fail_lpdma;
	}

	/* Attach children. */
	children = 0;
	for (child = OF_child(node); child != 0; child = OF_peer(child)) {
		if ((OF_getprop_alloc(child, "name", 1, (void **)&cname)) == -1)
			continue;
		if ((ddi = dma_setup_dinfo(dev, child, cname)) == NULL) {
			device_printf(dev, "<%s>: incomplete\n", cname);
			free(cname, M_OFWPROP);
			continue;
		}
		if (children != 0) {
			device_printf(dev, "<%s>: only one child per DMA "
			    "channel supported\n", cname);
			dma_destroy_dinfo(ddi);
			free(cname, M_OFWPROP);
			continue;
		}
		if ((cdev = device_add_child(dev, NULL, -1)) == NULL) {
			device_printf(dev, "<%s>: device_add_child failed\n",
			    cname);
			dma_destroy_dinfo(ddi);
			free(cname, M_OFWPROP);
			continue;
		}
		device_set_ivars(cdev, ddi);
		children++;
	}
	error = bus_generic_attach(dev);
	if (error != 0) {
		device_printf(dev, "bus_generic_attach failed\n");
		goto fail_lsi;
	}

	return (0);

 fail_lsi:
	lsi64854_detach(lsc);
 fail_lpdma:
	bus_dma_tag_destroy(lsc->sc_parent_dmat);
 fail_lres:
	bus_release_resource(dev, SYS_RES_MEMORY, lsc->sc_rid, lsc->sc_res);
	return (error);
}

static struct dma_devinfo *
dma_setup_dinfo(device_t dev, phandle_t node, char *name)
{
	struct dma_softc *dsc;
	struct dma_devinfo *ddi;
	struct sbus_regs *reg;
	uint32_t base, iv, *intr;
	int i, nreg, nintr, slot, rslot;

	dsc = device_get_softc(dev);

	ddi = malloc(sizeof(*ddi), M_DEVBUF, M_WAITOK | M_ZERO);
	if (ddi == NULL)
		return (NULL);
	resource_list_init(&ddi->ddi_rl);
	ddi->ddi_name = name;
	ddi->ddi_node = node;
	OF_getprop_alloc(node, "compatible", 1, (void **)&ddi->ddi_compat);
	OF_getprop_alloc(node, "device_type", 1, (void **)&ddi->ddi_type);
	OF_getprop_alloc(node, "model", 1, (void **)&ddi->ddi_model);
	slot = -1;
	nreg = OF_getprop_alloc(node, "reg", sizeof(*reg), (void **)&reg);
	if (nreg == -1) {
		dma_destroy_dinfo(ddi);
		return (NULL);
	}
	for (i = 0; i < nreg; i++) {
		base = reg[i].sbr_offset;
		if (SBUS_ABS(base)) {
			rslot = SBUS_ABS_TO_SLOT(base);
			base = SBUS_ABS_TO_OFFSET(base);
		} else
			rslot = reg[i].sbr_slot;
		if (slot != -1 && slot != rslot) {
			device_printf(dev, "<%s>: multiple slots\n", name);
			free(reg, M_OFWPROP);
			dma_destroy_dinfo(ddi);
			return (NULL);
		}
		slot = rslot;

		resource_list_add(&ddi->ddi_rl, SYS_RES_MEMORY, i, base,
		    base + reg[i].sbr_size, reg[i].sbr_size);
	}
	free(reg, M_OFWPROP);
	if (slot != dsc->sc_slot) {
		device_printf(dev, "<%s>: parent and child slot do not match\n",
		    name);
		dma_destroy_dinfo(ddi);
		return (NULL);
	}

	/*
	 * The `interrupts' property contains the SBus interrupt level.
	 */
	nintr = OF_getprop_alloc(node, "interrupts", sizeof(*intr),
	    (void **)&intr);
	if (nintr != -1) {
		for (i = 0; i < nintr; i++) {
			iv = intr[i];
			/*
			 * SBus card devices need the slot number encoded into
			 * the vector as this is generally not done.
			 */
			if ((iv & INTMAP_OBIO_MASK) == 0)
				iv |= slot << 3;
			/* Set the IGN as appropriate. */
			iv |= dsc->sc_ign << INTMAP_IGN_SHIFT;
			resource_list_add(&ddi->ddi_rl, SYS_RES_IRQ, i,
			    iv, iv, 1);
		}
		free(intr, M_OFWPROP);
	}
	return (ddi);
}

static void
dma_destroy_dinfo(struct dma_devinfo *dinfo)
{

	resource_list_free(&dinfo->ddi_rl);
	if (dinfo->ddi_compat != NULL)
		free(dinfo->ddi_compat, M_OFWPROP);
	if (dinfo->ddi_model != NULL)
		free(dinfo->ddi_model, M_OFWPROP);
	if (dinfo->ddi_type != NULL)
		free(dinfo->ddi_type, M_OFWPROP);
	free(dinfo, M_DEVBUF);
}

static int
dma_print_child(device_t dev, device_t child)
{
	struct dma_devinfo *ddi;
	struct resource_list *rl;
	int rv;

	ddi = device_get_ivars(child);
	rl = &ddi->ddi_rl;
	rv = bus_print_child_header(dev, child);
	rv += resource_list_print_type(rl, "mem", SYS_RES_MEMORY, "%#lx");
	rv += resource_list_print_type(rl, "irq", SYS_RES_IRQ, "%ld");
	rv += bus_print_child_footer(dev, child);
	return (rv);
}

static void
dma_probe_nomatch(device_t dev, device_t child)
{
	struct dma_devinfo *ddi;
	struct resource_list *rl;

	ddi = device_get_ivars(child);
	rl = &ddi->ddi_rl;
	device_printf(dev, "<%s>", ddi->ddi_name);
	resource_list_print_type(rl, "mem", SYS_RES_MEMORY, "%#lx");
        resource_list_print_type(rl, "irq", SYS_RES_IRQ, "%ld");
        printf(" type %s (no driver attached)\n",
            ddi->ddi_type != NULL ? ddi->ddi_type : "unknown");
}

static struct resource_list *
dma_get_resource_list(device_t dev, device_t child)
{
	struct dma_devinfo *ddi;

	ddi = device_get_ivars(child);
	return (&ddi->ddi_rl);
}

static const char *
dma_get_compat(device_t bus, device_t dev)
{
	struct dma_devinfo *dinfo;

	dinfo = device_get_ivars(dev);
	return (dinfo->ddi_compat);
}

static const char *
dma_get_model(device_t bus, device_t dev)
{
	struct dma_devinfo *dinfo;

	dinfo = device_get_ivars(dev);
	return (dinfo->ddi_model);
}

static const char *
dma_get_name(device_t bus, device_t dev)
{
	struct dma_devinfo *dinfo;

	dinfo = device_get_ivars(dev);
	return (dinfo->ddi_name);
}

static phandle_t
dma_get_node(device_t bus, device_t dev)
{
	struct dma_devinfo *dinfo;

	dinfo = device_get_ivars(dev);
	return (dinfo->ddi_node);
}

static const char *
dma_get_type(device_t bus, device_t dev)
{
	struct dma_devinfo *dinfo;

	dinfo = device_get_ivars(dev);
	return (dinfo->ddi_type);
}
