/*
 * Copyright (C) 2000, 2001 by Dmitry Morozhnikov, 
 * under the terms of the GNU General Public License, version 2.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <time.h>
#include <sysexits.h>
#include <errno.h>
#include "datfile2.h"
#include "datview.h"

int calcmaxw(dat_field_t *f, int n)
{
	int i, w;
	w = 0;
	for(i = 0 ; i < n ; i++)
	{
		if(f[i].length > w)
		  w = f[i].length + 1024;
	};
	return w;
};

char *debcd(unsigned char *ptr, dat_field_t *f, datview_opts_t *opts)
{
	unsigned char buf[256];
	unsigned char buf2[512 + 1];
	static unsigned char buf3[512 + 2];
	unsigned char t;
	int i, sign;
	unsigned char *p;
	memcpy(buf, ptr, f->length);
	memset(buf2, '\0', 512 + 1);
	memset(buf3, '\0', 512 + 2);
	for(i = 0 ; i < f->length ; i++)
	{
		t = buf[i];
		buf2[i * 2] = (t >> 4) + 0x30;
		buf2[i * 2 + 1] = (t & 0x0f) + 0x30;
	};
	sign = 0;
	if((f->length * 2 - f->decsig - f->decdec) > 0 && opts->ignsign == 0)
	{
		if(buf2[0] != 0x30)
		{
			sign = 1;
		};
	};
	p = buf2 + (f->length * 2 - f->decsig - f->decdec);
	i = 0;
	if(opts->nostripzdec != 1)
	{
		while(*p == '0' && i < f->decsig - 1)
		{
			p++;
			i++;
		};
	};
	if(sign == 1)
	{
		buf3[0] = '-';
	};
	memcpy(buf3 + sign, p, f->decsig);
	if(f->decdec != 0 || opts->showdot == 1)
	{
		buf3[f->decsig + sign - i] = '.';
		p = buf2 + (f->length * 2 - f->decdec);
		memcpy(buf3 + f->decsig + 1 + sign - i, p, f->decdec);
	};
	return buf3;
};

unsigned char *getfld(dat_field_t *f, void *rec,
					  datview_opts_t *opts, dat_field_int_t *fint, int fnum)
{
	static unsigned char buf[64 * 1024 + 1];
	if(fint[fnum].isdate == 1)
	{
		long d;
		int rc;
		struct tm *tm;
		d = absdate(*((long *)(rec + f->foffset + sizeof(dat_rec_t))),
					opts->forcedt);
		if(d == -1)
		{
			exit(EX_DATAERR);
		};
		if(opts->unixdt == 1)
		{
			sprintf(buf, "%lu", (unsigned long) d);
		}
		else
		{
			tm = localtime(&d);
			assert(tm);
			switch(opts->datefmt)
			{
			  case 'i':
				rc = strftime(buf, 1024, "%Y-%m-%d", tm);
				break;
			  case 's':
				rc = strftime(buf, 1024, "%m/%d/%Y", tm);
				break;
			  case 'l':
			  default:
				rc = strftime(buf, 1024, "%x", tm);
				break;
			};
			if(rc == 0)
			{
				fprintf(stderr, "Can`t strftime: %s\n", strerror(errno));
				exit(EX_DATAERR);
			};
			if(opts->quotedt == 1)
			{
				static unsigned char buf2[64 * 1024 + 1];
				sprintf(buf2, "%c%s%c",
						opts->quotechar, buf, opts->quotechar);
				strcpy(buf, buf2);
			};
		};
		return buf;
	};
	if(fint[fnum].istime == 1)
	{
		long d;
		int rc;
		struct tm *tm;
		d = abstime(*((long *)(rec + f->foffset + sizeof(dat_rec_t))),
					opts->forcedt);
		if(d == -1)
		{
			exit(EX_DATAERR);
		};
		if(opts->unixdt == 1)
		{
			sprintf(buf, "%lu", (unsigned long) d);
		}
		else
		{
			tm = localtime(&d);
			assert(tm);
			rc = strftime(buf, 1024, "%X", tm);
			if(rc == 0)
			{
				fprintf(stderr, "Can`t strftime: %s\n", strerror(errno));
				exit(EX_DATAERR);
			};
			if(opts->quotedt == 1)
			{
				static unsigned char buf2[64 * 1024 + 1];
				sprintf(buf2, "%c%s%c",
						opts->quotechar, buf, opts->quotechar);
				strcpy(buf, buf2);
			};
		};
		return buf;
	};
	switch(f->fldtype)
	{
		int b;
	  case DAT_FLDTYPE_LONG:
		sprintf(buf, "%li", *((long *)(rec + f->foffset + sizeof(dat_rec_t))));
		break;
	  case DAT_FLDTYPE_REAL:
		sprintf(buf, "%e", *((double *)(rec + f->foffset + sizeof(dat_rec_t))));
		break;
	  case DAT_FLDTYPE_STRING:
	  case DAT_FLDTYPE_STRINGP:
		memcpy(buf, (unsigned char *)(rec + f->foffset + sizeof(dat_rec_t)), f->length);
		buf[f->length] = '\0';
		break;
	  case DAT_FLDTYPE_SHORT:
		sprintf(buf, "%hi", *((short *)(rec + f->foffset + sizeof(dat_rec_t))));
		break;
	  case DAT_FLDTYPE_BYTE:
		b = *((char *)(rec + f->foffset + sizeof(dat_rec_t)));
		sprintf(buf, "%hi", b);
		break;
	  case DAT_FLDTYPE_GROUP:
		fprintf(stderr, "Can`t handle GROUPs in GROUP");
		exit(EX_SOFTWARE);
		break;
	  case DAT_FLDTYPE_DECIMAL:
		strcpy(buf, debcd((char *)rec + f->foffset + sizeof(dat_rec_t),
						  f, opts));
		break;
	  default:
		fprintf(stderr, "Unknown field type: 0x%x\n", f->fldtype);
		exit(EX_DATAERR);
		break;
	};
	return buf;
};

int browse(dat_header_t *h, dat_field_t *f, datview_opts_t *opts, 
		   int fd, int memofd, dat_field_int_t *fint)
{
	int i, maxw, rc, j;
	unsigned char *buf, *rec;
	dat_rec_t *dat_rec;
	int width;
	dat_field_t fldd, *fld;
	unsigned char *qstr;
	unsigned char *memo;
	
	fld = &fldd;
	maxw = calcmaxw(f, h->numflds);
	buf = (unsigned char *) malloc(maxw);
	assert(buf);
	rec = (unsigned char *) malloc(h->reclen);
	assert(rec);
	rc = lseek(fd, h->offset, SEEK_SET);
	if(rc == -1)
	{
		fprintf(stderr, "Can`t lseek to data section: %s\n", strerror(errno));
		exit(EX_DATAERR);
	};
	for(i = 0 ; i < h->numrecs ; i++)
	{
		rc = read(fd, rec, h->reclen);
		if(rc != h->reclen)
		{
			fprintf(stderr, "Can`t read record: %s\n", strerror(errno));
			exit(EX_DATAERR);
		};
		dat_rec = (dat_rec_t *) rec;
		if(opts->deleted == 'n' && (dat_rec->rhd & DAT_RHD_DELETED))
		{
			continue;
		};
		if(opts->deleted == 'd' && (dat_rec->rhd ^ DAT_RHD_DELETED))
		{
			continue;
		};
		if(opts->locked == 'n' && (dat_rec->rhd & DAT_RHD_HELD))
		{
			continue;
		};
		if(opts->locked == 'l' && (dat_rec->rhd ^ DAT_RHD_HELD))
		{
			continue;
		};
		width = 0;
		for(j = 0 ; j < h->numflds ; j++)
		{
			memcpy(fld, &f[j], sizeof(dat_field_t));
			if(opts->group == 'n' && fld->fldtype == DAT_FLDTYPE_GROUP)
			{
				continue;
			};
			if((opts->group == 'u' || opts->group == 'b')
			   && fld->fldtype == DAT_FLDTYPE_GROUP)
			{
				int jj = j;
				width = fld->length;
				buf[0] = '\0';
				while(width > 0)
				{
					j++;
					memcpy(fld, &f[j], sizeof(dat_field_t));
					strcat(buf, getfld(fld, rec, opts, fint, j));
					width -= fld->length;
				};
				if(opts->group == 'b')
				{
					j = jj;
				};
				if(opts->nostripsp != 1)
				{
					stripspaces(buf);
				};
				if(opts->quotestr == 1)
				{
					qstr = quotestrn(buf, strlen(buf), opts->quotechar);
				}
				else
				{
					qstr = buf;
				};
				if(j == 0)
				{
					printf("%s", qstr);
				}
				else
				{
					putchar(opts->separator);
					printf("%s", qstr);
				};
				continue;
			};
			strcpy(buf, getfld(fld, rec, opts, fint, j));
			if(opts->nostripsp != 1)
			{
				stripspaces(buf);
			};
			if(opts->quotestr == 1 &&
			   (fld->fldtype == DAT_FLDTYPE_STRING ||
				fld->fldtype == DAT_FLDTYPE_STRINGP))
			{
				if(opts->sqlquote == 1)
				{
					qstr = sqlquotestrn(buf, strlen(buf));
				}
				else
				{
					qstr = quotestrn(buf, strlen(buf), opts->quotechar);
				};
			}
			else
			{
				qstr = buf;
			};
			if(j == 0)
			{
				printf("%s", qstr);
			}
			else
			{
				putchar(opts->separator);
				printf("%s", qstr);
			};
		};
		if(opts->withmemo && memofd != -1)
		{
			int needfree = 0;
			putchar(opts->separator);
			if(dat_rec->rptr == 0 || (dat_rec->rhd & DAT_RHD_DELETED))
			{
				memo = "";
			}
			else
			{
				memo = memo_read(h, memofd, dat_rec->rptr, opts);
				if(opts->nostripsp != 1)
				{
					stripspaces(memo);
				};
				needfree = 1;
			};
			if(opts->quotestr == 1)
			{
				if(opts->sqlquote == 1)
				{
					qstr = sqlquotestrn(memo, strlen(memo));
				}
				else
				{
					qstr = quotestrn(memo, strlen(memo), opts->quotechar);
				};
			}
			else
			{
				qstr = memo;
			};
			printf("%s", qstr);
			if(needfree == 1)
			{
				free(memo);
			};
		};
		printf("\n");
	};
	return 0;
};
