/*
	seek.cc		Char Seek Functions
	Copyright (c) 1996-8,2000 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "seek.h"
#include "cxxlib.h"
#include "strmisc.h"

/* 
	Case-sensitive search with escapes and quotes skipped
	Returns:	position (always >= 0) if found
			< 0 if not found
*/
int	SeekCharIgnoreEscape(char c, const char *buffer, int from, int length)
{
	while (from < length) {
		if (buffer[from] == '\\') {
					// `\' Escape found
		
			from++;

					// Unexpected end of buffer
			if (from >= length)
				return static_cast<unsigned char>(c)-256;

					// Seek for a `\' and found `\'
			if (buffer[from] == '\\' && c == '\\')
				return from;		// Return position of
							// the last `\'
				
			from++;		// Skip it !
		}
		else if (buffer[from] == c) {
			return from;	// Return index where c is found
		}
		else {
					// Check for openning quotes
					
			switch(buffer[from]) {
				case '\"':
						// Get position of closing quote
				
					from = SeekCharIgnoreEscape('\"', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
				case '\'':
						// Get position of closing quote
				
					from = SeekCharIgnoreEscape('\'', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
				case '`':
						// Get position of closing quote
				
					from = SeekCharIgnoreEscape('`', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
			}
			
			from++;		// Skip ordinary char and
					// closing quotes
		}
	}
	return static_cast<unsigned char>(c)-256;
					// Not found, search char. is encoded 
					// and returned as a negative value
}


/* 
	Case-sensitive search
	nest = 0 no escapes & quotes interpretation
	       1 with quotes skipped (quotes can only nested once)
	       2 with escapes and quotes skipped (quotes can only nested once)
	Returns:	position (always >= 0) if found
			< 0 if not found
*/
int	SeekChar(char c, const char *buffer, int from, int length, int nest)
{
	if (nest == 2)
		return SeekCharIgnoreEscape(c, buffer, from, length);
		
	while (from < length) {
		if (buffer[from] == c) {
			return from;	// Return index where c is found
		}
		else if (nest) {
					// Check for openning quotes
					
			switch(buffer[from]) {
				case '\"':
						// Get position of closing quote
				
					from = SeekChar('\"', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
				case '\'':
						// Get position of closing quote
				
					from = SeekChar('\'', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
				case '`':
						// Get position of closing quote
				
					from = SeekChar('`', buffer, from+1,
							length);
							
						// Quote not found
					if (from < 0)
						return from;
					break;
			}
			from++;		// Skip ordinary char and
					// closing quotes
		}
		else
			from++;
	}
	return static_cast<unsigned char>(c)-256;	
					// Not found, search char. is encoded 
					// and returned as a negative value
}

/*
	SeekString - Find str in buffer[from]...buffer[length-1]
	str cannot contain any quote or escape
	Returns:	position (always >= 0) if found
			-1 if not found
*/

int	SeekString(const char *str, const char *buffer, int from, int length, int nest)
{
	int	ret;
	int	len = strlen(str);
	for ( ; ; ) {
					// Seek for the first char of the
					// string

		ret = SeekChar(str[0], buffer, from, length, nest);
		if (ret >= 0 && ret+len <= length) {
			if (strncmp(buffer+ret, str, len) == 0)	// Found 
				return ret;
			from = ret+1;	// Seek from next byte 
		}
		else {
			return -1;	// Not found
		}
	}
}

/*
	SeekCaseString - Find str in buffer[from]...buffer[length-1] 
			(case-insensitive)
	str cannot contain any quote or escape
	Returns:	position (always >= 0) if found
			-1 if not found
*/

int	SeekCaseString(const char *str, const char *buffer, int from, int length, int nest)
{
	int	ret;
	int	len = strlen(str);
	for ( ; ; ) {
	
				// Chars does not have upper/lower case
				// representations
		if (toupper(str[0]) == tolower(str[0])) {
			ret = SeekChar(str[0], buffer, from, length, nest);
		}
		else {
				// Get the first occurance of upper/lower
				// case chars
				
			ret = Min(SeekChar(toupper(str[0]), buffer, from, length, nest),
				SeekChar(tolower(str[0]), buffer, from, length, nest));
		}
		if (ret >= 0 && ret+len <= length) {
			if (CompareStringCase(buffer+ret, str, len) == 0)	// Found
				return ret;
			from = ret+1;	// Seek from next byte
		}
		else {
			return -1;	// Error
		}
	}
}

/*
	SeekTokenEnd - Return the first position from buffer[from] that  
			separates/ends tokens inside HTML tags.
			<xxx aaa=bbb ccc=ddd>
	Returns:	position (always >= 0) if found
			-1 if not found
*/

int	SeekTokenEnd(const char *buffer, int from, int length)
{
	while (from < length) {
		if (buffer[from] == ' ' || buffer[from] == '=' ||
		    buffer[from] == '>' || buffer[from] == '\n' ||
		    buffer[from] == '\r' || buffer[from] == '\t')
			return from;
		from++;
	}
	return -1;
}

/*
	SeekConfigTokenEnd - Return the first position from buffer[from] that  
			     separates/ends tokens inside configuration file.
	Returns:	position (always >= 0)
*/

int	SeekConfigTokenEnd(const char *buffer, int from, int length)
{
	while (from < length) {
		if (buffer[from] == ' ' || buffer[from] == ',' ||
		    buffer[from] == '\n' ||
		    buffer[from] == '\r' || buffer[from] == '\t')
			return from;
		from++;
	}
	return from;
}

/*
	SeekNonSpace - Return the first position from buffer[from] that is 
			not spaces.
	Returns:	position (always >= 0) if found
			-1 if not found
*/

int	SeekNonSpace(const char *buffer, int from, int length)
{
	while (from < length) {
		if (buffer[from] != ' ' && buffer[from] != '\t' &&
		    buffer[from] != '\n' && buffer[from] != '\r')
			return from;
		from++;
	}
	return -1;
}

/*
	SeekPastEqualSign - Skip spaces, `=' and then spaces
	Returns:	position (always >= 0) if found
			-1 if not found
*/

int	SeekPastEqualSign(const char *buffer, int from, int length)
{
	from = SeekNonSpace(buffer, from, length);	// Skip spaces
	if (from < 0)
		return from;
		
	if (buffer[from] != '=')			// Check for a `='
		return -1;
	from++;						// Next char
	
	return SeekNonSpace(buffer, from, length);	// Skip spaces
}

/*
	IsEmpty - Check if buffer contains solely spaces and tabs
	Returns:	1 buffer empty
			0 buffer not empty
*/

int	IsEmpty(const char *buffer, int from, int length)
{
	for (int i = from; i < length; i++) {
		if (buffer[i] != ' ' && buffer[i] != '\t')
			return 0;
	}
	return 1;
}
