/*
	gentree.cc	Generate Directory Tree For Display
	Copyright (c) 2000 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "gentree.h"

#include "confobj.h"
#include "buffer.h"
#include "miscobj.h"
#include "khwin.h"
#include "dirtree.h"
#include "cstrlib.h"

//#define DUMP_HTML	
#ifdef DUMP_HTML
# include <fcntl.h>
# include <sys/stat.h>
# include <sys/types.h>
#endif

string	saveCwd;

extern string	dirFile;

string	my_strftime(const char *format, const struct tm *tm)
{
	Buffer buf(1000);
	strftime(buf.GetPtr(), buf.GetAllocSize(), format, tm);
	return string(buf.GetPtr());
}

string	ExpandEscape(const string &src)	// Expand &, <, >, " to HTML escapes
{
	string	dest;
	size_t	len = src.size();
	for (size_t i = 0; i < len; i++) {
		switch(src[i]) {
			case '&':
				dest += "&amp;";
				break;
			case '\"':
				dest += "&quot;";
				break;
			case '<':
				dest += "&lt;";
				break;
			case '>':
				dest += "&gt;";
				break;
			default:
				dest += src[i];
		}
	}
	return dest;
}

/*************************************************************************
	Generate commonly used strings
*************************************************************************/

string	html_unreadable;
string	html_skipped;
string	html_not_in_file;
string	html_not_found;

size_t	html_unreadable_width;
size_t	html_skipped_width;
size_t	html_not_in_file_width;
size_t	html_not_found_width;

void	GenerateHTMLStrings()
{
	html_unreadable = _(" [unreadable]");
	html_skipped = _(" [skipped]");
	html_not_in_file = _(" [*]");
	html_not_found = _(" [not found]");

	html_unreadable_width = GetStringWidth(html_unreadable);
	html_skipped_width = GetStringWidth(html_skipped);
	html_not_in_file_width = GetStringWidth(html_not_in_file);
	html_not_found_width = GetStringWidth(html_not_found);
}

/*************************************************************************
	Generate HTML-like codes from the directory tree
*************************************************************************/

vector<size_t>	barStack;	// Positions of vertical bars
size_t	htmlRow, htmlCol;
size_t	curCol;			// curRow not needed

void	NewLineUpdateRowCol()
{
	if (curCol > htmlCol)
		htmlCol = curCol;
	htmlRow++;
	curCol = 0;
}

void	GenerateSymLinkHTML(Buffer &buffer, sptr<DirectoryEntry> &d,
			    const string &new_dir)
{
	bool	tagBegin = false;

	if (d->isLinkDestSkip == 0 
	    && d->isLinkDestUnreadable != 1) {

						// Write tags
		buffer.Puts("<A NAME=\"");
		buffer.Puts(QuoteURLChars(new_dir));
		buffer.Puts("\"></A><A HREF=\"#");
		buffer.Puts(QuoteURLChars(d->GetLinkDestStr()));
		buffer.Puts("\"><I>");

		tagBegin = true;
	}

			    			// Write dir. name
	buffer.Puts(ExpandEscape(d->GetNameStr()));
	curCol += GetStringWidth(d->GetNameStr());
			
			    			// Write graphical " -> "
	buffer.Puts(" &boxh;&rarr; ");
	curCol += 4;

			    			// Write linked dir.
	buffer.Puts(ExpandEscape(d->GetLinkStr()));
	curCol += GetStringWidth(d->GetLinkStr());

	switch(d->isLinkDestSkip) {
		case 0:
			if(d->isLinkDestUnreadable) {
				buffer.Puts(html_unreadable);
				curCol += html_unreadable_width;
			}
			break;
		case 1:
			buffer.Puts(html_skipped);
			curCol += html_skipped_width;
			break;
		case 2:
			buffer.Puts(html_not_in_file);
			curCol += html_not_in_file_width;
			break;
		case 3:
			buffer.Puts(html_not_found);
			curCol += html_not_found_width;
			break;
	}
			
	if (tagBegin) {
			    			// End <A HREF=...> tag
		buffer.Puts("</I></A>");
	}
}

void	GenerateDirHTML(Buffer &buffer, sptr<DirectoryEntry> &d,
			    const string &new_dir)
{
	if (d->isUnreadable != 1) {		// Exec permission
						// Write tags
		buffer.Puts("<A NAME=\"");
		buffer.Puts(QuoteURLChars(new_dir));
		buffer.Puts("\"></A><A CALL=\"");
		buffer.Puts(QuoteURLChars(new_dir));
		buffer.Puts("\">");
	}
			    		
			    			// Write dir. name
	buffer.Puts(ExpandEscape(d->GetNameStr()));
	curCol += GetStringWidth(d->GetNameStr());
			    		
	if (d->isUnreadable != 1) {		// Exec permission
			    			// Dir. can be accessed
			    				
			    			// End <A NAME=...> tag
		buffer.Puts("</A>");
	}				    		
}

void	RecursiveGenerateHTML(Buffer &buffer, sptr<DirectoryEntry> &d, 
			      size_t pos, const string &current_dir)
{
	if (d->isSkip) {
		if (d->isSkip == 2) {
			buffer.Puts(html_not_in_file);
			curCol += html_not_in_file_width;
			pos += html_not_in_file_width;
		}
		else {
			buffer.Puts(html_skipped);
			curCol += html_skipped_width;
			pos += html_skipped_width;
		}
	}
	if (d->isUnreadable) {
		buffer.Puts(html_unreadable);
		curCol += html_unreadable_width;
		pos += html_unreadable_width;
	}

	size_t	numSubDir = d->subDir.size();
	if (numSubDir == 0) {			// No more subdir
		return;				// Exit
	}

	size_t	curSubDir = 0;
	for (sptr_list<DirectoryEntry>::iterator iter = d->subDir.begin();
	     iter != d->subDir.end(); ++iter, ++curSubDir) {

		if (curSubDir == 0) {
			    			// First subdir entry
			    			// Draw a bar connected to 
			    			// parent dir
			buffer.Puts("&boxh;");
			curCol++;
			
			if (numSubDir == 1) {
			    			// Only one subdir present
			    			// Draw a bar
			    	buffer.Puts("&boxh;");
			    	curCol++;
			}
			else {
			    	buffer.Puts("&boxhd;");
			    	curCol++;
			    	
				barStack.push_back(pos+1);
			}
			buffer.Puts("&boxh;");
			curCol++;
		}
		else if (curSubDir == numSubDir-1) {
			    			// Last subdir entry
			    			
			buffer.Puts(" &boxur;&boxh;");
			curCol += 3;
			
			barStack.pop_back();
		}
		else {				// Middle entries
			buffer.Puts(" &boxvr;&boxh;");
			curCol += 3;
		}
			    
		string new_dir = current_dir;
		if (new_dir.size() && new_dir[new_dir.size()-1] != '/')
			new_dir += '/';
		new_dir += (*iter)->GetNameStr();

			    			// For links
		if ((*iter)->IsSymLink()) {
			GenerateSymLinkHTML(buffer, (*iter), new_dir);
		}
		else {
			GenerateDirHTML(buffer, (*iter), new_dir);
			
				// Note: subdir. allowed even inside
				// unreadable or skipped dir
			
			    			// Scan inside this dir
		    	RecursiveGenerateHTML(buffer, *iter,
					      pos+3+GetStringWidth((*iter)->GetNameStr()),
					      new_dir);
	    	}
		if (curSubDir != numSubDir-1) {	// Not the last one
						// New line
			buffer.Putc('\n');
			NewLineUpdateRowCol();
						// Add vertical bars
			for (size_t i = 0, j = 0; i < pos; i++) {
				if (j < barStack.size() && barStack[j] == i) {
					buffer.Puts("&boxv;");
					curCol++;
					
					j++;
				}
				else {
					buffer.Putc(' ');
					curCol++;
				}
			}
		}
	}
}

void	GenerateHTML(Buffer &buffer)
{
	htmlRow = 0;
	htmlCol = 0;
	curCol = 0;
	
	time_t scanTime = GetScanTime();
	struct tm *scanTime2 = localtime(&scanTime);

	ostrstream bufstr;
	gtout(bufstr, _("<TITLE>%$ %$   Last scan: %$</TITLE>"))
	     << progName << version
	     << my_strftime(_("%b %e  %H:%M"), scanTime2);
	bufstr << ends;
	buffer.Puts(bufstr.str());
	bufstr.freeze(false);

	buffer.Puts("<PRE>");		// Start preformatted text

	bool	first = true;

	for (sptr_list<DirectoryEntry>::iterator iter = dirTree.begin();
	     iter != dirTree.end(); ++iter) {
	
		if (!first) {		// New line need ?
			buffer.Putc('\n');
			NewLineUpdateRowCol();
		}
		else
			first = false;

		string new_dir = (*iter)->GetNameStr();

		if ((*iter)->IsSymLink()) {		// Symbolic link
			GenerateSymLinkHTML(buffer, (*iter), new_dir);
		}
		else {
			GenerateDirHTML(buffer, (*iter), new_dir);

			curCol = GetStringWidth(new_dir);

				// Note: subdir. allowed even inside
				// unreadable or skipped dir
						// Recursion...
			RecursiveGenerateHTML(buffer, *iter, curCol, new_dir);
		}
	}

	buffer.Puts("</PRE>");		// End preformatted text

	if (curCol != 0)		// Not end with new line
		NewLineUpdateRowCol();	// One more line needed
}

/*************************************************************************
	Find Match
*************************************************************************/

struct MatchInfo {
	string			name;
	sptr<DirectoryEntry>	dir;
	bool			is_curdir;

	MatchInfo(const string &name_, sptr<DirectoryEntry> &dir_,
		  bool is_curdir_) : 
		name(name_), dir(dir_), is_curdir(is_curdir_)
	{
	}
};

struct MatchInfoIsCurDir {
	bool operator()(MatchInfo &m) const {
		return m.is_curdir;
	}
};

void	RecursiveFindMatchDir(list<MatchInfo> &match_list, sptr<DirectoryEntry> &dir, 
			      const string &str, const string &current_dir,
			      bool add_curdir)
{
	if (dir->IsDir() && dir->isUnreadable != 1 &&
	    dir->GetNameStr().find(str) != string::npos) {	// Found

		match_list.push_back(MatchInfo(current_dir, dir, false));
	}
	else if (add_curdir && saveCwd.size() && current_dir == saveCwd) {
							// Write current directory
							// as well
		match_list.push_back(MatchInfo(saveCwd, dir, true));
	}

	for (sptr_list<DirectoryEntry>::iterator iter = dir->subDir.begin();
	     iter != dir->subDir.end(); ++iter) {

			    			// For dirs
		if ((*iter)->IsDir()) {
			string new_dir = current_dir;
			if (new_dir[new_dir.size()-1] != '/')
				new_dir += '/';
			new_dir += (*iter)->GetNameStr();
			
			RecursiveFindMatchDir(match_list, *iter, str, new_dir,
					      add_curdir);
		}
	}
}

size_t	FindMatchDir(list<MatchInfo> &match_list, const string &linktext,
		     bool add_curdir = false)
{
	match_list.clear();	
	
	if (linktext.size() == 0)		// In case getcwd fails
		return 0;
		
	for (sptr_list<DirectoryEntry>::iterator iter = dirTree.begin();
	     iter != dirTree.end(); ++iter) {
		string new_dir = (*iter)->GetNameStr();
		RecursiveFindMatchDir(match_list, *iter, linktext, new_dir,
				      add_curdir);
	}
	
	return match_list.size();
}


void	GenerateDirListHTML(list<MatchInfo> &match_list, Buffer &buffer)
{
	htmlRow = 0;
	htmlCol = 0;
	curCol = 0;
	
	size_t	tempNumMatch = match_list.size();
	size_t	numDigit = 0;
	while (tempNumMatch) {
		tempNumMatch /= 10;
		numDigit++;
	}
	
	time_t scanTime = GetScanTime();
	struct tm *scanTime2 = localtime(&scanTime);

	ostrstream bufstr;
	gtout(bufstr, _("<TITLE>%$ %$   Found: %$   Last scan: %$</TITLE>"))
	     << progName << version << match_list.size()
	     << my_strftime(_("%b %e  %H:%M"), scanTime2);
	bufstr << ends;
	buffer.Puts(bufstr.str());
	bufstr.freeze(false);

	buffer.Puts("<PRE>");		// Start preformatted text

	size_t	index = 1;

	size_t	max_length = 0;
	for (list<MatchInfo>::iterator iter = match_list.begin();
	     iter != match_list.end(); iter++) {
		size_t size = GetStringWidth(iter->name);
		if (size > max_length)
			max_length = size;
	}

	for (list<MatchInfo>::iterator iter = match_list.begin();
	     iter != match_list.end(); iter++) {
	
		if (index != 1) {		// New line need ?
			buffer.Putc('\n');
			NewLineUpdateRowCol();
		}

					// Write tags
		buffer.Puts("<A NAME=\"");
		buffer.Puts(QuoteURLChars(iter->name));
		buffer.Puts("\"></A><A CALL=\"");
		buffer.Puts(QuoteURLChars(iter->name));
		buffer.Puts("\">");

		buffer.Printf(" %*d ", numDigit, index);
		buffer.Puts(ExpandEscape(iter->name));
		buffer.Putc(' ');
		
		for (size_t i = 0; i < max_length-GetStringWidth(iter->name); i++) {
			buffer.Putc(' ');
		}

		    			// End <A NAME=...> tag
    		buffer.Puts("</A>");
		index++;
		curCol = 3+numDigit+max_length;
	}

	buffer.Puts("</PRE>");		// End preformatted text

	if (curCol != 0)		// Not end with new line
		NewLineUpdateRowCol();	// One more line needed
}

/*************************************************************************
	Add current working directory to dirTree
	Size effect: chdir() called if dir not found
*************************************************************************/

void	ExtendTree(sptr<DirectoryEntry> &d, const string &str)
{
	string::size_type delim;
	
	if ((delim = str.find('/')) != string::npos) {	// Contains subdir
							// Need to recurse
	
						// Split str into 2 string
		string str_first(str, 0, delim);
		string str_last(str, delim+1);

		sptr<DirectoryEntry> d2(new DirectoryEntry(str_first));
		d2->isSkip = 2;			// Mark as [*]

		d->subDir.push_back(d2);	// Add our own

		if (kcdConfig.cfgSortTree)
			d->subDir.sort();

		if (str_last.size()) {
			k_chdir(str_first);	// Require recursion
			ExtendTree(d2, str_last);
		}
	}
	else {					// No more subdir
		sptr<DirectoryEntry> d2(new DirectoryEntry(str));
		d2->isSkip = 2;			// Mark as [*]

		d->subDir.push_back(d2);	// Add our own

		if (kcdConfig.cfgSortTree)
			d->subDir.sort();
	}
}

FindDirInfo *RecursiveFindDir(sptr<DirectoryEntry> &d, const string &current_dir, 
			      const string &str, bool honor_cfg, bool change)
{
			// Check of subdir. of d
	for (sptr_list<DirectoryEntry>::iterator iter = d->subDir.begin();
	     iter != d->subDir.end(); ++iter) {
	     
		if ((*iter)->IsDir()) {		// Make sure it's not a
						// symbolic link

			string new_dir = current_dir;

						// Last char is not `/'
			if (new_dir[new_dir.size()-1] != '/') {
				new_dir += '/';		// Add one
			}
			new_dir += (*iter)->GetNameStr();	// Add dir. name

					// Name to long, impossible to match
					// against this dir.
			if (new_dir.size() > str.size())
				continue;	// Skip this
					
					// Same length
					// Match possible
			if (new_dir.size() == str.size()) {

				if (str != new_dir)
					continue;	// Not matched
							// Skip this
				return new FindDirInfo((*iter)(), str);	// Matched
			}

				// Now new_dir.size() < str.size()

				// Can match first part of dir.
			if (new_dir.size() != 1 && str[new_dir.size()] != '/')
				continue;


					// We're left with this case
					// str     -> /xxxx/yyyy
					// new_dir -> /xxxx
					
					// This case does not happen here
					// (new_dir is not top-level)
					// str     -> /xxxx
					// new_dir -> /

			if (str.substr(0, new_dir.size()) == new_dir) {

				// Note: subdir. allowed even inside
				// unreadable or skipped dir
					
						// Recursion...
				return RecursiveFindDir(*iter, new_dir, 
							str, honor_cfg, change);
			}
		}
	}

	if (change) {			// No match, new one added
					// if change == true

		if (honor_cfg && FindSkipDir(current_dir))
			return 0;

//		if (!d->isSkipped && kcdConfig.cfgAutoScan) {
//		}
		

		// No match, need to branch the current dir from here

			// Two cases:
			// Need to check whether parent dir. is root 
			// or not
		if (current_dir.size() == 1) {		// Branching from root dir

			k_chdir("/");			// Required to obtain permission
							// in ExtendTree(...)
		
							// Ignore the first `/'
			ExtendTree(d, str.substr(1));
		}
		else {			// str[current_dir.size()] == `/'
					// Branching from other dir
				
			k_chdir(current_dir);		// Required to obtain permission
							// in ExtendTree(...)

							// Ignore the `/'
			ExtendTree(d, str.substr(current_dir.size()+1));
		}
		return new FindDirInfo(d(), current_dir);
	}
	else
		return 0;
}

FindDirInfo *FindDir(const string &str, bool honor_cfg, bool change_top, 
		     bool change_sub)
{
	if (str.size() == 0)
		return 0;
						// Look for top-level directories

	for (sptr_list<DirectoryEntry>::iterator iter = dirTree.begin();
	     iter != dirTree.end(); ++iter) {
	
		if ((*iter)->IsDir()) {		// Make sure it's not a
						// symbolic link

			string new_dir = (*iter)->GetNameStr();
				
					// Name to long, impossible to match
					// against this dir.
			if (new_dir.size() > str.size())
				continue;	// Skip this
					
					// Same length
					// Match possible
			if (new_dir.size() == str.size()) {

				if (str != new_dir)
					continue;	// Not matched
							// Skip this
				return new FindDirInfo((*iter)(), str);	// Matched
			}

				// Now new_dir.size() < str.size()

				// Can match first part of dir.
			if (new_dir.size() != 1 && str[new_dir.size()] != '/')
				continue;

				// We're left with two cases
				// str     -> /xxxx		(first case)
				// new_dir -> /
				// str     -> /xxxx/yyyy	(second case)
				// new_dir -> /xxxx

			if (str.substr(0, new_dir.size()) == new_dir) {

				// Note: subdir. allowed even inside
				// unreadable or skipped dir
					
						// Recursion...
				return RecursiveFindDir(*iter, new_dir, 
							str, honor_cfg, change_sub);
			}
		}
	}

				// Cannot find current dir. in top-level dir
	if (change_top) {
		if (honor_cfg && ! FindStartDir(str))
			return 0;

		sptr<DirectoryEntry> d(new DirectoryEntry(str));
		d->isSkip = 2;		// Mark as [*]
		dirTree.push_back(d);	// Add our own top-level entry

		if (kcdConfig.cfgSortTree)
			dirTree.sort();

		return new FindDirInfo(d(), str);
	}
	else
		return 0;
}

FindDirInfo *FindDir(const string &str, bool honor_cfg, bool change = true)
{
	return FindDir(str, honor_cfg, change, change);
}

/*************************************************************************
	Change directory
*************************************************************************/

void	DirListCD(list<MatchInfo> &match_list, const string &init_pos,
		  const string &str)
{
	list<MatchInfo>::iterator iter;
	list<MatchInfo>::iterator current;
	for (iter = match_list.begin();
	     iter != match_list.end(); ++iter) {
		if ((*iter).is_curdir || (*iter).name == init_pos) {
			current = iter;
			break;
		}
	}
	if (iter == match_list.end()) {
						// Should not reach here
		throw ErrorGenericFile(_("cannot find directory "
					 "containing the string %$"),
				       str);
	}

	iter = current;
	next_loop(match_list, iter);
	while (k_access((*iter).name, F_OK)) {
		cout << flush;			// Avoid out-of-order display
		gtout(cerr, _("%$: skipping %$"
			      " - directory no longer exist\n"))
		     << progName << (*iter).name;

		next_loop(match_list, iter);
		if (iter == current) {
						// Can reach here
			throw ErrorGenericFile(_("cannot find directory "
						 "containing the string %$"),
					       str);
		}
	}

	FILE	*file = k_fopen(dirFile, "wb");
	if (file == NULL) {			// Cannot create or truncate
						// ~/.kcd.newdir
		throw ErrorGenericFile(_("cannot create file %$"),
				       dirFile);
	}
	
						// Output desired dir
	k_fputs((*iter).name, file);
	fclose(file);
}

void	CD(const string &linktext)
{
	LoadDirFile(1);

	Buffer	buffer;

	if (saveCwd.size())				// Current directory is available
		FindDir(saveCwd, false);

	string	hyperInitPos;
	string	hyperFindLink;
	
	GenerateHTMLStrings();

	if (linktext.size()) {				// kcd DIR
		list<MatchInfo>	match_list;
							// We add current
							// directory into the mix
							// to aid cycling through
							// all match directory
							// in the command line
		size_t	ret = FindMatchDir(match_list, linktext, true);

		size_t	count_adj = 0;
		for (list<MatchInfo>::iterator iter = match_list.begin();
		     iter != match_list.end(); ++iter) {
			if ((*iter).is_curdir) {
				count_adj = 1;
				break;
			}
		}

		if (ret > static_cast<size_t>(kcdConfig.cfgShowListThreshold)+count_adj) {

							// Show whole list

							// Current directory is not
							// need here.  Remove it.
			match_list.remove_if(MatchInfoIsCurDir());
			GenerateDirListHTML(match_list, buffer);
			
			hyperInitPos = "";
			hyperFindLink = "";
			
		}
		else {					// Jump immediately
			DirListCD(match_list, saveCwd, linktext);
			return;
		}
	}
	else {						// Browse dir tree
		GenerateHTML(buffer);

		hyperInitPos = saveCwd;
		hyperFindLink = "";
	}

#ifdef DUMP_HTML
	{
		int handle = open("/home/lerdsuwa/devel/kcd/debug", O_RDWR|O_CREAT, 0700);
		write(handle, buffer.GetPtr(), strlen(buffer.GetPtr()));
		close(handle);
	}
#endif

	HyperDocument		html(buffer.GetPtr(), buffer.GetSize());

	{
						// Display
		NCurses::Init();

		try {

			if (COLS < 50 || LINES < 10) {
				throw ErrorScreenTooSmall();
			}
			else {
				khGeometryManager	geoMan(modeHVScroll);
				khScreenManager		scrnMan(geoMan);

				khStatusWindowWithFind	findBar(scrnMan, idStatWindow, NULL);
				khURLWindowWithFind	urlBar(scrnMan, idURLWindow);
				khHScrollBar		hscroll(scrnMan, idHScrollBar);
				khVScrollBar		vscroll(scrnMan, idVScrollBar);
				khScrollBox		cursorRest(scrnMan, idScrollBox);
	
				urlBar.SetTitle(" ");		// Empty title
				scrnMan.SetCursor(&cursorRest);
				scrnMan.RequestUpdateAndRestCursor();

							// Use pre-computed 
							// document size
				khHyperWindowWithFind	hyper(scrnMan, idHyperWindow,
							      html, findBar, urlBar,
							      hscroll, vscroll, hyperInitPos,
							      1, htmlRow, htmlCol);

				findBar.SetFindWindow(&hyper);
				hscroll.SetWin(&hyper);	// Handle scroll through mouse
				vscroll.SetWin(&hyper);
				
							// Set cfgSpaceSelect flag
				khStatusWindowWithFind::spaceSelect = kcdConfig.cfgSpaceSelect;

							// Send all keyboard input
							// to hyper
				scrnMan.SetFocus(&hyper);
			
							// Paint screen
				scrnMan.RequestUpdateAndRestCursor();
				hyper.KeyboardLoop();
			}
		}
		catch (...) {
			move(LINES-1, 0);
			printf("\n\n");		// Avoid printing error
						// message on top of drawn
						// screen
			NCurses::End();
			throw;
		}
		NCurses::End();
	}
}

