/*
	editbox.cc	Edit Box
	Copyright (c) 1996-9,2000,2001 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "editbox.h"
#include CXX__HEADER_cctype

EditBox::EditBox(WINDOW *w, int x, int y, size_t len)
{
	win = w;
	xPos = x;
	yPos = y;

	normAttr = A_NORMAL;		// Default attribute
	normAttrPtr = &normAttr;	// Point to our own copy of attribute

	modified = true;

	boxLength = len;
	Clear();
}

void	EditBox::SetWin(WINDOW *w)
{
	win = w;
}

void	EditBox::SetAttr(int norm)
{
	normAttr = norm;
	normAttrPtr = &normAttr;	// Point to our own copy of attribute
}

void	EditBox::SetAttr(int *normPtr)
{
	normAttrPtr = normPtr;
}

void	EditBox::Show()
{
	attr_t	oldAttr;
	short oldPair;
	my_wattr_get(win, &oldAttr, &oldPair);
	
	wattrset(win, *normAttrPtr);
	wmove(win, yPos, xPos);		// Clear space
	for (size_t i = 0; i <= boxLength; i++)
		waddch(win, ' ');
		
	wmove(win, yPos, xPos);		// Print input
	for (size_t j = 0; j < buffer.size(); j++)
		waddch(win, buffer[j]);
	wmove(win, yPos, xPos+curPos);		// Place cursor
	my_wattr_set(win, oldAttr, oldPair);
}

void	EditBox::DoRestCursor()
{
	wmove(win, yPos, xPos+curPos);		// Place cursor
}

void	EditBox::Clear()
{
	buffer = "";

	decoder.new_seq();			// Discard partially decoded sequence
	modified = true;
	decoder.set_need_update();
	unicode_key = 0;
		
	curPos = 0;
	insert = true;
}

int	EditBox::ProcessKey(int key)
{
	if (key <= 0xFF) {
		for ( ; ; ) {
			decoder.decode(unicode_key, key);
			if (decoder.is_need_update())		// Complete sequence
				break;
			else if (decoder.is_need_char())	// Incomplete sequence
				return 0;
			else {
				bool	is_extra_char = decoder.is_extra_char();
				decoder.new_seq();		// Reset errors
				unicode_key = 0;

				ProcessKey('?');		// Display '?'
				if (!is_extra_char)
					return 0;
			}
		}
	}
	else {
		bool incomplete = decoder.is_need_char();

		decoder.new_seq();		// Discard partially decoded sequence
		unicode_key = 0;

		if (incomplete)
			ProcessKey('?');	// Display '?'

						// Continue processing control keys
	}

	int	ret = 0;
	switch(key) {
		case 0:
		case '\r':
		case '\n':
		case '\t':
			break;
		case KEY_HOME:
			curPos = 0;
			wmove(win, yPos, xPos+curPos);
			break;
		case KEY_END:
			curPos = GetWStringWidth(buffer);
			wmove(win, yPos, xPos+curPos);
			break;
		case KEY_LEFT:
			if(curPos > 0)
				curPos = GetWStringWidthFirstCharAtPos(buffer, curPos-1);
			wmove(win, yPos, xPos+curPos);
			break;
		case KEY_RIGHT:
			if(curPos < GetWStringWidth(buffer))
				curPos = GetWStringWidthFirstCharAtPos(buffer, curPos+1);
			wmove(win, yPos, xPos+curPos);
			break;
		case KEY_IC:
			insert = !insert;
			break;
		case KEY_DC:
			if(curPos < GetWStringWidth(buffer) && buffer.size()) {

				size_t	del_pos = GetWStringWidthFirstCharAtPos(buffer, curPos+1);
				buffer.erase(del_pos-1, 1);
				curPos = GetWStringWidth(buffer, del_pos-1);

				Show();
				ret = 1;

				modified = true;
				decoder.set_need_update();
			}
			break;
		case 0177:			// Backspace
		case '\b':
		case KEY_BACKSPACE:
			if(curPos > 0) {

				size_t	del_pos = GetWStringWidthFirstCharAtPos(buffer, curPos);
				buffer.erase(del_pos-1, 1);
				curPos = GetWStringWidth(buffer, del_pos-1);

				Show();
				ret = 1;

				modified = true;
				decoder.set_need_update();
			}
			break;
		default:
			if (key > 255) {	// Ignore control keys
				break;
			}
						// Insert or overtype at the end of string
			if (insert || curPos == GetWStringWidth(buffer)) {

						// Enough room for a new character
#ifdef USE_UTF8_MODE
				if (GetWStringWidth(buffer)+wcwidth(unicode_key) <= boxLength) {
					if (iswprint(unicode_key)) {
#else
				if (GetWStringWidth(buffer)+1 <= boxLength) {
					if (isprint(unicode_key)) {
#endif
								// Get this character position
						size_t	ins_pos = GetWStringWidthFirstCharAtPos(buffer, curPos);
						buffer.insert(ins_pos, 1, unicode_key);
						curPos = GetWStringWidth(buffer, ins_pos+1);

						Show();
						ret = 1;

						modified = true;
						decoder.set_need_update();
					}
				}
			}
						// Overtype at the beginning/middle of string
			else if (buffer.size()) {

#ifdef USE_UTF8_MODE
				if (iswprint(unicode_key)) {
#else
				if (isprint(unicode_key)) {
#endif

					size_t	new_pos = curPos;
					size_t	del_pos;
					do {
						del_pos = GetWStringWidthFirstCharAtPos(buffer, curPos+1);
						buffer.erase(del_pos-1, 1);
						new_pos = GetWStringWidth(buffer, del_pos-1);
					} while (new_pos > curPos);
					buffer.insert(del_pos-1, 1, unicode_key);
					curPos = GetWStringWidth(buffer, del_pos);

					Show();
					ret = 1;

					modified = true;
					decoder.set_need_update();
				}
			}
			break;
	}

	return ret;
}

const string	&EditBox::GetString()
{
#ifdef USE_UTF8_MODE
	static mbstate_t state;
#endif
	if (modified) {
#ifdef USE_UTF8_MODE
		if (IsUTF8Mode()) {
			out_buffer = "";
			Buffer	buf(1000);
			for (size_t i = 0; i < buffer.size(); ++i) {
				memset(&state, 0, sizeof(mbstate_t));
				if (wcrtomb(buf.GetPtr(), buffer[i], &state)
				    == static_cast<size_t>(-1))
					throw ErrorBadSequence();
				out_buffer += buf.GetPtr();
			}
		}
		else
#endif
		     {
			out_buffer = "";
			for (size_t i = 0; i < buffer.size(); ++i)
				out_buffer += buffer[i];
		}
		modified = false;
	}
	return out_buffer;
}
