/*
 * Convert ASCII numbers to or from BCD representations.
 * $Id: bcd.c 1.1 Fri, 04 Apr 1997 09:29:03 -0500 dyfet $
 * Copyright (c) 1997 by Tycho Softworks.
 * For condititions of distribution and use, see product license.
 *
 * Abstract:
 *	BCD is a format for storing packed decimal data within a nibble.
 *	We assume "big endian" BCD, in that the high nibble contains the
 *	first digit within a byte.  Any unused digits are filled with the
 *	nibble value of $F.  This module provides the services needed to
 *	convert BCD data to and from ASCII strings.
 *
 * Functions:
 *	str2bcd() - convert ASCII string to packed bcd.
 *	bcd2str() - convert packed bcd to ASCII string.
 */

#include <other/strcvt.h>

/*
 * Convert null terminated ASCII string to packed bcd data.
 *
 * Abstract:
 *	An ASCII input string is converted to binary packed decimal
 *	data.  Any unused digits are filled with $f.  The size specified
 *	for digits is filled, either with available digits from the input
 *	string, or with $f nibbles once no more digits are available.
 *
 * Paramaters:
 *	bcd - pointer to start of bcd data to store.
 *	str - null terminated input string.
 *	max - maximum number of digits in bcd data.
 *
 * Returns:
 *	pointer to first non-BCD digit in input string.
 *
 * Exceptions:
 *	If NULL input string, returns NULL.  If more digits exist in
 *	the input string than are available in bcd storage (data
 *	overflow), the remaining digits are ignored.
 */

char	*str2bcd(uchar *bcd, char *str, int max)
{
	uchar	packed = 0xff;
	bool	low = FALSE;

	if(!str)
		return NULL;

	while(isdigit(*str))
	{
		packed = packed / 16; 
		packed |= ((*(str++) - '0') * 16);
	 	if(low)
		{
			*(bcd++) = packed;
			packed = 0xff;
			if(!--max)
				return str;

			low = FALSE;
		}
		else
			low = TRUE;
	}	
	if(low)
	{
		*(bcd++) = packed;
		--max;
	}
	while(max--)
		*(bcd++) = 0xff;
				
	return str;
}

/*
 * Convert binary coded data to ASCII text.
 *
 * Abstract:
 *	A block of bcd data of up to a specified length is read, each
 *	digit being converted to an ASCII character code.  If a $f is
 *	found before 'len' digits are examined, then the program
 *	completes with a shorter number.
 *
 * Paramaters:
 *	str - start of output string to receive converted data.
 *	bcd - bcd data input.
 *	len - maximum number of bcd digits.
 *
 * Return:
 *	pointer to next available bcd number in memory (or past end of
 *	bcd data block).  The output string (str) is also given a null
 *	terminating byte.
 *
 * Exceptions:
 *	A NULL bcd pointer returns a NULL string.
 */

uchar	*bcd2str(char *str, uchar *bcd, int len)
{
	uchar	nib1, nib2;
	if(!bcd)
		return NULL;

	while(len--)
	{
		nib1 = *bcd / 16;
		nib2 = *bcd % 16;
		if(nib1 < 16)
			*(str++) = nib1 + '0';	
		else
			break;
		if(nib2 < 16)
			*(str++) = nib2 + '0';
		else
			break;
		++bcd;
	}
	*str = 0;
	return bcd;
}
	
