/*GPL*START*
 * shelfcalc - misc calculations around shelves
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#include "tappconfig.h"
#include <stdlib.h>

// history:
// 1998: (started in Oct 1998)
// 26 Oct 03:02 maxnum added
// 01 Dec 20:30 v0.2.0 price field added


const char *options[] = {
   "#usage='Usage: %n LIST [OPTIONS]\n\n"
     "misc. calculations around shelves\n"
     "LIST is a list of lengths wich are of the form LENGTH[:MAXNUM][@PRICE]\n"
     ,
     "#trailer='\n%n version %v *** (C) 1998 by Johannes Overmann\ncomments, bugs and suggestions welcome: %e\n%gpl'",
     "#onlycl", // only command line options
     "#stopat--", // stop option scanning after a -- parameter
     "#ignore_negnum", // so -20 is not an option
     // options
     "name=split-width        , type=double, char=s, param=WIDTH, help=print all combinations of creating WIDTH out of the lengths specified as normal arguments, headline=options:",
     "name=tolerance          , type=double, char=t, default=0.001, param=TOL, help=allow a tolerance of TOL while checking against a specific width",
     "name=show-all           , type=switch, char=a, help=show all solutions: also those which only differ in order of the elements",
     "name=single-line        , type=switch, char=c, help=compact format: print only a single line",
   
   
     "name=verbose            , type=switch, char=v, help=verbose execution, headline=common options:",
     "EOL" // end of list
};

bool verbose;




// split width
// args == length arguments
// sol == solutions
// part_sol == partial solution
void split_width(double width, double tol, const TArray<double>& args, const TArray<int>& part_sol, 
		 const TArray<int>& part_avail, TArray<TArray<int> >& sol, bool show_all) {
   for(int i=(show_all||(part_sol.num()==0))?0:part_sol.lastElement(); i < args.num(); i++) {
      if(part_avail[i]==0) continue;
      TArray<int> s(part_sol);
      s += i;
      
      // test for fit
      if((args[i] >= width-tol) && (args[i] <= width+tol)) {
	 sol += s;
      }
      // try new solution
      if(args[i] < width+tol) {
	 TArray<int> avail(part_avail);
	 avail[i]--;
	 split_width(width - args[i], tol, args, s, avail, sol, show_all);	 
      }
   }
}


int main(int argc, char *argv[]) {
   // get parameters
   TAppConfig ac(options, "options", argc, argv, 0, 0, VERSION);
   
   // setup
   verbose = ac("verbose");   
   bool sline = ac("single-line");
   
   // split-width
   if(ac.wasSetIn("split-width")!=DEFAULT) {
      double width = ac.getDouble("split-width");
      double tol   = ac.getDouble("tolerance");
      TArray<double> a;
      TArray<int> avail;
      TArray<double> prices;
      int i,j;
      double sum;
      double price;
      bool calc_price = false;
      const char *p;
      char *q;
      
      // read params
      for(i=0; i < ac.numParam(); i++) {
	 // get length
	 TString s(ac.param(i));
	 p = s;
	 a[i] = strtod(p, &q);
	 p = q;

	 // get avail
	 avail[i] = -1; // unlimited availability
	 if(*p==':') {
	    p++;
	    if(*p==0) p--; // force error
	    else {
	       avail[i] = strtol(p, &q, 10);
	       p = q;
	    }
	 } 
	 
	 // get price
      	 prices[i] = 0.0; // nothing
	 if(*p=='@') {
	    p++;
	    if(*p==0) p--; // force error
	    else {
	       prices[i] = strtod(p, &q);
	       p = q;
	    }
	 }
	 if(prices[i]!=0.0) calc_price = true;

	 if(*p) userError("error while parsing element '%s'! (arg %d)\n", *ac.param(i), i+1);
	 if(a[i] <= 0.0) userError("length arguments must be > 0 ('%s', arg %d)\n", *ac.param(i), i+1);
	 if(avail[i] <= -2) userError("maxnum arguments must be >=0! ('%s', arg %d)\n", *ac.param(i), i+1);
      }

      // try combinations 
      TArray<TArray<int> > sol; // solutions
      split_width(width, tol, a, TArray<int>(), avail, sol, ac("show-all"));
      
      // print solutions
      if(sol.num()==0) {
	 printf("no combinations found\n");
      } else {
	 if(!sline) {
	    printf("%d combinations found:\n", sol.num());
	    for(i=0; i < sol.num(); i++) {
	       sum = 0;
	       price = 0;
	       for(j=0; j < sol[i].num(); j++) {
		  sum   +=      a[sol[i][j]];
		  price += prices[sol[i][j]];
	       }
	       if(calc_price) printf("%9.2f  ", price);
	       printf("%6g = ", sum);
	       for(j=0; j < sol[i].num()-1; j++) {
		  printf("%g + ", a[sol[i][j]]);
	       }
	       printf("%g\n", a[sol[i][j]]);
	    }
	 } else {
	    printf("%6g: ", width);
	    for(i=0; i < sol.num(); i++) {
	       for(j=0; j < sol[i].num()-1; j++) {
		  printf("%g+", a[sol[i][j]]);
	       }
	       printf("%g%s", a[sol[i][j]], i==sol.num()-1?"\n":", ");
	    }	    
	 }
      }
   }
   
   // normal exit
   return 0;
}


