/*GPL*START*
 * mailsort - a mail sorting utility
 * 
 * Copyright (C) 1998 by Johannes Overmann <overmann@iname.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * *GPL*END*/  

#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include "tappconfig.h"
#include "tregex.h"
#include "tfiletools.h"

// 1998:
// Wed 08 Jan: v0.0 started (to replace nsmover which sucks)
// Mon 12 Jan: v0.1 first usable version (no big test yet)
// Mon 12 Jan: v0.2 statistics added
// Mon 24 Aug: v0.3 compile with -Weffc++ (EGCS)
// Wed 30 Sep: v0.3.1 prepare for sunsite

// app version in Makefile!

const char *options[] = {
   "#usage='Usage: [OPTION...]\n\n"
     "this program sorts mails from an inbox to several folders,\n"
     "all files have to be in the standard sendmail format\n",
   "#trailer='\n%n version %v *** (C) 1997 by Johannes Overmann\ncomments, bugs and suggestions welcome: %e\n%gpl'",
   // options
   "name=maildir          , type=string, char=b, musthave, param=PATH, help='this is the base directory of all mailfiles (inbox and folders), note that this can be overridden by an absolute path', headline=options:",
   "name=inbox            , type=string, char=i, musthave, param=FILE, help='read mails from FILE'",
   "name=filter           , type=string, char=f, string-mode-append, string-append-separator='\1', help='this defines a filter rule, syntax: <folder> <regex>'",
   
   "name=dummy            , type=switch, char=0, help='do *not* write/change anything, just test', headline=common options:",
   "name=quiet            , type=switch, char=q, help='quiet execution, no progress'",
   "EOL" // end of list
};



// config
const char *color_bold = "\033[33;01m";
const char *color_nor  = "\033[00m";
const TString mail_separator("From ");
const int nice_folder_str_len = 20;
const int line_progress = 57

;

// Filter type
struct Filter {
   Filter(): num_match(0), folder(), folder_str(), regex_str(), regex(0) {};
   Filter(const Filter& a): num_match(a.num_match), folder(a.folder), 
   folder_str(a.folder_str), regex_str(a.regex_str), regex(a.regex) {};
   const Filter& operator=(const Filter& a) {
      num_match  = a.num_match;
      folder     = a.folder;
      folder_str = a.folder_str;
      regex_str  = a.regex_str;
      regex      = a.regex;
      return *this;
   }
   int num_match;
   TString folder;
   TString folder_str;
   TString regex_str;
   TRegEx *regex;
};

void dump(const Filter&) {printf("Filter");};
void dump(FILE *f, const Filter&) {fprintf(f, "Filter");};
TString className(const Filter&) {return "Filter";}



int main(int argc, char *argv[]) {
   int i,j;

   
   // get parameters
   TAppConfig ac(options, "options", argc, argv, "MAILSORT", "mailsortrc", VERSION);
   
   
   // setup
   bool quiet = ac("quiet");   
   bool dummy = ac("dummy");
   TString maildir = ac.getString("maildir");
   TArray<TString> filter = split(ac.getString("filter"), "\1", false, false);
   TString inbox_name = maildir;
   if(ac.getString("inbox").isAbsolutePath()) 
     inbox_name = ac.getString("inbox");
   else
     inbox_name += "/" + ac.getString("inbox");
   inbox_name.normalizePath();
   
   
   // dummy message
   if(dummy && (!quiet)) printf("dummy mode: nothing will change ...\n");
   
   
   // check/build filters
   TArray<Filter> filt;
   for(i=0; i < filter.num(); ++i) {
      int scanner = 0;
      filter[i].skipSpace(scanner);
      TString f = filter[i].scanUpTo(scanner, " \t\n=");
      filter[i].perhapsSkipOneChar(scanner, '=');
      filter[i].skipSpace(scanner);
      TString r = filter[i].scanRest(scanner);
      r.cropSpace();
      if(f && r) {
	 filt[i].folder_str = f;
	 if(f.isAbsolutePath())
	   filt[i].folder = f;
	 else 
	   filt[i].folder = maildir + "/" + f;
	 filt[i].folder.normalizePath();
	 filt[i].num_match = 0;
	 filt[i].regex_str = r;
	 filt[i].regex = new TRegEx(r, REG_EXTENDED|REG_NOSUB|REG_NEWLINE);
	 filt[i].regex->exitOnError();
	 continue;
      }
      userError("invalid filter statement '%s'! (should be: filter <folder> <regex>)\n", *filter[i]);
   }   
   if(filt.num()==0)
     userError("no filters specified!\n");
   
   
   // print filters
   if(!quiet) {
      printf("%d filters found:\n", filt.num());      
      for(i=0; i< filt.num(); i++) {
	 printf("'%s%s%s'%s <== '%s%s%s'\n", 
		color_bold, *filt[i].folder_str, color_nor, 
		*TString(' ', nice_folder_str_len-filt[i].folder_str.len()),
		color_bold, *filt[i].regex_str, color_nor);
      }
   }

   
   // read inbox
   TArray<TString> inbox;
   {
      FILE *f = fopen(inbox_name, "r");
      if(f==0) {
	 userError("can't open inbox '%s' for reading!\n", *inbox_name);
      }
      inbox = loadTextFile(f);
      fclose(f);
      if(!quiet)
	printf("%d lines read from inbox '%s'\n", inbox.num(), *inbox_name);
   }
   

   // filter mails
   TString new_inbox;   
   TAssocArray<TString,TString> new_folder;
   int start=0;
   TString match_folder;
   int total_mails=0;
   for(i=0; i < inbox.num(); ++i) {
      if(inbox[i].hasPrefix(mail_separator) && i) {	 
	 // new mail begins, save old
	 if(match_folder) {
	    for(j=start; j < i; ++j) 
	      new_folder[match_folder] += inbox[j];
	 } else {
	    for(j=start; j < i; ++j) 
	      new_inbox += inbox[j];
	 }
	 total_mails++;
	 // start new mail
	 start = i;
	 match_folder.empty();
      }
      if(!match_folder) {
	 // try to match line
	 for(j=0; j < filt.num(); ++j) {
	    if(filt[j].regex->match(inbox[i])) {
	       match_folder = filt[j].folder;
	       ++filt[j].num_match;
	       break;
	    }
	 }
      }      
      if(!quiet) {
	 // progress
	 if((i%line_progress)==0) {
	    printf("%d/%d   \r", i, inbox.num());
	    fflush(stdout);
	 }
      }
   }
   // last mail
   if(match_folder) {
      for(j=start; j < i; ++j) 
	new_folder[match_folder] += inbox[j];
   } else {
      for(j=start; j < i; ++j) 
	new_inbox += inbox[j];
   }
   total_mails++;
   
   
   // append to folder files
   for(TAssocArrayIter<TString,TString> p(new_folder); p; ++p) {
      if(*p) {
	 if(!quiet) printf("appending mails to folder '%s%s%s'\n", 
			   color_bold, *(p()), color_nor);
	 if(!dummy) {
	    FILE *f = fopen(p(), "a");
	    if(f==0)
	      userError("can't open folder '%s' for appending!\n", *(p()));
	    if((*p).write(f) != (*p).len()) 
	      userError("error while writing to folder '%s'! (disk full?)\n", *(p()));	   
	    fclose(f);
	 }
      }
   }
   
   
   // update inbox
   {
      if(!quiet)
	printf("updating inbox '%s'\n", *inbox_name);
      TString tmp_inbox = inbox_name + ".tmp";
      if(!dummy) {
	 FILE *f = fopen(tmp_inbox, "w");
	 if(f==0)
	   userError("can't open temp inbox '%s' for writing!\n", *tmp_inbox);
	 new_inbox.write(f);
	 fclose(f);
	 if(rename(tmp_inbox, inbox_name)) {
	    userError("can't rename '%s' as '%s'!\n", *tmp_inbox, *inbox_name);
	 }
      }
   }
   
   
   // normal exit
   int moved_mails=0;
   for(i=0; i< filt.num(); ++i) {
      delete filt[i].regex;
      if(!quiet) {
	 printf("%3d: '%s%s%s'%s <== '%s%s%s'\n", filt[i].num_match,
		color_bold, *filt[i].folder_str, color_nor, 
		*TString(' ', nice_folder_str_len-filt[i].folder_str.len()),
		color_bold, *filt[i].regex_str, color_nor);	 
	 moved_mails += filt[i].num_match;
      }
   }
   if(!quiet)
     printf("%3d/%d mails moved\n", moved_mails, total_mails);
   return 0;
}


