/* xtimer.c -- very simple application timers. */

/* Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Written by Tudor Hulubei and Andrei Pitis.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <signal.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "xmalloc.h"
#include "xstring.h"
#include "xtimer.h"
#include "misc.h"


static int xtimers = 0;
static int xtimer_period = 60;
static xtimer_t **xtimer_data = NULL;


/* Register a new timer.  */

int
xtimer_register(xtimer_hook)
    xtimer_function xtimer_hook;
{
    xtimers++;

    xtimer_data = (xtimer_t **)xrealloc(xtimer_data,
                                        xtimers * sizeof(xtimer_t*));
    xtimer_data[xtimers - 1] = (xtimer_t *)xmalloc(sizeof(xtimer_t));
    xtimer_data[xtimers] = NULL;
    memset(xtimer_data[xtimers - 1], 0, sizeof(xtimer_t));
    xtimer_data[xtimers - 1]->hook = xtimer_hook;
    return xtimers - 1;
}


/* Set xtimer flags.  Actually used only to specify the safe-to-call
   status.  */

void
xtimer_set_flags(xtimer, flags)
    int xtimer;
    int flags;
{
    xtimer_data[xtimer]->flags = flags;
}


/* This is the xtimer signal handler.  This function gets called at every
   SIGALRM signal and calls those xtimer hooks that are available at the
   time. The other hooks are ignored, only their statistics being updated.  */

RETSIGTYPE
xtimer_signal_handler(signum)
    int signum;
{
    int xtimer;

    signal(signum, xtimer_signal_handler);

    /* Tell the system to send the alarm signal again after about
       xtimer_period seconds.  I don't want to call time() & localtime()
       here to get the exact amount of time after which the next alarm
       should be scheduled in order to prevent races: localtime() is
       called from some other places too and I don't know if it is
       reentrant.  I have tooked a simpler approach and scheduled the
       next alarm signal after xtimer_period seconds.  */
    alarm(xtimer_period);

    for (xtimer = 0; xtimer < xtimers; xtimer++)
    {
        xtimer_data[xtimer]->interrupts++;

        if (xtimer_data[xtimer]->flags & XT_SAFE_TO_CALL)
        {
            xtimer_data[xtimer]->hits++;
            (*xtimer_data[xtimer]->hook)(xtimer_data[xtimer], signum);
        }
    }
}


/* Initialize the xtimer signal handling.  */

void
xtimer(status)
    int status;
{
    signal(SIGALRM, status ? xtimer_signal_handler : SIG_IGN);
    alarm(xtimer_period - get_local_time()->tm_sec);
}
