/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: util.cc,v 1.2 1998/05/22 18:33:19 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "util.h"
#include "Sample.h"

char *strdupCC(const char *s)
{
  char *ret;
  long len;

  if (s == NULL) {
    return NULL;
  }

  len = strlen(s);

  ret = new char[len + 1];

  strcpy(ret, s);

  return ret;
}

long fullRead(int fd, void *buf, long count)
{
  long n = 0;
  long nread = 0;
  
  do {
    do {
      n = read(fd, buf + nread, count);
    } while (n < 0 && (errno == EAGAIN || errno == EINTR));

    if (n < 0) {
      return -1;
    }

    if (n == 0) {
      return nread;
    }
    
    count -= n;
    nread += n;
  } while (count > 0);

  return nread;
}
  
// Returns length in samples for given audio file.
// return: 1: file cannot be opened
//         2: 'fstat' failed
//         3: file header corruption
//         0: OK
int audioDataLength(const char *fname, unsigned long *length)
{
  int fd;
  struct stat buf;
  long headerLength = 0;

  *length = 0;

  if (access(fname, R_OK) != 0) {
    return 1;
  }

  if (audioFileType(fname) == 2) {
    if ((headerLength = waveHeaderLength(fname)) < 0) {
      return 3;
    }
  }

  if ((fd = open(fname, O_RDONLY)) > 0) {
    if (fstat(fd, &buf) == 0) {
      *length = (buf.st_size - headerLength) / sizeof(Sample);
    }
    else {
      close(fd);
      return 2;
    }
    close(fd);
  }
  else {
    return 1;
  }

  return 0;
}

long readLong(FILE *fp)
{
  unsigned char c1 = getc(fp);
  unsigned char c2 = getc(fp);
  unsigned char c3 = getc(fp);
  unsigned char c4 = getc(fp);

  return ((long)c4 << 24) | ((long)c3 << 16) | ((long)c2 << 8) | (long)c1;
}

short readShort(FILE *fp)
{
  unsigned char c1 = getc(fp);
  unsigned char c2 = getc(fp);

  return ((short)c2 << 8) | (short)c1;
}
  

long waveHeaderLength(const char *filename)
{
  FILE *fp;
  char    magic[4];
  long headerLen = 0;
  long len;
  short waveFormat;
  short waveChannels;
  long waveRate;
  short waveBits;

  if ((fp = fopen(filename, "r")) == NULL) {
    fprintf(stderr, "Cannot open audio file \"%s\" for reading: %s\n",
	    filename, strerror(errno));
    return -1;
  }

  if (fread(magic, sizeof(char), 4, fp) != 4 ||
      strncmp("RIFF", magic, 4) != 0) {
    fprintf(stderr, "%s: not a WAVE file.\n", filename);
    fclose(fp);
    return -1;
  }

  readLong(fp);

  if (fread(magic, sizeof(char), 4, fp) != 4 ||
      strncmp("WAVE", magic, 4) != 0) {
    fprintf(stderr, "%s: not a WAVE file.\n", filename);
    fclose(fp);
    return -1;
  }

  // search for format chunk
  for (;;) {
    if (fread(magic, sizeof(char), 4, fp) != 4) {
      fprintf(stderr, "%s: corrupted WAVE file.\n", filename);
      fclose(fp);
      return -1;
    }

    len = readLong(fp);

    if (strncmp("fmt ", magic, 4) == 0) {
      // format chunk found
      break;
    }

    // skip chunk data
    while (len > 0 && !feof(fp)) {
      getc(fp);
      len--;
    }
  }

  if (len < 16) {
    fprintf(stderr, "%s: corrupted WAVE file.\n", filename);
    fclose(fp);
    return -1;
  }

  waveFormat = readShort(fp);

  if (waveFormat != 1) {
    // not PCM format
    fprintf(stderr, "%s: not in PCM format.\n", filename);
    fclose(fp);
    return -1;
  }

  waveChannels = readShort(fp);
  if (waveChannels != 2) {
    fprintf(stderr, "%s: found %d channels, require 2 channels.\n",
	    filename, waveChannels);
    fclose(fp);
    return -1;
  }

  waveRate = readLong(fp);
  if (waveRate != 44100) {
     fprintf(stderr, "%s: found sampling rate %ld, require 44100.\n",
	    filename, waveRate);
     fclose(fp);
     return -1;
  }
  
  readLong(fp); // average bytes/second
  readShort(fp); // block align
  
  waveBits = readShort(fp);
  if (waveBits != 16) {
    fprintf(stderr, "%s: found %d bits per sample, require 16.\n",
	    filename, waveBits);
    fclose(fp);
    return -1;
  }
  
  len -=16;
  while (len > 0 && !feof(fp)) {
    getc(fp);
    len--;
  }

  // search wave data chunk
  for (;;) {
    if (fread(magic, sizeof(char), 4, fp) != 4) {
      fprintf(stderr, "%s: corrupted WAVE file.\n", filename);
      fclose(fp);
      return -1;
    }
    
    len = readLong(fp);
    if (strncmp("data", magic, 4) == 0) {
      // found data chunk
      break;
    }
     
    while (len > 0 && !feof(fp)) {
      getc(fp);
      len--;
    }
  }

  if ((headerLen = ftell(fp)) < 0) {
    fprintf(stderr, "%s: cannot determine file position: %s\n",
	    filename, strerror(errno));
    fclose(fp);
    return -1;
  }

  fclose(fp);

  //printf("%s: header length: %ld\n", filename, headerLen);

  return headerLen;
}


// determines type of audio file
// return: 1: raw samples
//         2: wav file
int audioFileType(const char *filename)
{
  char *p;

  if ((p = strrchr(filename, '.')) != NULL) {
    if (strcmp(p, ".wav") == 0) {
      return 2;
    }
  }

  return 1;
}

void swapSamples(Sample *buf, unsigned long len)
{
  unsigned long i;

  for (i = 0; i < len; i++) {
    buf[i].swap();
  }
}
