/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: Track.cc,v 1.2 1998/05/22 18:33:19 mueller Exp $";

#include <config.h>

#include <stddef.h>
#include <assert.h>
#include <ctype.h>

#include "Track.h"

Track::Track(Type t) : length_(0), start_(0)
{
  type_ = t;

  nofSubTracks_ = 0;
  subTracks_ = lastSubTrack_ = NULL;

  nofIndices_ = 0;
  index_ = new Msf[98](0);

  readPos_ = 0;
  readSubTrack_ = NULL;
  open_ = 0;

  isrcValid_ = 0;

  flags_.copy = 1;         // digital copy permitted
  flags_.preEmphasis = 0;  // no pre-emphasis
  flags_.audioType = 0;    // two channel audio
}

Track::Track(const Track &obj) : length_(obj.length_), start_(obj.start_)
{
  int i;
  SubTrack *run;

  type_ = obj.type_;

  nofSubTracks_ = obj.nofSubTracks_;
  subTracks_ = lastSubTrack_ = NULL;
  for (run = obj.subTracks_; run != NULL; run = run->next_) {
    if (subTracks_ == NULL) {
      subTracks_ = lastSubTrack_ = new SubTrack(*run);
    }
    else {
      lastSubTrack_->next_ = new SubTrack(*run);
      lastSubTrack_ = lastSubTrack_->next_;
    }
  }

  nofIndices_ = obj.nofIndices_;
  index_ = new Msf[98](0);
  for (i = 0; i < nofIndices_; i++) {
    index_[i] = obj.index_[i];
  }
  
  readPos_ = 0;
  readSubTrack_ = NULL;
  open_ = 0;

  isrcValid_ = obj.isrcValid_;
  memcpy(isrcCountry_, obj.isrcCountry_, 2);
  memcpy(isrcOwner_, obj.isrcOwner_, 3);
  memcpy(isrcYear_, obj.isrcYear_, 2);
  memcpy(isrcSerial_, obj.isrcSerial_, 5);

  flags_ = obj.flags_;
}


Track::~Track()
{
  SubTrack *run = subTracks_;
  SubTrack *next = NULL;

  while (run != NULL) {
    next = run->next_;
    delete run;
    run = next;
  }

  delete[] index_;
}


// Appends given sub-track to list of sub-tracks.
// return: 0: OK
//         1: tried to append PAD sub-track
int Track::append(const SubTrack &strack)
{
  if (strack.type() == SubTrack::PAD) {
    return 1;
  }

  if (lastSubTrack_ != NULL && lastSubTrack_->type() == SubTrack::PAD) {
    // remove padding sub track
    SubTrack *run = subTracks_;

    while (run->next_ != lastSubTrack_) {
      run = run->next_;
    }
    
    delete run->next_;
    run->next_ = NULL;
    lastSubTrack_ = run;
    nofSubTracks_ -= 1;
  }

  // append sub track
  if (subTracks_ == NULL) {
    subTracks_ = lastSubTrack_ = new SubTrack(strack);
  }
  else {
    lastSubTrack_->next_ = new SubTrack(strack);
    lastSubTrack_ = lastSubTrack_->next_;
  }

  nofSubTracks_ += 1;

  update();

  return 0;
}


// Traverses all sub-tracks to update summary data of track.
void Track::update()
{
  unsigned long slength = 0; // length of track in samples
  unsigned long padLen;      // padding length
  SubTrack *run;
  SubTrack *pad = NULL;
  SubTrack *prevPad = NULL;

  // collect length in samples of all sub-tracks and set start field of each
  // sub-track
  for (run = subTracks_; run != NULL; run = run->next_) {
    run->start(slength); // set start position of sub-track

    if (run->type() == SubTrack::PAD) {
      // track contains at least one padding sub-track at the end of the
      // sub-track list
      pad = run;
      break;
    }
    else {
      prevPad = run; // to collect sub-track preceeding a padding sub-track
    }
    
    slength += run->length();
  }
  
  if ((padLen = slength % SAMPLES_PER_BLOCK) != 0) {
    padLen = SAMPLES_PER_BLOCK - padLen;
    if (pad == NULL) {
      // add padding sub-track
      lastSubTrack_->next_ = new SubTrack(SubTrack::PAD, 
					  slength,
					  AudioData(AudioData::SILENCE,
						    padLen));
      lastSubTrack_ = lastSubTrack_->next_;
      nofSubTracks_ += 1;
    }
    else {
      pad->length(padLen);
    }
    slength += padLen;
  }
  else {
    // remove existing padding sub-track 
    if (pad != NULL) {
      delete pad, pad = NULL;
      prevPad->next_ = NULL;
      nofSubTracks_ -= 1;
      lastSubTrack_ = prevPad;
    }
  }
  // at this point 'slength' should be a multiple of 'SAMPLES_PER_BLOCK'
  assert(slength % SAMPLES_PER_BLOCK == 0);

  length_ = Msf(slength / SAMPLES_PER_BLOCK);

  // reset 'start_' if necessary
  if (start_.lba() >= length_.lba()) {
    start_ = Msf(0);
  }
}

// sets logical start of track, everthing before start (if != 0) is taken
// as pre-gap
// return: 0: OK
//         1: given start behind track end
int Track::start(Msf s)
{
  if (s.lba() >= length_.lba()) {
    return 1;
  }

  start_ = s;

  return 0;
}

// initiates reading track
// return: 0: OK
//         1: data file could not be opened
//         2: could not seek to start position
int Track::openData() const
{
  int ret = 0;
  assert(open_ == 0);

  ((Track *)this)->open_ = 1;
  ((Track *)this)->readPos_ = 0;

  if (type_ == AUDIO) {
    ((Track *)this)->readSubTrack_ = subTracks_;
    ret = readSubTrack_->openData();
  }

  return ret;
}

void Track::closeData() const
{
  ((Track *)this)->open_ = 0;
  ((Track *)this)->readPos_ = 0;
  if (readSubTrack_ != NULL) {
    readSubTrack_->closeData();
  }
  ((Track *)this)->readSubTrack_ = NULL;
}


long Track::readData(char *buf, long len) const
{
  long ret;

  assert(open_ != 0);

  if (readPos_ + len > length_.lba()) {
    if ((len = length_.lba() - readPos_) <= 0) {
      return 0;
    }
  }

  switch (type_) {
  case AUDIO:
    ret = readAudioData((Sample *)buf, len * SAMPLES_PER_BLOCK);
    if (ret > 0) {
      assert(ret % SAMPLES_PER_BLOCK == 0);
      ret /= SAMPLES_PER_BLOCK;
    }
    break;

  default:
    ret = 0;
  }

  if (ret > 0) {
    ((Track *)this)->readPos_ += ret;
  }

  return ret;
}

long Track::readAudioData(Sample *buf, long len) const
{
  long actLen;
  long count = len;
  long nread = 0;

  while (count > 0) {
    actLen = readSubTrack_->readData(buf + nread, count);

    if (actLen < 0) {
      return -1;
    }

    if (actLen != count) {
      // end of audio data in sub-track reached, skip to next sub-track
      readSubTrack_->closeData();

      ((Track *)this)->readSubTrack_ = readSubTrack_->next_;

      // next sub-track must exists since requested length match available data
      assert(readSubTrack_ != NULL); 

      if (readSubTrack_->openData() != 0) {
	return -1;
      }
    }

    count -= actLen;
    nread += actLen;
  }

  return len;
}

// Appends given index to index increment list.
// return: 0: OK
//         1: > 98 index increments 
//         2: index at or beyond track end
//         3: index at start
int Track::appendIndex(const Msf &index)
{
  if (nofIndices_ == 98) {
    return 1;
  }

  if (index.lba() >= (length_ - start_).lba()) {
    return 2;
  }

  if (index.lba() == 0) {
    return 3;
  }

  index_[nofIndices_] = index;
  nofIndices_ += 1;

  return 0;
}

// returns index increment
Msf Track::getIndex(int i) const
{
  if (i >= nofIndices_ || i < 0) {
    return Msf(0);
  }
  else {
    return index_[i];
  }
}

int Track::check() const
{
  SubTrack *st;
  int ret;

  for (st = subTracks_; st != NULL; st = st->next_) {
    if ((ret = st->check()) != 0) {
      return ret;
    }
  }

  return 0;
}

// Sets ISRC code. Expected string: "CCOOOYYSSSSS"
//                 C: country code (ASCII)
//                 O: owner code (ASCII)
//                 Y: year ('0'-'9')
//                 S: serial number ('0'-'9')
// return: 0: OK
//         1: ilegal ISRC string
int Track::isrc(const char *isrc)
{
  int i;

  if (strlen(isrc) != 12) {
    return 1;
  }

  for (i=0; i < 5; i++) {
    if (!(isdigit(isrc[i]) || isupper(isrc[i]))) {
      return 1;
    }
  }

  for (i = 5; i < 12; i++) {
    if (!isdigit(isrc[i])) {
      return 1;
    }
  }

  isrcCountry_[0] = isrc[0];
  isrcCountry_[1] = isrc[1];

  isrcOwner_[0] = isrc[2];
  isrcOwner_[1] = isrc[3];
  isrcOwner_[2] = isrc[4];


  // store BCD
  isrcYear_[0] = isrc[5] - '0';
  isrcYear_[1] = isrc[6] - '0';
  
  isrcSerial_[0] = isrc[7] - '0';
  isrcSerial_[1] = isrc[8] - '0';
  isrcSerial_[2] = isrc[9] - '0';
  isrcSerial_[3] = isrc[10] - '0';
  isrcSerial_[4] = isrc[11] - '0';
  
  isrcValid_ = 1;

  return 0;
}

int Track::isPadded() const
{
  if (lastSubTrack_ != NULL && lastSubTrack_->type() == SubTrack::PAD) {
    return 1;
  }

  return 0;
}

// writes out track data in TOC file syntax
void Track::print(ostream &out) const
{
  SubTrack *st;

  out << "TRACK " ;

  switch (type()) {
  case AUDIO:
    out << "AUDIO";
    break;
  case MODE1:
    out << "MODE1";
    break;
  case MODE2:
    out << "MODE2";
    break;
  case MODE2FORM1:
    out << "MODE2FORM1";
    break;
  case MODE2FORM2:
    out << "MODE2FORM2";
    break;
  }

  out << endl;

  if (!copyPermitted())
    out << "NO ";
  out << "COPY" << endl;

  if (!preEmphasis())
    out << "NO ";
  out << "PRE_EMPHASIS" << endl;

  if (audioType() == 0)
    out << "TWO_CHANNEL_AUDIO" << endl;
  else
    out << "FOUR_CHANNEL_AUDIO" << endl;
  
  if (isrcValid()) {
    out << "ISRC \"" << isrcCountry(0) << isrcCountry(1)
	<< isrcOwner(0) << isrcOwner(1) << isrcOwner(2)
	<< (char)(isrcYear(0) + '0') << (char)(isrcYear(1) + '0')
	<< (char)(isrcSerial(0) + '0') << (char)(isrcSerial(1) + '0')
	<< (char)(isrcSerial(2) + '0') << (char)(isrcSerial(3) + '0')
	<< (char)(isrcSerial(4) + '0') << "\"" << endl;
  }


  for (st = subTracks_; st != NULL; st = st->next_) {
    st->print(out);
  }

  if (start_.lba() != 0) {
    out << "START " << start_.str() << endl;
  }
  
}
