/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: main.cc,v $
 * Revision 1.3  1998/07/28 13:49:35  mueller
 * Command SHOW_TOC: Check consistency of toc-file after call to
 * 'showToc()'.
 *
 */

static char rcsid[] = "$Id: main.cc,v 1.3 1998/07/28 13:49:35 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <fstream.h>

#include "util.h"
#include "Toc.h"
#include "ScsiIf.h"
#include "dao.h"
#include "CDD2600.h"
#include "PlextorReader.h"

enum Command { SHOW_TOC, SHOW_DATA, READ_TEST, SIMULATE, WRITE, READ_TOC };

static const char *PRGNAME = NULL;
static const char *SCSI_DEVICE = "/dev/cdrecorder";
static const char *TOC_FILE = NULL;
static const char *DRIVER_ID = NULL;
static int WRITING_SPEED = -1;
static int EJECT = 0;
static int SWAP = 0;
static Command COMMAND;

typedef CdrDriver *(*CdrDriverConstructor)(ScsiIf *, Toc *);

struct DriverTable {
  char *driverId;
  char *vendor;
  char *model;
  CdrDriverConstructor constructor;
};

DriverTable DRIVER_TABLE[] = {
  { "cdd2600",
    "PHILIPS",
    "CDD2600",
    &(CDD2600::instance)
  },
  { "plextor",
    "PLEXTOR",
    "CD-ROM",
    &(PlextorReader::instance)
  },
  { NULL,
    NULL,
    NULL,
    NULL
  }
};

void usage()
{
  fprintf(stderr, "Cdrdao version %s\n\n", VERSION);
  fprintf(stderr, "Writes audio CD-R in disc-at-once mode.\n\n");

  fprintf(stderr, "Usage: %s command [options] toc-file\n", PRGNAME);

  fprintf(stderr, 
"  command: show-toc  - prints out toc and exits\n\
           read-toc  - create toc file from audio CD\n\
           show-data - prints out audio data and exits\n\
           read-test - reads all audio files and exits\n\
           simulate  - performs a write simulation\n\
           write     - writes CD\n\n");

  fprintf(stderr, 
"  options: --device <device>       - sets generic SCSI device of CD-writer\n\
                                     (default: /dev/cdrecorder)\n\
           --driver <driver-id>    - force usage of specified driver\n\
           --speed <writing-speed> - selects writing speed\n\
           --eject                 - ejects cd after writing or simulation\n\
           --swap                  - swap byte order of audio files\n");
}

int parseCmdline(int argc, char **argv)
{
  if (argc < 1) {
    return 1;
  }

  if (strcmp(*argv, "show-toc") == 0) {
    COMMAND = SHOW_TOC;
  }
  else if (strcmp(*argv, "read-toc") == 0) {
    COMMAND = READ_TOC;
  }
  else if (strcmp(*argv, "show-data") == 0) {
    COMMAND = SHOW_DATA;
  }
  else if (strcmp(*argv, "read-test") == 0) {
    COMMAND = READ_TEST;
  }
  else if (strcmp(*argv, "simulate") == 0) {
    COMMAND = SIMULATE;
  }
  else if (strcmp(*argv, "write") == 0) {
    COMMAND = WRITE;
  }
  else {
    fprintf(stderr, "Illegal command: %s\n", *argv);
    return 1;
  }

  argc--, argv++;

  while (argc > 0 && (*argv)[0] == '-') {
    if ((*argv)[1] != '-') {
      fprintf(stderr, "Illegal option: %s\n", *argv);
      return 1;
    }
    else {
      if (strcmp((*argv) + 2, "device") == 0) {
	if (argc < 2) {
	  fprintf(stderr, "Missing argument after: %s\n", *argv);
	  return 1;
	}
	else {
	  SCSI_DEVICE = argv[1];
	  argc--, argv++;
	}
      }
      else if (strcmp((*argv) + 2, "speed") == 0) {
	if (argc < 2) {
	  fprintf(stderr, "Missing argument after: %s\n", *argv);
	  return 1;
	}
	else {
	  WRITING_SPEED = atol(argv[1]);
	  if (WRITING_SPEED < 0) {
	    fprintf(stderr, "Illegal writing speed: %s\n", argv[1]);
	    return 1;
	  }
	  argc--, argv++;
	}
      }
      else if (strcmp((*argv) + 2, "eject") == 0) {
	EJECT = 1;
      }
      else if (strcmp((*argv) + 2, "swap") == 0) {
	SWAP = 1;
      }
      else if (strcmp((*argv) + 2, "driver") == 0) {
	if (argc < 2) {
	  fprintf(stderr, "Missing argument after: %s\n", *argv);
	  return 1;
	}
	else {
	  DRIVER_ID = argv[1];
	  argc--, argv++;
	}
      }
      else {
	fprintf(stderr, "Illegal option: %s\n", *argv);
	return 1;
      }

      argc--, argv++;
    }
  }

  if (argc < 1) {
    fprintf(stderr, "Missing toc-file.\n");
    return 1;
  }
  else if (argc > 1) {
    fprintf(stderr, "Expecting only one toc-file.\n");
    return 1;
  }

  TOC_FILE = *argv;

  return 0;
}

void showToc(const Toc *toc)
{
  const Track *t;
  Msf start, end, index;
  int i;
  int n;
  int tcount = 1;

  if (toc->catalogValid()) {
    fprintf(stdout, "CATALOG NUMBER: ");
    for (i = 0; i < 13; i++) {
      fputc(toc->catalog(i) + '0', stdout);
    }
    fprintf(stdout, "\n");
  }

  for (t = toc->first(start, end); t != NULL; t = toc->next(start, end)) {
    fprintf(stdout, "TRACK %2d:\n", tcount);
    if (t->isrcValid()) {
      fprintf(stdout, "          ISRC %c%c %c%c%c %c%c %c%c%c%c%c\n",
	      t->isrcCountry(0), t->isrcCountry(1),
	      t->isrcOwner(0), t->isrcOwner(1), t->isrcOwner(2),
	      t->isrcYear(0) + '0', t->isrcYear(1) + '0',
	      t->isrcSerial(0) + '0', t->isrcSerial(1) + '0',
	      t->isrcSerial(2) + '0', t->isrcSerial(3) + '0',
	      t->isrcSerial(4) + '0');
    }
    fprintf(stdout, "          COPY%sPERMITTED\n",
	    t->copyPermitted() ? " " : " NOT ");
    fprintf(stdout, "          %sPRE-EMPHASIS\n",
	    t->preEmphasis() ? "" : "NO ");
    fprintf(stdout, "          %s CHANNEL AUDIO\n",
	    t->audioType() == 0 ? "TWO" : "FOUR");
    if (t->start().lba() != 0) {
      fprintf(stdout, "          PREGAP %s(%6ld)\n", 
	      t->start().str(), t->start().lba());
    }
    fprintf(stdout, "          START  %s(%6ld)\n",
	    start.str(), start.lba());
    n = t->nofIndices();
    for (i = 0; i < n; i++) {
      index = start + t->getIndex(i);
      fprintf(stdout, "          INDEX  %s(%6ld)\n",
	      index.str(), index.lba());
    }

    fprintf(stdout, "          END%c   %s(%6ld)\n",
	    t->isPadded() ? '*' : ' ', end.str(), end.lba());

    tcount++;
  }
} 

void showData(const Toc *toc, int swap)
{
  long length = toc->length().lba();
  Sample buf[SAMPLES_PER_BLOCK];
  int i;
  unsigned long sampleNr = 0;

  if (toc->openData() != 0) {
    fprintf(stderr, "Cannot open audio data.\n");
    return;
  }

  while (length > 0) {
    if (toc->readData((char *)buf, 1) != 1) {
      fprintf(stderr, "Read of audio data failed.\n");
      return;
    }

    if (swap) {
      swapSamples(buf, SAMPLES_PER_BLOCK);
    }

    for (i = 0; i < SAMPLES_PER_BLOCK; i++) {
      printf("%7lu:%5d %5d\n", sampleNr, buf[i].left(), buf[i].right());
      sampleNr++;
    }
    length -= 1;
  }
}

CdrDriver *selectDriver(ScsiIf *scsiIf, Toc *toc)
{
  DriverTable *run = DRIVER_TABLE;

  if (DRIVER_ID != NULL) {
    while (run->driverId != NULL) {
      if (strcmp(DRIVER_ID, run->driverId) == 0) {
	return run->constructor(scsiIf, toc);
      }
      run++;
    }
    fprintf(stderr, "\n%s: Illegal driver ID, available drivers:\n",
	    DRIVER_ID);
  }
  else {
    while (run->driverId != NULL) {
      if (strcmp(run->vendor, scsiIf->vendor()) == 0 &&
	  strstr(scsiIf->product(), run->model) != NULL) {
	return run->constructor(scsiIf, toc);
      }
      run++;
    }
    fprintf(stderr, "\nNo driver found for your model, available drivers:\n");
  }

  run = DRIVER_TABLE;
  while (run->driverId != NULL) {
    fprintf(stderr, "%s: %s %s\n", run->driverId, run->vendor, run->model);
    run++;
  }

  return NULL;
}

int main(int argc, char **argv)
{
  int ret;

  PRGNAME = *argv;

  if (parseCmdline(argc - 1, argv + 1) != 0) {
    usage();
    return -1;
  }

  Toc *toc = NULL;
  if (COMMAND != READ_TOC) {
    toc = Toc::read(TOC_FILE);

    if (toc == NULL) {
      return -1;
    }

    if (COMMAND != SHOW_TOC) {
      if (toc->check() != 0) {
	fprintf(stderr, "Toc file '%s' is inconsistent - exiting.\n",
		TOC_FILE);
	return -1;
      }
    }
  }

  ScsiIf *scsiIf = NULL;
  CdrDriver *cdr = NULL;

  if (COMMAND == SIMULATE || COMMAND == WRITE || COMMAND == READ_TOC) {
    scsiIf = new ScsiIf(SCSI_DEVICE);

    if (scsiIf->init() != 0) {
      return -1;
    }
  
    printf("%s: %s\t%s\tRev: %s (%s)\n", SCSI_DEVICE, scsiIf->vendor(),
	   scsiIf->product(), scsiIf->revision(), scsiIf->revisionDate());

    if ((cdr = selectDriver(scsiIf, toc)) == NULL) {
      return -1;
    }

    printf("Using driver: %s\n\n", cdr->driverName());

    if (WRITING_SPEED >= 0) {
      if (cdr->speed(WRITING_SPEED) != 0) {
	fprintf(stderr, "Writing speed %d not supported by device.\n",
		WRITING_SPEED);
	return -1;
      }
    }
  }

  switch (COMMAND) {
  case SHOW_TOC:
    showToc(toc);
    if (toc->check() != 0) {
      fprintf(stderr, "Toc file '%s' is inconsistent.\n",
	      TOC_FILE);
    }
    break;

  case SHOW_DATA:
    showData(toc, SWAP);
    break;

  case READ_TEST:
    fprintf(stdout, "Starting read test...\n");
    if (writeDiscAtOnce(toc, NULL, SWAP, 1) != 0) {
      fprintf(stderr, "\nRead test failed.\n");
      return -1;
    }
    break;

  case READ_TOC:
    fprintf(stdout, "Reading toc data...\n");

    ret = cdr->testUnitReady(1);

    if (ret == 1) {
      // scsi command failed
      return -1;
    }
    else if (ret >= 2) {
      fprintf(stderr, "\nUnit not ready.\n");
      return -1;
    }

    if (access(TOC_FILE, W_OK) == 0) {
      fprintf(stderr, "File \"%s\" exists, will not overwrite.\n",
	      TOC_FILE);
      return -1;
    }

    if ((toc = cdr->readDiscToc()) == NULL) {
      return -1;
    }
    else {
      ofstream out(TOC_FILE);
      if (!out) {
	fprintf(stderr, "Cannot open \"%s\" for writing: %s\n",
		TOC_FILE, strerror(errno));
	return -1;
      }
      toc->print(out);
    }
    break;
    
  case WRITE:
    cdr->simulate(0);
    // fall through
    
  case SIMULATE:
    ret = cdr->testUnitReady(1);

    if (ret == 1) {
      // scsi command failed
      return -1;
    }
    else if (ret >= 2) {
      fprintf(stderr, "\nUnit not ready.\n");
      return -1;
    }

    if (cdr->rezeroUnit() != 0 ||
	cdr->startStopUnit(1) != 0 || 
	cdr->preventMediumRemoval(1) != 0) {
      return -1;
    }

    printf("Starting write ");
    if (cdr->simulate()) {
      printf("simulation ");
    }
    printf("at speed %d...\n", cdr->speed());

    if (writeDiscAtOnce(toc, cdr, SWAP, 0) != 0) {
      if (cdr->simulate()) {
	fprintf(stderr, "\nSimulation failed.\n");
      }
      else {
	fprintf(stderr, "\nWriting failed.\n");
      }
      cdr->preventMediumRemoval(0);
      return -1;
    }

    if (cdr->simulate()) {
      fprintf(stdout, "Simulation finished successfully.\n");
    }
    else {
      fprintf(stdout, "Writing finished successfully.\n");
    }

    if (cdr->preventMediumRemoval(0) != 0) {
      return -1;
    }

    if (EJECT) {
      cdr->loadUnload(1);
    }
    break;
  }

  return 0;
}
