/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

static char rcsid[] = "$Id: CDD2600.cc,v 1.3 1998/05/22 18:29:16 mueller Exp $";

#include <config.h>

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <assert.h>

#include "CDD2600.h"

#include "Toc.h"

CDD2600::CDD2600(ScsiIf *scsiIf, Toc *toc) : CdrDriver(scsiIf, toc)
{
  driverName_ = "CDD2600 - Version 1.0";
  
  leadInLength_ = leadOutLength_ = 0;
  speed_ = 2;
  simulate_ = 1;
}

// static constructor
CdrDriver *CDD2600::instance(ScsiIf *scsiIf, Toc *toc)
{
  return new CDD2600(scsiIf, toc);
}

// sets speed
// return: 0: OK
//         1: illegal speed
int CDD2600::speed(int s)
{
  if (s >= 0 && s <= 2) {
    speed_ = s;
    return 0;
  }
  else {
    return 1;
  }
}

// loads ('unload' == 0) or ejects ('unload' == 1) tray
// return: 0: OK
//         1: scsi command failed

int CDD2600::loadUnload(int unload) const
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);

  cmd[0] = 0xe7; // MEDIUM LOAD/UNLOAD
  if (unload) {
    cmd[8] |= 0x01;
  }
  
  switch (scsiIf_->sendCmd(cmd, 10, NULL, 0, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot load/unload medium.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot load/unload medium.\n");
      return 1;
    }
  }

  return 0;
}

// sets the block length
// return: 0: OK
//         1: scsi command failed
int CDD2600::modeSelectBlockSize(int blockSize)
{
  unsigned char cmd[10];
  unsigned char data[16];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  memset(data, 0, 16);

  cmd[0] = 0x55; // MODE SELECT
  cmd[7] = 0;
  cmd[8] = 16;

  data[6] = 0;
  data[7] = 8;

  data[13] = blockSize >> 16;
  data[14] = blockSize >> 8;
  data[15] = blockSize;

  switch(scsiIf_->sendCmd(cmd, 10, data, 16, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot set block size to %d.\n", blockSize);
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot set block size to %d.\n", blockSize);
      return 1;
    }
  }

  return 0;
}

// sets speed and simulation mode
// return: 0: OK
//         1: scsi command failed
int CDD2600::modeSelectSpeed()
{
  unsigned char cmd[10];
  unsigned char data[16];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  memset(data, 0, 16);

  cmd[0] = 0x55; // MODE SELECT
  cmd[1] |= 0x10;
  cmd[7] = 0;
  cmd[8] = 16;

  data[6] = 0;
  data[7] = 0;

  data[8] = 0x23;
  data[9] = 0x06;
  data[10] = speed_;
  data[11] = (simulate_ != 0 ? 1 : 0);

  switch(scsiIf_->sendCmd(cmd, 10, data, 16, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot set speed/simulation mode.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot set speed/simulation mode.\n");
      return 1;
    }
  }

  return 0;
}

// Sets catalog number if valid in given TOC.
// return: 0: OK
//         1: scsi command failed
int CDD2600::modeSelectCatalog(const Toc *toc)
{
  unsigned char cmd[10];
  unsigned char data[18];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  memset(data, 0, 18);

  cmd[0] = 0x55; // MODE SELECT
  cmd[1] |= 0x10;
  cmd[7] = 0;
  cmd[8] = 18;

  data[6] = 0; // no block descriptior
  data[7] = 0;
  
  data[8] = 0x22;
  data[9] = 0x08;
  if (toc->catalogValid()) {
    data[10] = 0x01;
    data[11] = (toc->catalog(0) << 4) | toc->catalog(1);
    data[12] = (toc->catalog(2) << 4) | toc->catalog(3);
    data[13] = (toc->catalog(4) << 4) | toc->catalog(5);
    data[14] = (toc->catalog(6) << 4) | toc->catalog(7);
    data[15] = (toc->catalog(8) << 4) | toc->catalog(9);
    data[16] = (toc->catalog(10) << 4) | toc->catalog(11);
    data[17] = (toc->catalog(12) << 4);
  }

  switch(scsiIf_->sendCmd(cmd, 10, data, 18, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot set catalog number.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot set catalog number.\n");
      return 1;
    }
  }

  return 0;
}

// sets various audio play parameters, output channels are set to stereo mode
// and given volumen
// immediate: 0: wait until audio play command finished
//            1: command finishs immediately after playback has started
// sotc:      0: play across track boundaries
//            1: stop playing at track boundaries
int CDD2600::modeSelectPlay(int immediate, int sotc, unsigned char volume)
{
  unsigned char cmd[10];
  unsigned char data[24];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  memset(data, 0, 24);

  cmd[0] = 0x55; // MODE SELECT
  cmd[1] |= 0x10;
  cmd[7] = 0;
  cmd[8] = 24;

  data[6] = 0; // no block descriptior
  data[7] = 0;
  
  data[8] = 0x0e; // PLAY page code
  data[9] = 14; // parameter length
  if (immediate != 0) {
    data[10] |= 0x04;
  }
  if (sotc != 0) {
    data[10] |= 0x02;
  }
  data[16] = 1;
  data[17] = volume;
  data[18] = 2;
  data[19] = volume;

  switch(scsiIf_->sendCmd(cmd, 10, data, 24, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot set play parameters.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot set play parameters.\n");
      return 1;
    }
  }

  return 0;
}

// requests length of lead-in and lead-out and stores data in member variables.
// return: 0: OK
//         1: scsi command failed
int CDD2600::readSessionInfo()
{
  unsigned char cmd[10];
  unsigned char data[4];
  const unsigned char *sense;
  int senseLen;

  memset(cmd, 0, 10);
  memset(data, 0, 4);

  cmd[0] = 0xee; // READ SESSION INFO
  cmd[8] = 4;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, 4)) {
  case 1:
    fprintf(stderr, "Cannot read session info.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read session info.\n");
      return 1;
    }
  }

  leadInLength_ = (data[0] << 8) | data[1];
  leadOutLength_ = (data[2] << 8) | data[3];

  /*
  fprintf(stderr, "lead-in: %ld, lead-out: %ld\n", leadInLength_,
	  leadOutLength_);
	  */
  return 0;
}

// Sends initiating write session command for disc at once writing. This
// includes the complete table of contents with all related data.
// return: 0: OK
//         1: scsi command failed
int CDD2600::writeSession(const Toc *toc)
{
  unsigned char cmd[10];
  unsigned char *data = NULL;
  unsigned int dataLen = 0;
  unsigned char *tp = NULL;
  unsigned int tdl = 0; // track descriptor length
  const unsigned char *sense;
  int senseLen;
  int indexCount = 0;
  int i;
  int n;
  const Track *t;
  Msf start, end, index;

  // count total number of index increments
  for (t = toc->first(start, end); t != NULL; t = toc->next(start, end)) {
    indexCount += t->nofIndices();
  }

  dataLen = toc->nofTracks() * 20 + indexCount * 4;

  /*
  fprintf(stdout, "%d tracks, %d indexes -> dataLen %u\n", toc->nofTracks(),
	  indexCount, dataLen);
  */

  data = (unsigned char *)calloc(dataLen, sizeof(char));
  tp = data;

  memset(cmd, 0, 10);

  cmd[0] = 0xed; // write session
  cmd[6] = 0; // CD-DA, no new program area
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  
  for (t = toc->first(start, end); t != NULL; t = toc->next(start, end)) {
    n = t->nofIndices();
    tdl = 20 + n * 4;
    tp[0] = tdl >> 8;
    tp[1] = tdl;

    tp[2] = 0;
    if (t->preEmphasis()) {
      tp[2] |= 0x01;
    }
    if (t->copyPermitted()) {
      tp[2] |= 0x02;
    }
    if (t->audioType() == 1) {
      tp[2] |= 0x08;
    }

    if (t->isrcValid()) {
      tp[2] |= 0x80;
      
      tp[3] = ascii2Isrc(t->isrcCountry(0));
      tp[4] = ascii2Isrc(t->isrcCountry(1));
      tp[5] = ascii2Isrc(t->isrcOwner(0));
      tp[6] = ascii2Isrc(t->isrcOwner(1));
      tp[7] = ascii2Isrc(t->isrcOwner(2));
      tp[8] = (t->isrcYear(0) << 4) | t->isrcYear(1);
      tp[9] = (t->isrcSerial(0) << 4) | t->isrcSerial(1);
      tp[10] = (t->isrcSerial(2) << 4) | t->isrcSerial(3);
      tp[11] = (t->isrcSerial(4) << 4);
    }

    //fprintf(stdout, "Track start: %s(0x%06lx)\n", start.str(), start.lba());
    tp[12] = start.lba() >> 24;
    tp[13] = start.lba() >> 16;
    tp[14] = start.lba() >> 8;
    tp[15] = start.lba();

    for (i = 0; i < n; i++) {
      index = start + t->getIndex(i);
      //fprintf(stdout, "      index: %s(0x%06lx)\n", index.str(), index.lba());
      
      tp[16 + i * 4] = index.lba() >> 24;
      tp[17 + i * 4] = index.lba() >> 16;
      tp[18 + i * 4] = index.lba() >> 8;
      tp[19 + i * 4] = index.lba();
    }

    //fprintf(stdout, "      end  : %s(0x%06lx)\n", end.str(), end.lba());
    
    tp[16 + n * 4] = end.lba() >> 24;
    tp[17 + n * 4] = end.lba() >> 16;
    tp[18 + n * 4] = end.lba() >> 8;
    tp[19 + n * 4] = end.lba();

    tp += tdl;
  }

  //printf("tp: %d\n", tp - data);

  switch(scsiIf_->sendCmd(cmd, 10, data, dataLen, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot write disc toc.\n");
    free(data);
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot write disc toc.\n");
      free(data);
      return 1;
    }
  }

  free(data);
  return 0;
}

int CDD2600::writeZeros(long count) const
{
  assert(blockLength_ > 0);
  int blocksPerWrite = scsiIf_->maxDataLen() / blockLength_;
  int n;
  char *data;
  long cnt = 0;
  long total = count * blockLength_;

  data = (char *)calloc(blocksPerWrite * blockLength_, sizeof(char));

  while (count > 0) {
    n = (count > blocksPerWrite ? blocksPerWrite : count);

    if (writeData(data, n) != 0) {
      free(data);
      return 1;
    }
    
    cnt += n * blockLength_;

    if ((cnt >> 20) > 0) {
      fprintf(stdout, "Wrote %ld of %ld MB.\r", cnt >> 20, total >> 20);
      fflush(stdout);
    }

    count -= n;
  }

  free(data);
  return 0;
}

int CDD2600::initDao()
{
  blockLength_ = AUDIO_BLOCK_LEN;

  if (modeSelectBlockSize(blockLength_) != 0 ||
      modeSelectSpeed() != 0 ||
      modeSelectCatalog(toc_) != 0 ||
      readSessionInfo() != 0) {
    return 1;
  }

  return 0;
}

int CDD2600::startDao()
{
  if (writeSession(toc_) != 0) {
    return 1;
  }

  fprintf(stdout, "Writing lead-in and gap...\n");

  // write lead-in
  if (writeZeros(leadInLength_) != 0) {
    flushCache();
    return 1;
  }

  // write gap (2 seconds)
  if (writeZeros(150) != 0) {
    flushCache();
    return 1;
  }
  
  fprintf(stdout, "\n");

  return 0;
}

int CDD2600::finishDao()
{
  fprintf(stdout, "Writing lead-out...\n");

  // write lead-out
  if (writeZeros(leadOutLength_) != 0) {
    flushCache();
    return 1;
  }

  fprintf(stdout, "\nFlushing cache...\n");
  
  if (flushCache() != 0) {
    return 1;
  }

  fprintf(stdout, "\n");

  return 0;
}

Toc *CDD2600::readDiscToc()
{
  Toc *toc = NULL;
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 100 * 8 + 4;
  unsigned char data[100 * 8 + 4];
  unsigned char *p = NULL;
  int nofTracks = 0;
  long trackStart[100];
  int i;
  long discStart = 0;

  // read disc toc
  memset(cmd, 0, 10);
  cmd[0] = 0x43; // READ TOC
  cmd[6] = 0; // return info for all tracks
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;
  cmd[9] = 0; // return tack information

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read disc toc.\n");
    return NULL;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read disc toc.\n");
      return NULL;
    }
  }
  
  
  nofTracks = data[3] - data[2] + 1;
  if (nofTracks > 99) {
    fprintf(stderr, "Got illegal toc data.\n");
    return NULL;
  }


  for (i = 0, p = data + 4; i <= nofTracks; i++, p += 8) {
    if (i < nofTracks && (p[1] & 0x0f) != 0x0f && (p[1] & 0x04) != 0) {
      fprintf(stderr, "Disc contains data track - aborting.\n");
      return NULL;
    }
    trackStart[i] = p[4] << 24;
    trackStart[i] |= p[5] << 16;
    trackStart[i] |= p[6] << 8;
    trackStart[i] |= p[7];

  }

  if (modeSelectPlay(0, 1, 0) != 0) {
    return NULL;
  }
  
  toc = new Toc;
  
  long lastpregap = 0;
  long pregap = 0;

  for (i = 0, p = data + 4; i < nofTracks; i++, p += 8) {
    Track t(Track::AUDIO);

    Msf trackLength(trackStart[i+1] - trackStart[i]);

    printf("Analyzing track %d: start %s, ", i + 1, Msf(trackStart[i]).str());
    printf("length %s...\n", trackLength.str());

    if ((p[1] & 0x0f) != 0x0f) {
      t.preEmphasis(p[1] & 0x01);
      t.copyPermitted(p[1] & 0x02);
      t.audioType(p[1] & 0x08);
    }

    if (i == 0 && trackStart[i] != 0) {
      discStart = trackStart[i];
      t.append(SubTrack(SubTrack::DATA,
			AudioData(AudioData::SILENCE, 
				  Msf(discStart).samples())));
    }

#if 0
    lastpregap = pregap;
    pregap = 0;

    if (i < nofTracks - 1) {
      // find pregap of next track
      pregap = findPregap(i + 2, trackStart[i], 
			  trackStart[i+1] - trackStart[i] + 150);
      //printf("Pregap: %ld, track start: %ld\n", pregap,  trackStart[i]);
      if (pregap >= trackStart[i+1]) {
	pregap = 0;
      }
      else if (pregap > 0) {
	pregap = trackStart[i+1] - pregap;
      }
    }
#endif

    if (lastpregap > 0) {
      printf("Found pre-gap: %s\n", Msf(lastpregap).str());
    }

    trackLength = Msf(trackStart[i+1] - trackStart[i] - pregap + lastpregap);

    t.append(SubTrack(SubTrack::DATA,
		      AudioData("data.wav", 
				Msf(trackStart[i] - discStart
				    - lastpregap).samples(), 
				trackLength.samples())));
    if (i == 0) {
      t.start(Msf(discStart));
    }
    else {
      t.start(Msf(lastpregap));
    }

    if (readIsrc(i + 1, &t)) {
      printf("Found ISRC code.\n");
    }

    toc->append(&t);

  }

  playAudioBlock(trackStart[0]);
  if (readCatalog(toc)) {
    printf("Found disc catalogue number.\n");
  }

  return toc;
}

// tries to read catalog number from disc and adds it to 'toc'
// return: 1 if valid catalog number was found, else 0

int CDD2600::readCatalog(Toc *toc)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char catalog[14];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x02; // get media catalog number
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read sub channel data.\n");
    return 0;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read sub channel data.\n");
      return 0;
    }
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 13; i++) {
      catalog[i] = data[0x09 + i];
    }
    catalog[13] = 0;

    if (toc->catalog(catalog) == 0) {
      return 1;
    }
  }

  return 0;
}

// plays one audio block starting from given position
void CDD2600::playAudioBlock(long start)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;

  // play one audio block
  memset(cmd, 0, 10);
  cmd[0] = 0x45; // PLAY AUDIO
  cmd[2] = start >> 24;
  cmd[3] = start >> 16;
  cmd[4] = start >> 8;
  cmd[5] = start;
  cmd[8] = 1; // 1 block

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, NULL, 0)) {
  case 1:
    fprintf(stderr, "Cannot play audio block.\n");
    return;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot play audio block.\n");
      return;
    }
  }
}

// tries to read ISRC code of given track number and stores it in
// given track.
// return: 1 if valid ISRC code was found, else 0
int CDD2600::readIsrc(int trackNr, Track *track)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];
  char isrc[13];
  int i;

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x03; // get ISRC code
  cmd[6] = trackNr;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read ISRC code.\n");
    return 0;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read ISRC code.\n");
      return 0;
    }
  }

  if (data[0x08] & 0x80) {
    for (i = 0; i < 12; i++) {
      isrc[i] = data[0x09 + i];
    }
    isrc[12] = 0;

    if (track->isrc(isrc) == 0) {
      return 1;
    }
  }

  return 0;
}

int CDD2600::readSubChannelData(int *trackNr, int *indexNr, long *relPos)
{
  unsigned char cmd[10];
  const unsigned char *sense;
  int senseLen;
  unsigned short dataLen = 0x30;
  unsigned char data[0x30];

  // read sub channel information
  memset(cmd, 0, 10);
  cmd[0] = 0x42; // READ SUB CHANNEL
  cmd[2] = 0x40; // get sub channel data
  cmd[3] = 0x00; // get sub Q channel data
  cmd[6] = 0;
  cmd[7] = dataLen >> 8;
  cmd[8] = dataLen;

  switch(scsiIf_->sendCmd(cmd, 10, NULL, 0, data, dataLen)) {
  case 1:
    fprintf(stderr, "Cannot read sub Q channel data.\n");
    return 1;

  case 2:
    sense = scsiIf_->getSense(senseLen);

    if (decodeSense(sense, senseLen, 0) != 0) {
      fprintf(stderr, "Cannot read sub Q channel data.\n");
      return 1;
    }
  }

  *trackNr = data[6];
  *indexNr = data[7];
  *relPos = 0;
  *relPos |= data[0x0c] << 24;
  *relPos |= data[0x0d] << 16;
  *relPos |= data[0x0e] << 8;
  *relPos |= data[0x0f];
  
  return 0;
}

long CDD2600::findPregap(int trackNr, long trackStart, long trackLength)
{
  int actTrack;
  int actIndex;
  long start = trackStart - 150;
  long end = trackStart + trackLength - 1;
  long mid;
  long relPos;

  while (start < end) {
    mid = start + ((end - start) / 2);
    
    //printf("Checking block %ld... ", mid);
    playAudioBlock(mid);
    if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
      return 0;
    }
    //printf("Found track %d, index %d, pos %ld\n", actTrack, actIndex,
    //       relPos);
    if (actTrack < trackNr) {
      playAudioBlock(mid+1);
      if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
	return 0;
      }
      if (actTrack == trackNr) {
	if (actIndex == 0) {
	  //printf("Found pregap at %ld\n", mid + 1);
	  return mid;
	}
	else {
	  return 0;
	}
      }
      else {
	start = mid + 1;
      }
      
    }
    else {
      end = mid;
    }
  }

  return 0;
}

#if 0
long CDD2600::findPregap(int trackNr, long trackStart, long trackLength)
{
  int actTrack;
  int actIndex;
  long i;
  long start = trackStart - 150;
  long end = trackStart + trackLength - 1;
  long mid;
  long relPos;

  while (start < end) {
    mid = start + ((end - start) / 2);
    
    printf("Checking block %ld... ", mid);
    playAudioBlock(mid);
    if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
      return 0;
    }
    printf("Found track %d, index %d, pos %ld\n", actTrack, actIndex, relPos);
    if (actTrack == trackNr && relPos < 0) {
      playAudioBlock(mid+1);
      if (readSubChannelData(&actTrack, &actIndex, &relPos) != 0) {
	return 0;
      }
      if (actTrack == trackNr && relPos >= 0) {
	printf("Found pregap at %ld\n", mid + 1);
	return mid + 1;
      }
      else {
	start = mid + 1;
      }
      
    }
    else {
      if (actTrack < trackNr) {
	start = mid + 1;
      }
      else {
	end = mid;
      }
    }
  }

  return 0;
}
#endif
